<?php
/**
 * Param type 'iconpicker'
 *
 * Use it to create a set of fields that let users pick icons from predefined icon libraries.
 *
 * @see https://kb.wpbakery.com/docs/inner-api/vc_map/#vc_map()-ParametersofparamsArray
 */

if ( ! defined( 'ABSPATH' ) ) {
	die( '-1' );
}

/**
 * Class Vc_IconPicker
 *
 * @since 4.4
 * See example usage in shortcode 'vc_icon'
 *
 *      `` example
 *        array(
 *            'type' => 'iconpicker',
 *            'heading' => esc_html__( 'Icon', 'js_composer' ),
 *            'param_name' => 'icon_fontawesome',
 *            'settings' => array(
 *                'emptyIcon' => false, // default true, display an "EMPTY"
 *     icon? - if false it will display first icon from set as default.
 *                'iconsPerPage' => 200, // default 100, how many icons
 *     per/page to display
 *            ),
 *            'dependency' => array(
 *                'element' => 'type',
 *                'value' => 'fontawesome',
 *            ),
 *        ),
 * vc_filter: vc_iconpicker-type-{your_icon_font_name} - filter to add new icon
 *     font type. see example for vc_iconpicker-type-fontawesome in bottom of
 *     this file Also // SEE HOOKS FOLDER FOR FONTS REGISTERING/ENQUEUE IN BASE
 * @path "/include/autoload/hook-vc-iconpicker-param.php"
 */
class Vc_IconPicker {
	/**
	 * Param settings.
	 *
	 * @since 4.4
	 * @var array
	 */
	protected $settings;

	/**
	 * The current value of the icon picker field.
	 *
	 * @since 4.4
	 * @var string
	 */
	protected $value;

	/**
	 * An optional source array for icons, which can be passed or set using filters.
	 *
	 * @since 4.4
	 * @var array
	 */
	protected $source = [];

	/**
	 * Vc_IconPicker constructor.
	 *
	 * @param array $settings - param field data array.
	 * @param string $value - param field value.
	 * @since 4.4
	 */
	public function __construct( $settings, $value ) {
		$this->settings = $settings;
		$this->setDefaults();

		$this->value = $value; // param field value.
	}

	/**
	 * Set default function will extend current settings with defaults
	 * It can be used in Vc_IconPicker::render, but also it is passed to input
	 * field and was hooked in composer-atts.js file See vc.atts.iconpicker in
	 * wp-content/plugins/js_composer/assets/js/params/composer-atts.js init
	 * method
	 *  - it initializes javascript logic, you can provide ANY default param to
	 * it with 'settings' key
	 *
	 * @since 4.4
	 */
	protected function setDefaults() {
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['type'] ) ) {
			$this->settings['settings']['type'] = 'fontawesome'; // Default type for icons.
		}

		// More about this you can read in http://codeb.it/fonticonpicker/.
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['hasSearch'] ) ) {
			// Whether or not to show the search bar.
			$this->settings['settings']['hasSearch'] = true;
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['emptyIcon'] ) ) {
			// Whether or not empty icon should be shown on the icon picker.
			$this->settings['settings']['emptyIcon'] = true;
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['allCategoryText'] ) ) {
			// If categorized then use this option.
			$this->settings['settings']['allCategoryText'] = esc_html__( 'From all categories', 'js_composer' );
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['unCategorizedText'] ) ) {
			// If categorized then use this option.
			$this->settings['settings']['unCategorizedText'] = esc_html__( 'Uncategorized', 'js_composer' );
		}

		/**
		 * Source for icons, can be passed via "mapping" or with filter vc_iconpicker-type-{your_type} (default fontawesome)
		 * vc_filter: vc_iconpicker-type-{your_type} (default fontawesome)
		 */
		if ( isset( $this->settings['settings'], $this->settings['settings']['source'] ) ) {
			$this->source = $this->settings['settings']['source'];
			unset( $this->settings['settings']['source'] ); // We don't need this on frontend.(js).
		}
	}

	/**
	 * Render edit form field type 'iconpicker' with selected settings and
	 * provided value. It uses javascript file vc-icon-picker
	 * (vc_iconpicker_base_register_js, vc_iconpicker_editor_jscss), see
	 * wp-content/plugins/js_composer/include/autoload/hook-vc-iconpicker-param.php
	 * folder
	 *
	 * @return string - rendered param field for editor panel
	 * @since 4.4
	 */
	public function render() {

		$output = '<div class="vc-iconpicker-wrapper"><select class="vc-iconpicker">';

		// call filter vc_iconpicker-type-{your_type}, e.g. vc_iconpicker-type-fontawesome with passed source from shortcode(default empty array). to get icons.
		$arr = apply_filters( 'vc_iconpicker-type-' . esc_attr( $this->settings['settings']['type'] ), $this->source );
		if ( isset( $this->settings['settings'], $this->settings['settings']['emptyIcon'] ) && true === $this->settings['settings']['emptyIcon'] ) {
			array_unshift( $arr, [] );
		}
		if ( ! empty( $arr ) ) {
			foreach ( $arr as $group => $icons ) {
				if ( ! is_array( $icons ) || ! is_array( current( $icons ) ) ) {
					$class_key = empty( $icons ) ? '' : key( $icons );
					$output .= vc_get_template( 'params/iconpicker/single_icon.php',
						[
							'class_key' => $class_key,
							'selected' => null !== $this->value && 0 === strcmp( $class_key, $this->value ) ? 'selected' : '',
							'icon' => current( $icons ),
						]
					);
				} else {
					$output .= vc_get_template( 'params/iconpicker/icon_group.php',
						[
							'icons' => $icons,
							'group' => $group,
							'value' => $this->value,
						]
					);
				}
			}
		}
		$output .= '</select></div>';

		$output .= '<input name="' . esc_attr( $this->settings['param_name'] ) . '" class="wpb_vc_param_value  ' . esc_attr( $this->settings['param_name'] ) . ' ' . esc_attr( $this->settings['type'] ) . '_field" type="hidden" value="' . esc_attr( $this->value ) . '" ' . ( ( isset( $this->settings['settings'] ) && ! empty( $this->settings['settings'] ) ) ? ' data-settings="' . esc_attr( wp_json_encode( $this->settings['settings'] ) ) . '" ' : '' ) . ' />';

		return $output;
	}
}

/**
 * Function for rendering param in edit form (add element)
 * Parse settings from vc_map and entered values.
 *
 * @param array $settings
 * @param string $value
 * @param string $tag
 *
 * @return string - rendered template for params in edit form
 *
 * @since 4.4
 */
function vc_iconpicker_form_field( $settings, $value, $tag ) { // phpcs:ignore:Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed

	$icon_picker = new Vc_IconPicker( $settings, $value );

	return apply_filters( 'vc_iconpicker_render_filter', $icon_picker->render() );
}

// SEE HOOKS FOLDER FOR FONTS REGISTERING/ENQUEUE IN BASE @path "/include/autoload/hook-vc-iconpicker-param.php".

add_filter( 'vc_iconpicker-type-fontawesome', 'vc_iconpicker_type_fontawesome' );

/**
 * Fontawesome icons from FontAwesome :)
 *
 * @param array $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown.
 *
 * @return array - of icons for iconpicker, can be categorized, or not.
 * @version 4.7 $fontawesome_icons
 * @since 4.4
 */
function vc_iconpicker_type_fontawesome( $icons ) {
	// Categorized icons ( you can also output simple array ( key=> value ), where key = icon class, value = icon readable name ).
	$fontawesome_icons = [
		'Accessibility' => [
			[ 'fa fa-brands fa-accessible-icon' => 'Accessible Icon' ],
			[ 'fa fa-solid fa-address-card' => 'Address Card' ],
			[ 'fa fa-regular fa-address-card' => 'Address Card' ],
			[ 'fa fa-solid fa-audio-description' => 'Audio Description' ],
			[ 'fa fa-solid fa-braille' => 'Braille' ],
			[ 'fa fa-solid fa-circle-info' => 'Circle Info' ],
			[ 'fa fa-solid fa-circle-question' => 'Circle Question' ],
			[ 'fa fa-regular fa-circle-question' => 'Circle Question' ],
			[ 'fa fa-solid fa-closed-captioning' => 'Closed Captioning' ],
			[ 'fa fa-regular fa-closed-captioning' => 'Closed Captioning' ],
			[ 'fa fa-solid fa-ear-deaf' => 'Ear Deaf' ],
			[ 'fa fa-solid fa-ear-listen' => 'Ear Listen' ],
			[ 'fa fa-solid fa-eye' => 'Eye' ],
			[ 'fa fa-regular fa-eye' => 'Eye' ],
			[ 'fa fa-solid fa-eye-low-vision' => 'Eye Low Vision' ],
			[ 'fa fa-solid fa-fingerprint' => 'Fingerprint' ],
			[ 'fa fa-solid fa-hands' => 'Hands' ],
			[ 'fa fa-solid fa-hands-asl-interpreting' => 'Hands Asl Interpreting' ],
			[ 'fa fa-solid fa-handshake-angle' => 'Handshake Angle' ],
			[ 'fa fa-solid fa-person-cane' => 'Person Cane' ],
			[ 'fa fa-solid fa-person-walking-with-cane' => 'Person Walking With Cane' ],
			[ 'fa fa-solid fa-phone-volume' => 'Phone Volume' ],
			[ 'fa fa-solid fa-question' => 'Question' ],
			[ 'fa fa-solid fa-tty' => 'Tty' ],
			[ 'fa fa-solid fa-universal-access' => 'Universal Access' ],
			[ 'fa fa-solid fa-wheelchair' => 'Wheelchair' ],
			[ 'fa fa-solid fa-wheelchair-move' => 'Wheelchair Move' ],
		],
		'Alert' => [
			[ 'fa fa-solid fa-bell' => 'Bell' ],
			[ 'fa fa-regular fa-bell' => 'Bell' ],
			[ 'fa fa-solid fa-bell-slash' => 'Bell Slash' ],
			[ 'fa fa-regular fa-bell-slash' => 'Bell Slash' ],
			[ 'fa fa-solid fa-circle-exclamation' => 'Circle Exclamation' ],
			[ 'fa fa-solid fa-circle-radiation' => 'Circle Radiation' ],
			[ 'fa fa-solid fa-exclamation' => 'Exclamation' ],
			[ 'fa fa-solid fa-question' => 'Question' ],
			[ 'fa fa-solid fa-radiation' => 'Radiation' ],
			[ 'fa fa-solid fa-skull-crossbones' => 'Skull Crossbones' ],
			[ 'fa fa-solid fa-triangle-exclamation' => 'Triangle Exclamation' ],
		],
		'Alphabet' => [
			[ 'fa fa-solid fa-a' => 'A' ],
			[ 'fa fa-solid fa-address-card' => 'Address Card' ],
			[ 'fa fa-regular fa-address-card' => 'Address Card' ],
			[ 'fa fa-solid fa-b' => 'B' ],
			[ 'fa fa-solid fa-c' => 'C' ],
			[ 'fa fa-solid fa-circle-h' => 'Circle H' ],
			[ 'fa fa-solid fa-d' => 'D' ],
			[ 'fa fa-solid fa-e' => 'E' ],
			[ 'fa fa-solid fa-f' => 'F' ],
			[ 'fa fa-solid fa-g' => 'G' ],
			[ 'fa fa-solid fa-h' => 'H' ],
			[ 'fa fa-solid fa-i' => 'I' ],
			[ 'fa fa-solid fa-j' => 'J' ],
			[ 'fa fa-solid fa-k' => 'K' ],
			[ 'fa fa-solid fa-l' => 'L' ],
			[ 'fa fa-solid fa-m' => 'M' ],
			[ 'fa fa-solid fa-o' => 'O' ],
			[ 'fa fa-solid fa-p' => 'P' ],
			[ 'fa fa-solid fa-q' => 'Q' ],
			[ 'fa fa-solid fa-r' => 'R' ],
			[ 'fa fa-solid fa-s' => 'S' ],
			[ 'fa fa-solid fa-square-h' => 'Square H' ],
			[ 'fa fa-solid fa-t' => 'T' ],
			[ 'fa fa-solid fa-u' => 'U' ],
			[ 'fa fa-solid fa-v' => 'V' ],
			[ 'fa fa-solid fa-w' => 'W' ],
			[ 'fa fa-solid fa-x' => 'X' ],
			[ 'fa fa-solid fa-z' => 'Z' ],
		],
		'Animals' => [
			[ 'fa fa-solid fa-bugs' => 'Bugs' ],
			[ 'fa fa-solid fa-cat' => 'Cat' ],
			[ 'fa fa-solid fa-cow' => 'Cow' ],
			[ 'fa fa-solid fa-crow' => 'Crow' ],
			[ 'fa fa-solid fa-dog' => 'Dog' ],
			[ 'fa fa-solid fa-dove' => 'Dove' ],
			[ 'fa fa-solid fa-dragon' => 'Dragon' ],
			[ 'fa fa-solid fa-feather' => 'Feather' ],
			[ 'fa fa-solid fa-feather-pointed' => 'Feather Pointed' ],
			[ 'fa fa-solid fa-fish' => 'Fish' ],
			[ 'fa fa-solid fa-fish-fins' => 'Fish Fins' ],
			[ 'fa fa-solid fa-frog' => 'Frog' ],
			[ 'fa fa-solid fa-hippo' => 'Hippo' ],
			[ 'fa fa-solid fa-horse' => 'Horse' ],
			[ 'fa fa-solid fa-horse-head' => 'Horse Head' ],
			[ 'fa fa-solid fa-kiwi-bird' => 'Kiwi Bird' ],
			[ 'fa fa-solid fa-locust' => 'Locust' ],
			[ 'fa fa-solid fa-mosquito' => 'Mosquito' ],
			[ 'fa fa-solid fa-otter' => 'Otter' ],
			[ 'fa fa-solid fa-paw' => 'Paw' ],
			[ 'fa fa-solid fa-shield-cat' => 'Shield Cat' ],
			[ 'fa fa-solid fa-shield-dog' => 'Shield Dog' ],
			[ 'fa fa-solid fa-shrimp' => 'Shrimp' ],
			[ 'fa fa-solid fa-spider' => 'Spider' ],
			[ 'fa fa-solid fa-worm' => 'Worm' ],
		],
		'Arrows' => [
			[ 'fa fa-solid fa-angle-down' => 'Angle Down' ],
			[ 'fa fa-solid fa-angle-left' => 'Angle Left' ],
			[ 'fa fa-solid fa-angle-right' => 'Angle Right' ],
			[ 'fa fa-solid fa-angle-up' => 'Angle Up' ],
			[ 'fa fa-solid fa-angles-down' => 'Angles Down' ],
			[ 'fa fa-solid fa-angles-left' => 'Angles Left' ],
			[ 'fa fa-solid fa-angles-right' => 'Angles Right' ],
			[ 'fa fa-solid fa-angles-up' => 'Angles Up' ],
			[ 'fa fa-solid fa-arrow-down' => 'Arrow Down' ],
			[ 'fa fa-solid fa-arrow-down-1-9' => 'Arrow Down 1 9' ],
			[ 'fa fa-solid fa-arrow-down-9-1' => 'Arrow Down 9 1' ],
			[ 'fa fa-solid fa-arrow-down-a-z' => 'Arrow Down A Z' ],
			[ 'fa fa-solid fa-arrow-down-long' => 'Arrow Down Long' ],
			[ 'fa fa-solid fa-arrow-down-short-wide' => 'Arrow Down Short Wide' ],
			[ 'fa fa-solid fa-arrow-down-up-across-line' => 'Arrow Down Up Across Line' ],
			[ 'fa fa-solid fa-arrow-down-up-lock' => 'Arrow Down Up Lock' ],
			[ 'fa fa-solid fa-arrow-down-wide-short' => 'Arrow Down Wide Short' ],
			[ 'fa fa-solid fa-arrow-down-z-a' => 'Arrow Down Z A' ],
			[ 'fa fa-solid fa-arrow-left' => 'Arrow Left' ],
			[ 'fa fa-solid fa-arrow-left-long' => 'Arrow Left Long' ],
			[ 'fa fa-solid fa-arrow-pointer' => 'Arrow Pointer' ],
			[ 'fa fa-solid fa-arrow-right' => 'Arrow Right' ],
			[ 'fa fa-solid fa-arrow-right-arrow-left' => 'Arrow Right Arrow Left' ],
			[ 'fa fa-solid fa-arrow-right-from-bracket' => 'Arrow Right From Bracket' ],
			[ 'fa fa-solid fa-arrow-right-long' => 'Arrow Right Long' ],
			[ 'fa fa-solid fa-arrow-right-to-bracket' => 'Arrow Right To Bracket' ],
			[ 'fa fa-solid fa-arrow-rotate-left' => 'Arrow Rotate Left' ],
			[ 'fa fa-solid fa-arrow-rotate-right' => 'Arrow Rotate Right' ],
			[ 'fa fa-solid fa-arrow-trend-down' => 'Arrow Trend Down' ],
			[ 'fa fa-solid fa-arrow-trend-up' => 'Arrow Trend Up' ],
			[ 'fa fa-solid fa-arrow-turn-down' => 'Arrow Turn Down' ],
			[ 'fa fa-solid fa-arrow-turn-up' => 'Arrow Turn Up' ],
			[ 'fa fa-solid fa-arrow-up' => 'Arrow Up' ],
			[ 'fa fa-solid fa-arrow-up-1-9' => 'Arrow Up 1 9' ],
			[ 'fa fa-solid fa-arrow-up-9-1' => 'Arrow Up 9 1' ],
			[ 'fa fa-solid fa-arrow-up-a-z' => 'Arrow Up A Z' ],
			[ 'fa fa-solid fa-arrow-up-from-bracket' => 'Arrow Up From Bracket' ],
			[ 'fa fa-solid fa-arrow-up-long' => 'Arrow Up Long' ],
			[ 'fa fa-solid fa-arrow-up-right-dots' => 'Arrow Up Right Dots' ],
			[ 'fa fa-solid fa-arrow-up-right-from-square' => 'Arrow Up Right From Square' ],
			[ 'fa fa-solid fa-arrow-up-short-wide' => 'Arrow Up Short Wide' ],
			[ 'fa fa-solid fa-arrow-up-wide-short' => 'Arrow Up Wide Short' ],
			[ 'fa fa-solid fa-arrow-up-z-a' => 'Arrow Up Z A' ],
			[ 'fa fa-solid fa-arrows-down-to-line' => 'Arrows Down To Line' ],
			[ 'fa fa-solid fa-arrows-left-right' => 'Arrows Left Right' ],
			[ 'fa fa-solid fa-arrows-left-right-to-line' => 'Arrows Left Right To Line' ],
			[ 'fa fa-solid fa-arrows-rotate' => 'Arrows Rotate' ],
			[ 'fa fa-solid fa-arrows-spin' => 'Arrows Spin' ],
			[ 'fa fa-solid fa-arrows-split-up-and-left' => 'Arrows Split Up And Left' ],
			[ 'fa fa-solid fa-arrows-to-circle' => 'Arrows To Circle' ],
			[ 'fa fa-solid fa-arrows-to-dot' => 'Arrows To Dot' ],
			[ 'fa fa-solid fa-arrows-to-eye' => 'Arrows To Eye' ],
			[ 'fa fa-solid fa-arrows-turn-right' => 'Arrows Turn Right' ],
			[ 'fa fa-solid fa-arrows-turn-to-dots' => 'Arrows Turn To Dots' ],
			[ 'fa fa-solid fa-arrows-up-down' => 'Arrows Up Down' ],
			[ 'fa fa-solid fa-arrows-up-down-left-right' => 'Arrows Up Down Left Right' ],
			[ 'fa fa-solid fa-arrows-up-to-line' => 'Arrows Up To Line' ],
			[ 'fa fa-solid fa-caret-down' => 'Caret Down' ],
			[ 'fa fa-solid fa-caret-left' => 'Caret Left' ],
			[ 'fa fa-solid fa-caret-right' => 'Caret Right' ],
			[ 'fa fa-solid fa-caret-up' => 'Caret Up' ],
			[ 'fa fa-solid fa-chevron-down' => 'Chevron Down' ],
			[ 'fa fa-solid fa-chevron-left' => 'Chevron Left' ],
			[ 'fa fa-solid fa-chevron-right' => 'Chevron Right' ],
			[ 'fa fa-solid fa-chevron-up' => 'Chevron Up' ],
			[ 'fa fa-solid fa-circle-arrow-down' => 'Circle Arrow Down' ],
			[ 'fa fa-solid fa-circle-arrow-left' => 'Circle Arrow Left' ],
			[ 'fa fa-solid fa-circle-arrow-right' => 'Circle Arrow Right' ],
			[ 'fa fa-solid fa-circle-arrow-up' => 'Circle Arrow Up' ],
			[ 'fa fa-solid fa-circle-chevron-down' => 'Circle Chevron Down' ],
			[ 'fa fa-solid fa-circle-chevron-left' => 'Circle Chevron Left' ],
			[ 'fa fa-solid fa-circle-chevron-right' => 'Circle Chevron Right' ],
			[ 'fa fa-solid fa-circle-chevron-up' => 'Circle Chevron Up' ],
			[ 'fa fa-solid fa-circle-down' => 'Circle Down' ],
			[ 'fa fa-regular fa-circle-down' => 'Circle Down' ],
			[ 'fa fa-solid fa-circle-left' => 'Circle Left' ],
			[ 'fa fa-regular fa-circle-left' => 'Circle Left' ],
			[ 'fa fa-solid fa-circle-right' => 'Circle Right' ],
			[ 'fa fa-regular fa-circle-right' => 'Circle Right' ],
			[ 'fa fa-solid fa-circle-up' => 'Circle Up' ],
			[ 'fa fa-regular fa-circle-up' => 'Circle Up' ],
			[ 'fa fa-solid fa-clock-rotate-left' => 'Clock Rotate Left' ],
			[ 'fa fa-solid fa-cloud-arrow-down' => 'Cloud Arrow Down' ],
			[ 'fa fa-solid fa-cloud-arrow-up' => 'Cloud Arrow Up' ],
			[ 'fa fa-solid fa-down-left-and-up-right-to-center' => 'Down Left And Up Right To Center' ],
			[ 'fa fa-solid fa-down-long' => 'Down Long' ],
			[ 'fa fa-solid fa-download' => 'Download' ],
			[ 'fa fa-solid fa-left-long' => 'Left Long' ],
			[ 'fa fa-solid fa-left-right' => 'Left Right' ],
			[ 'fa fa-solid fa-location-arrow' => 'Location Arrow' ],
			[ 'fa fa-solid fa-maximize' => 'Maximize' ],
			[ 'fa fa-solid fa-recycle' => 'Recycle' ],
			[ 'fa fa-solid fa-repeat' => 'Repeat' ],
			[ 'fa fa-solid fa-reply' => 'Reply' ],
			[ 'fa fa-solid fa-reply-all' => 'Reply All' ],
			[ 'fa fa-solid fa-retweet' => 'Retweet' ],
			[ 'fa fa-solid fa-right-from-bracket' => 'Right From Bracket' ],
			[ 'fa fa-solid fa-right-left' => 'Right Left' ],
			[ 'fa fa-solid fa-right-long' => 'Right Long' ],
			[ 'fa fa-solid fa-right-to-bracket' => 'Right To Bracket' ],
			[ 'fa fa-solid fa-rotate' => 'Rotate' ],
			[ 'fa fa-solid fa-rotate-left' => 'Rotate Left' ],
			[ 'fa fa-solid fa-rotate-right' => 'Rotate Right' ],
			[ 'fa fa-solid fa-share' => 'Share' ],
			[ 'fa fa-solid fa-share-from-square' => 'Share From Square' ],
			[ 'fa fa-regular fa-share-from-square' => 'Share From Square' ],
			[ 'fa fa-solid fa-shuffle' => 'Shuffle' ],
			[ 'fa fa-solid fa-sort' => 'Sort' ],
			[ 'fa fa-solid fa-sort-down' => 'Sort Down' ],
			[ 'fa fa-solid fa-sort-up' => 'Sort Up' ],
			[ 'fa fa-solid fa-square-arrow-up-right' => 'Square Arrow Up Right' ],
			[ 'fa fa-solid fa-square-caret-down' => 'Square Caret Down' ],
			[ 'fa fa-regular fa-square-caret-down' => 'Square Caret Down' ],
			[ 'fa fa-solid fa-square-caret-left' => 'Square Caret Left' ],
			[ 'fa fa-regular fa-square-caret-left' => 'Square Caret Left' ],
			[ 'fa fa-solid fa-square-caret-right' => 'Square Caret Right' ],
			[ 'fa fa-regular fa-square-caret-right' => 'Square Caret Right' ],
			[ 'fa fa-solid fa-square-caret-up' => 'Square Caret Up' ],
			[ 'fa fa-regular fa-square-caret-up' => 'Square Caret Up' ],
			[ 'fa fa-solid fa-square-up-right' => 'Square Up Right' ],
			[ 'fa fa-solid fa-turn-down' => 'Turn Down' ],
			[ 'fa fa-solid fa-turn-up' => 'Turn Up' ],
			[ 'fa fa-solid fa-up-down' => 'Up Down' ],
			[ 'fa fa-solid fa-up-down-left-right' => 'Up Down Left Right' ],
			[ 'fa fa-solid fa-up-long' => 'Up Long' ],
			[ 'fa fa-solid fa-up-right-and-down-left-from-center' => 'Up Right And Down Left From Center' ],
			[ 'fa fa-solid fa-up-right-from-square' => 'Up Right From Square' ],
			[ 'fa fa-solid fa-upload' => 'Upload' ],
		],
		'Astronomy' => [
			[ 'fa fa-solid fa-binoculars' => 'Binoculars' ],
			[ 'fa fa-solid fa-globe' => 'Globe' ],
			[ 'fa fa-solid fa-meteor' => 'Meteor' ],
			[ 'fa fa-solid fa-moon' => 'Moon' ],
			[ 'fa fa-regular fa-moon' => 'Moon' ],
			[ 'fa fa-solid fa-satellite' => 'Satellite' ],
			[ 'fa fa-solid fa-satellite-dish' => 'Satellite Dish' ],
			[ 'fa fa-solid fa-shuttle-space' => 'Shuttle Space' ],
			[ 'fa fa-solid fa-user-astronaut' => 'User Astronaut' ],
		],
		'Automotive' => [
			[ 'fa fa-solid fa-bus' => 'Bus' ],
			[ 'fa fa-solid fa-bus-simple' => 'Bus Simple' ],
			[ 'fa fa-solid fa-car' => 'Car' ],
			[ 'fa fa-solid fa-car-battery' => 'Car Battery' ],
			[ 'fa fa-solid fa-car-burst' => 'Car Burst' ],
			[ 'fa fa-solid fa-car-on' => 'Car On' ],
			[ 'fa fa-solid fa-car-rear' => 'Car Rear' ],
			[ 'fa fa-solid fa-car-side' => 'Car Side' ],
			[ 'fa fa-solid fa-car-tunnel' => 'Car Tunnel' ],
			[ 'fa fa-solid fa-caravan' => 'Caravan' ],
			[ 'fa fa-solid fa-charging-station' => 'Charging Station' ],
			[ 'fa fa-solid fa-gas-pump' => 'Gas Pump' ],
			[ 'fa fa-solid fa-gauge' => 'Gauge' ],
			[ 'fa fa-solid fa-gauge-high' => 'Gauge High' ],
			[ 'fa fa-solid fa-gauge-simple' => 'Gauge Simple' ],
			[ 'fa fa-solid fa-gauge-simple-high' => 'Gauge Simple High' ],
			[ 'fa fa-solid fa-motorcycle' => 'Motorcycle' ],
			[ 'fa fa-solid fa-oil-can' => 'Oil Can' ],
			[ 'fa fa-solid fa-spray-can-sparkles' => 'Spray Can Sparkles' ],
			[ 'fa fa-solid fa-taxi' => 'Taxi' ],
			[ 'fa fa-solid fa-trailer' => 'Trailer' ],
			[ 'fa fa-solid fa-truck' => 'Truck' ],
			[ 'fa fa-solid fa-truck-field' => 'Truck Field' ],
			[ 'fa fa-solid fa-truck-field-un' => 'Truck Field Un' ],
			[ 'fa fa-solid fa-truck-medical' => 'Truck Medical' ],
			[ 'fa fa-solid fa-truck-monster' => 'Truck Monster' ],
			[ 'fa fa-solid fa-truck-pickup' => 'Truck Pickup' ],
			[ 'fa fa-solid fa-van-shuttle' => 'Van Shuttle' ],
		],
		'Buildings' => [
			[ 'fa fa-solid fa-archway' => 'Archway' ],
			[ 'fa fa-solid fa-arrow-right-to-city' => 'Arrow Right To City' ],
			[ 'fa fa-solid fa-building' => 'Building' ],
			[ 'fa fa-regular fa-building' => 'Building' ],
			[ 'fa fa-solid fa-building-circle-arrow-right' => 'Building Circle Arrow Right' ],
			[ 'fa fa-solid fa-building-circle-check' => 'Building Circle Check' ],
			[ 'fa fa-solid fa-building-circle-exclamation' => 'Building Circle Exclamation' ],
			[ 'fa fa-solid fa-building-circle-xmark' => 'Building Circle Xmark' ],
			[ 'fa fa-solid fa-building-columns' => 'Building Columns' ],
			[ 'fa fa-solid fa-building-flag' => 'Building Flag' ],
			[ 'fa fa-solid fa-building-lock' => 'Building Lock' ],
			[ 'fa fa-solid fa-building-ngo' => 'Building Ngo' ],
			[ 'fa fa-solid fa-building-shield' => 'Building Shield' ],
			[ 'fa fa-solid fa-building-un' => 'Building Un' ],
			[ 'fa fa-solid fa-building-user' => 'Building User' ],
			[ 'fa fa-solid fa-building-wheat' => 'Building Wheat' ],
			[ 'fa fa-solid fa-campground' => 'Campground' ],
			[ 'fa fa-solid fa-church' => 'Church' ],
			[ 'fa fa-solid fa-city' => 'City' ],
			[ 'fa fa-solid fa-dungeon' => 'Dungeon' ],
			[ 'fa fa-solid fa-gopuram' => 'Gopuram' ],
			[ 'fa fa-solid fa-hospital' => 'Hospital' ],
			[ 'fa fa-regular fa-hospital' => 'Hospital' ],
			[ 'fa fa-solid fa-hospital-user' => 'Hospital User' ],
			[ 'fa fa-solid fa-hotel' => 'Hotel' ],
			[ 'fa fa-solid fa-house' => 'House' ],
			[ 'fa fa-solid fa-house-chimney' => 'House Chimney' ],
			[ 'fa fa-solid fa-house-chimney-crack' => 'House Chimney Crack' ],
			[ 'fa fa-solid fa-house-chimney-medical' => 'House Chimney Medical' ],
			[ 'fa fa-solid fa-house-chimney-window' => 'House Chimney Window' ],
			[ 'fa fa-solid fa-house-circle-check' => 'House Circle Check' ],
			[ 'fa fa-solid fa-house-circle-exclamation' => 'House Circle Exclamation' ],
			[ 'fa fa-solid fa-house-circle-xmark' => 'House Circle Xmark' ],
			[ 'fa fa-solid fa-house-crack' => 'House Crack' ],
			[ 'fa fa-solid fa-house-fire' => 'House Fire' ],
			[ 'fa fa-solid fa-house-flag' => 'House Flag' ],
			[ 'fa fa-solid fa-house-lock' => 'House Lock' ],
			[ 'fa fa-solid fa-house-medical' => 'House Medical' ],
			[ 'fa fa-solid fa-house-medical-circle-check' => 'House Medical Circle Check' ],
			[ 'fa fa-solid fa-house-medical-circle-exclamation' => 'House Medical Circle Exclamation' ],
			[ 'fa fa-solid fa-house-medical-circle-xmark' => 'House Medical Circle Xmark' ],
			[ 'fa fa-solid fa-house-medical-flag' => 'House Medical Flag' ],
			[ 'fa fa-solid fa-igloo' => 'Igloo' ],
			[ 'fa fa-solid fa-industry' => 'Industry' ],
			[ 'fa fa-solid fa-kaaba' => 'Kaaba' ],
			[ 'fa fa-solid fa-landmark' => 'Landmark' ],
			[ 'fa fa-solid fa-landmark-dome' => 'Landmark Dome' ],
			[ 'fa fa-solid fa-landmark-flag' => 'Landmark Flag' ],
			[ 'fa fa-solid fa-monument' => 'Monument' ],
			[ 'fa fa-solid fa-mosque' => 'Mosque' ],
			[ 'fa fa-solid fa-mountain-city' => 'Mountain City' ],
			[ 'fa fa-solid fa-oil-well' => 'Oil Well' ],
			[ 'fa fa-solid fa-place-of-worship' => 'Place Of Worship' ],
			[ 'fa fa-solid fa-school' => 'School' ],
			[ 'fa fa-solid fa-school-circle-check' => 'School Circle Check' ],
			[ 'fa fa-solid fa-school-circle-exclamation' => 'School Circle Exclamation' ],
			[ 'fa fa-solid fa-school-circle-xmark' => 'School Circle Xmark' ],
			[ 'fa fa-solid fa-school-flag' => 'School Flag' ],
			[ 'fa fa-solid fa-school-lock' => 'School Lock' ],
			[ 'fa fa-solid fa-shop' => 'Shop' ],
			[ 'fa fa-solid fa-shop-lock' => 'Shop Lock' ],
			[ 'fa fa-solid fa-store' => 'Store' ],
			[ 'fa fa-solid fa-synagogue' => 'Synagogue' ],
			[ 'fa fa-solid fa-tent' => 'Tent' ],
			[ 'fa fa-solid fa-tent-arrow-down-to-line' => 'Tent Arrow Down To Line' ],
			[ 'fa fa-solid fa-tent-arrow-left-right' => 'Tent Arrow Left Right' ],
			[ 'fa fa-solid fa-tent-arrow-turn-left' => 'Tent Arrow Turn Left' ],
			[ 'fa fa-solid fa-tent-arrows-down' => 'Tent Arrows Down' ],
			[ 'fa fa-solid fa-tents' => 'Tents' ],
			[ 'fa fa-solid fa-toilet-portable' => 'Toilet Portable' ],
			[ 'fa fa-solid fa-toilets-portable' => 'Toilets Portable' ],
			[ 'fa fa-solid fa-torii-gate' => 'Torii Gate' ],
			[ 'fa fa-solid fa-tower-observation' => 'Tower Observation' ],
			[ 'fa fa-solid fa-tree-city' => 'Tree City' ],
			[ 'fa fa-solid fa-vihara' => 'Vihara' ],
			[ 'fa fa-solid fa-warehouse' => 'Warehouse' ],
		],
		'Business' => [
			[ 'fa fa-solid fa-address-book' => 'Address Book' ],
			[ 'fa fa-regular fa-address-book' => 'Address Book' ],
			[ 'fa fa-solid fa-address-card' => 'Address Card' ],
			[ 'fa fa-regular fa-address-card' => 'Address Card' ],
			[ 'fa fa-solid fa-arrows-spin' => 'Arrows Spin' ],
			[ 'fa fa-solid fa-arrows-to-dot' => 'Arrows To Dot' ],
			[ 'fa fa-solid fa-arrows-to-eye' => 'Arrows To Eye' ],
			[ 'fa fa-solid fa-bars-progress' => 'Bars Progress' ],
			[ 'fa fa-solid fa-bars-staggered' => 'Bars Staggered' ],
			[ 'fa fa-solid fa-book' => 'Book' ],
			[ 'fa fa-solid fa-box-archive' => 'Box Archive' ],
			[ 'fa fa-solid fa-boxes-packing' => 'Boxes Packing' ],
			[ 'fa fa-solid fa-briefcase' => 'Briefcase' ],
			[ 'fa fa-solid fa-building' => 'Building' ],
			[ 'fa fa-regular fa-building' => 'Building' ],
			[ 'fa fa-solid fa-bullhorn' => 'Bullhorn' ],
			[ 'fa fa-solid fa-bullseye' => 'Bullseye' ],
			[ 'fa fa-solid fa-business-time' => 'Business Time' ],
			[ 'fa fa-solid fa-cake-candles' => 'Cake Candles' ],
			[ 'fa fa-solid fa-calculator' => 'Calculator' ],
			[ 'fa fa-solid fa-calendar' => 'Calendar' ],
			[ 'fa fa-regular fa-calendar' => 'Calendar' ],
			[ 'fa fa-solid fa-calendar-days' => 'Calendar Days' ],
			[ 'fa fa-regular fa-calendar-days' => 'Calendar Days' ],
			[ 'fa fa-solid fa-certificate' => 'Certificate' ],
			[ 'fa fa-solid fa-chart-line' => 'Chart Line' ],
			[ 'fa fa-solid fa-chart-pie' => 'Chart Pie' ],
			[ 'fa fa-solid fa-chart-simple' => 'Chart Simple' ],
			[ 'fa fa-solid fa-city' => 'City' ],
			[ 'fa fa-solid fa-clipboard' => 'Clipboard' ],
			[ 'fa fa-regular fa-clipboard' => 'Clipboard' ],
			[ 'fa fa-solid fa-clipboard-question' => 'Clipboard Question' ],
			[ 'fa fa-solid fa-compass' => 'Compass' ],
			[ 'fa fa-regular fa-compass' => 'Compass' ],
			[ 'fa fa-solid fa-copy' => 'Copy' ],
			[ 'fa fa-regular fa-copy' => 'Copy' ],
			[ 'fa fa-solid fa-copyright' => 'Copyright' ],
			[ 'fa fa-regular fa-copyright' => 'Copyright' ],
			[ 'fa fa-solid fa-envelope' => 'Envelope' ],
			[ 'fa fa-regular fa-envelope' => 'Envelope' ],
			[ 'fa fa-solid fa-envelope-circle-check' => 'Envelope Circle Check' ],
			[ 'fa fa-solid fa-envelope-open' => 'Envelope Open' ],
			[ 'fa fa-regular fa-envelope-open' => 'Envelope Open' ],
			[ 'fa fa-solid fa-eraser' => 'Eraser' ],
			[ 'fa fa-solid fa-fax' => 'Fax' ],
			[ 'fa fa-solid fa-file' => 'File' ],
			[ 'fa fa-regular fa-file' => 'File' ],
			[ 'fa fa-solid fa-file-circle-plus' => 'File Circle Plus' ],
			[ 'fa fa-solid fa-file-lines' => 'File Lines' ],
			[ 'fa fa-regular fa-file-lines' => 'File Lines' ],
			[ 'fa fa-solid fa-floppy-disk' => 'Floppy Disk' ],
			[ 'fa fa-regular fa-floppy-disk' => 'Floppy Disk' ],
			[ 'fa fa-solid fa-folder' => 'Folder' ],
			[ 'fa fa-regular fa-folder' => 'Folder' ],
			[ 'fa fa-solid fa-folder-minus' => 'Folder Minus' ],
			[ 'fa fa-solid fa-folder-open' => 'Folder Open' ],
			[ 'fa fa-regular fa-folder-open' => 'Folder Open' ],
			[ 'fa fa-solid fa-folder-plus' => 'Folder Plus' ],
			[ 'fa fa-solid fa-folder-tree' => 'Folder Tree' ],
			[ 'fa fa-solid fa-glasses' => 'Glasses' ],
			[ 'fa fa-solid fa-globe' => 'Globe' ],
			[ 'fa fa-solid fa-highlighter' => 'Highlighter' ],
			[ 'fa fa-solid fa-house-laptop' => 'House Laptop' ],
			[ 'fa fa-solid fa-industry' => 'Industry' ],
			[ 'fa fa-solid fa-landmark' => 'Landmark' ],
			[ 'fa fa-solid fa-laptop-file' => 'Laptop File' ],
			[ 'fa fa-solid fa-list-check' => 'List Check' ],
			[ 'fa fa-solid fa-magnifying-glass-arrow-right' => 'Magnifying Glass Arrow Right' ],
			[ 'fa fa-solid fa-magnifying-glass-chart' => 'Magnifying Glass Chart' ],
			[ 'fa fa-solid fa-marker' => 'Marker' ],
			[ 'fa fa-solid fa-mug-saucer' => 'Mug Saucer' ],
			[ 'fa fa-solid fa-network-wired' => 'Network Wired' ],
			[ 'fa fa-solid fa-note-sticky' => 'Note Sticky' ],
			[ 'fa fa-regular fa-note-sticky' => 'Note Sticky' ],
			[ 'fa fa-solid fa-paperclip' => 'Paperclip' ],
			[ 'fa fa-solid fa-paste' => 'Paste' ],
			[ 'fa fa-regular fa-paste' => 'Paste' ],
			[ 'fa fa-solid fa-pen' => 'Pen' ],
			[ 'fa fa-solid fa-pen-clip' => 'Pen Clip' ],
			[ 'fa fa-solid fa-pen-fancy' => 'Pen Fancy' ],
			[ 'fa fa-solid fa-pen-nib' => 'Pen Nib' ],
			[ 'fa fa-solid fa-pen-to-square' => 'Pen To Square' ],
			[ 'fa fa-regular fa-pen-to-square' => 'Pen To Square' ],
			[ 'fa fa-solid fa-pencil' => 'Pencil' ],
			[ 'fa fa-solid fa-percent' => 'Percent' ],
			[ 'fa fa-solid fa-person-chalkboard' => 'Person Chalkboard' ],
			[ 'fa fa-solid fa-phone' => 'Phone' ],
			[ 'fa fa-solid fa-phone-flip' => 'Phone Flip' ],
			[ 'fa fa-solid fa-phone-slash' => 'Phone Slash' ],
			[ 'fa fa-solid fa-phone-volume' => 'Phone Volume' ],
			[ 'fa fa-solid fa-print' => 'Print' ],
			[ 'fa fa-solid fa-registered' => 'Registered' ],
			[ 'fa fa-regular fa-registered' => 'Registered' ],
			[ 'fa fa-solid fa-scale-balanced' => 'Scale Balanced' ],
			[ 'fa fa-solid fa-scale-unbalanced' => 'Scale Unbalanced' ],
			[ 'fa fa-solid fa-scale-unbalanced-flip' => 'Scale Unbalanced Flip' ],
			[ 'fa fa-solid fa-scissors' => 'Scissors' ],
			[ 'fa fa-solid fa-signature' => 'Signature' ],
			[ 'fa fa-solid fa-sitemap' => 'Sitemap' ],
			[ 'fa fa-solid fa-socks' => 'Socks' ],
			[ 'fa fa-solid fa-square-envelope' => 'Square Envelope' ],
			[ 'fa fa-solid fa-square-pen' => 'Square Pen' ],
			[ 'fa fa-solid fa-square-phone' => 'Square Phone' ],
			[ 'fa fa-solid fa-square-phone-flip' => 'Square Phone Flip' ],
			[ 'fa fa-solid fa-square-poll-horizontal' => 'Square Poll Horizontal' ],
			[ 'fa fa-solid fa-square-poll-vertical' => 'Square Poll Vertical' ],
			[ 'fa fa-solid fa-stapler' => 'Stapler' ],
			[ 'fa fa-solid fa-table' => 'Table' ],
			[ 'fa fa-solid fa-table-columns' => 'Table Columns' ],
			[ 'fa fa-solid fa-tag' => 'Tag' ],
			[ 'fa fa-solid fa-tags' => 'Tags' ],
			[ 'fa fa-solid fa-thumbtack' => 'Thumbtack' ],
			[ 'fa fa-solid fa-thumbtack-slash' => 'Thumbtack Slash' ],
			[ 'fa fa-solid fa-timeline' => 'Timeline' ],
			[ 'fa fa-solid fa-trademark' => 'Trademark' ],
			[ 'fa fa-solid fa-vault' => 'Vault' ],
			[ 'fa fa-solid fa-wallet' => 'Wallet' ],
		],
		'Camping' => [
			[ 'fa fa-solid fa-binoculars' => 'Binoculars' ],
			[ 'fa fa-solid fa-bottle-water' => 'Bottle Water' ],
			[ 'fa fa-solid fa-bucket' => 'Bucket' ],
			[ 'fa fa-solid fa-campground' => 'Campground' ],
			[ 'fa fa-solid fa-caravan' => 'Caravan' ],
			[ 'fa fa-solid fa-compass' => 'Compass' ],
			[ 'fa fa-regular fa-compass' => 'Compass' ],
			[ 'fa fa-solid fa-faucet' => 'Faucet' ],
			[ 'fa fa-solid fa-faucet-drip' => 'Faucet Drip' ],
			[ 'fa fa-solid fa-fire' => 'Fire' ],
			[ 'fa fa-solid fa-fire-burner' => 'Fire Burner' ],
			[ 'fa fa-solid fa-fire-flame-curved' => 'Fire Flame Curved' ],
			[ 'fa fa-solid fa-frog' => 'Frog' ],
			[ 'fa fa-solid fa-kit-medical' => 'Kit Medical' ],
			[ 'fa fa-solid fa-map' => 'Map' ],
			[ 'fa fa-regular fa-map' => 'Map' ],
			[ 'fa fa-solid fa-map-location' => 'Map Location' ],
			[ 'fa fa-solid fa-map-location-dot' => 'Map Location Dot' ],
			[ 'fa fa-solid fa-mattress-pillow' => 'Mattress Pillow' ],
			[ 'fa fa-solid fa-mosquito' => 'Mosquito' ],
			[ 'fa fa-solid fa-mosquito-net' => 'Mosquito Net' ],
			[ 'fa fa-solid fa-mountain' => 'Mountain' ],
			[ 'fa fa-solid fa-mountain-sun' => 'Mountain Sun' ],
			[ 'fa fa-solid fa-people-roof' => 'People Roof' ],
			[ 'fa fa-solid fa-person-hiking' => 'Person Hiking' ],
			[ 'fa fa-solid fa-person-shelter' => 'Person Shelter' ],
			[ 'fa fa-solid fa-route' => 'Route' ],
			[ 'fa fa-solid fa-signs-post' => 'Signs Post' ],
			[ 'fa fa-solid fa-tarp' => 'Tarp' ],
			[ 'fa fa-solid fa-tarp-droplet' => 'Tarp Droplet' ],
			[ 'fa fa-solid fa-tent' => 'Tent' ],
			[ 'fa fa-solid fa-tent-arrow-down-to-line' => 'Tent Arrow Down To Line' ],
			[ 'fa fa-solid fa-tent-arrow-left-right' => 'Tent Arrow Left Right' ],
			[ 'fa fa-solid fa-tent-arrow-turn-left' => 'Tent Arrow Turn Left' ],
			[ 'fa fa-solid fa-tent-arrows-down' => 'Tent Arrows Down' ],
			[ 'fa fa-solid fa-tents' => 'Tents' ],
			[ 'fa fa-solid fa-toilet-paper' => 'Toilet Paper' ],
			[ 'fa fa-solid fa-trailer' => 'Trailer' ],
			[ 'fa fa-solid fa-tree' => 'Tree' ],
		],
		'Charity' => [
			[ 'fa fa-solid fa-circle-dollar-to-slot' => 'Circle Dollar To Slot' ],
			[ 'fa fa-solid fa-dollar-sign' => 'Dollar Sign' ],
			[ 'fa fa-solid fa-dove' => 'Dove' ],
			[ 'fa fa-solid fa-gift' => 'Gift' ],
			[ 'fa fa-solid fa-globe' => 'Globe' ],
			[ 'fa fa-solid fa-hand-holding-dollar' => 'Hand Holding Dollar' ],
			[ 'fa fa-solid fa-hand-holding-droplet' => 'Hand Holding Droplet' ],
			[ 'fa fa-solid fa-hand-holding-hand' => 'Hand Holding Hand' ],
			[ 'fa fa-solid fa-hand-holding-heart' => 'Hand Holding Heart' ],
			[ 'fa fa-solid fa-hands-holding-child' => 'Hands Holding Child' ],
			[ 'fa fa-solid fa-hands-holding-circle' => 'Hands Holding Circle' ],
			[ 'fa fa-solid fa-handshake' => 'Handshake' ],
			[ 'fa fa-regular fa-handshake' => 'Handshake' ],
			[ 'fa fa-solid fa-handshake-angle' => 'Handshake Angle' ],
			[ 'fa fa-solid fa-handshake-simple' => 'Handshake Simple' ],
			[ 'fa fa-solid fa-heart' => 'Heart' ],
			[ 'fa fa-regular fa-heart' => 'Heart' ],
			[ 'fa fa-solid fa-leaf' => 'Leaf' ],
			[ 'fa fa-solid fa-parachute-box' => 'Parachute Box' ],
			[ 'fa fa-solid fa-piggy-bank' => 'Piggy Bank' ],
			[ 'fa fa-solid fa-ribbon' => 'Ribbon' ],
			[ 'fa fa-solid fa-seedling' => 'Seedling' ],
		],
		'Charts-diagrams' => [
			[ 'fa fa-solid fa-bars-progress' => 'Bars Progress' ],
			[ 'fa fa-solid fa-chart-area' => 'Chart Area' ],
			[ 'fa fa-solid fa-chart-bar' => 'Chart Bar' ],
			[ 'fa fa-regular fa-chart-bar' => 'Chart Bar' ],
			[ 'fa fa-solid fa-chart-column' => 'Chart Column' ],
			[ 'fa fa-solid fa-chart-diagram' => 'Chart Diagram' ],
			[ 'fa fa-solid fa-chart-gantt' => 'Chart Gantt' ],
			[ 'fa fa-solid fa-chart-line' => 'Chart Line' ],
			[ 'fa fa-solid fa-chart-pie' => 'Chart Pie' ],
			[ 'fa fa-solid fa-chart-simple' => 'Chart Simple' ],
			[ 'fa fa-solid fa-circle-half-stroke' => 'Circle Half Stroke' ],
			[ 'fa fa-solid fa-diagram-next' => 'Diagram Next' ],
			[ 'fa fa-solid fa-diagram-predecessor' => 'Diagram Predecessor' ],
			[ 'fa fa-solid fa-diagram-project' => 'Diagram Project' ],
			[ 'fa fa-solid fa-diagram-successor' => 'Diagram Successor' ],
			[ 'fa fa-solid fa-hexagon-nodes' => 'Hexagon Nodes' ],
			[ 'fa fa-solid fa-hexagon-nodes-bolt' => 'Hexagon Nodes Bolt' ],
			[ 'fa fa-solid fa-square-poll-horizontal' => 'Square Poll Horizontal' ],
			[ 'fa fa-solid fa-square-poll-vertical' => 'Square Poll Vertical' ],
		],
		'Childhood' => [
			[ 'fa fa-solid fa-apple-whole' => 'Apple Whole' ],
			[ 'fa fa-solid fa-baby' => 'Baby' ],
			[ 'fa fa-solid fa-baby-carriage' => 'Baby Carriage' ],
			[ 'fa fa-solid fa-baseball-bat-ball' => 'Baseball Bat Ball' ],
			[ 'fa fa-solid fa-bath' => 'Bath' ],
			[ 'fa fa-solid fa-bucket' => 'Bucket' ],
			[ 'fa fa-solid fa-cake-candles' => 'Cake Candles' ],
			[ 'fa fa-solid fa-child' => 'Child' ],
			[ 'fa fa-solid fa-child-dress' => 'Child Dress' ],
			[ 'fa fa-solid fa-child-reaching' => 'Child Reaching' ],
			[ 'fa fa-solid fa-children' => 'Children' ],
			[ 'fa fa-solid fa-cookie' => 'Cookie' ],
			[ 'fa fa-solid fa-cookie-bite' => 'Cookie Bite' ],
			[ 'fa fa-solid fa-cubes-stacked' => 'Cubes Stacked' ],
			[ 'fa fa-solid fa-gamepad' => 'Gamepad' ],
			[ 'fa fa-solid fa-hands-holding-child' => 'Hands Holding Child' ],
			[ 'fa fa-solid fa-ice-cream' => 'Ice Cream' ],
			[ 'fa fa-solid fa-mitten' => 'Mitten' ],
			[ 'fa fa-solid fa-person-biking' => 'Person Biking' ],
			[ 'fa fa-solid fa-person-breastfeeding' => 'Person Breastfeeding' ],
			[ 'fa fa-solid fa-puzzle-piece' => 'Puzzle Piece' ],
			[ 'fa fa-solid fa-robot' => 'Robot' ],
			[ 'fa fa-solid fa-school' => 'School' ],
			[ 'fa fa-solid fa-shapes' => 'Shapes' ],
			[ 'fa fa-solid fa-snowman' => 'Snowman' ],
		],
		'Clothing-fashion' => [
			[ 'fa fa-solid fa-glasses' => 'Glasses' ],
			[ 'fa fa-solid fa-graduation-cap' => 'Graduation Cap' ],
			[ 'fa fa-solid fa-hat-cowboy' => 'Hat Cowboy' ],
			[ 'fa fa-solid fa-hat-cowboy-side' => 'Hat Cowboy Side' ],
			[ 'fa fa-solid fa-hat-wizard' => 'Hat Wizard' ],
			[ 'fa fa-solid fa-mitten' => 'Mitten' ],
			[ 'fa fa-solid fa-shirt' => 'Shirt' ],
			[ 'fa fa-solid fa-shoe-prints' => 'Shoe Prints' ],
			[ 'fa fa-solid fa-socks' => 'Socks' ],
			[ 'fa fa-solid fa-user-tie' => 'User Tie' ],
			[ 'fa fa-solid fa-vest' => 'Vest' ],
			[ 'fa fa-solid fa-vest-patches' => 'Vest Patches' ],
		],
		'Coding' => [
			[ 'fa fa-solid fa-barcode' => 'Barcode' ],
			[ 'fa fa-solid fa-bars' => 'Bars' ],
			[ 'fa fa-solid fa-bars-staggered' => 'Bars Staggered' ],
			[ 'fa fa-solid fa-bath' => 'Bath' ],
			[ 'fa fa-solid fa-box-archive' => 'Box Archive' ],
			[ 'fa fa-solid fa-bug' => 'Bug' ],
			[ 'fa fa-solid fa-bug-slash' => 'Bug Slash' ],
			[ 'fa fa-solid fa-chart-diagram' => 'Chart Diagram' ],
			[ 'fa fa-solid fa-circle-nodes' => 'Circle Nodes' ],
			[ 'fa fa-solid fa-code' => 'Code' ],
			[ 'fa fa-solid fa-code-branch' => 'Code Branch' ],
			[ 'fa fa-solid fa-code-commit' => 'Code Commit' ],
			[ 'fa fa-solid fa-code-compare' => 'Code Compare' ],
			[ 'fa fa-solid fa-code-fork' => 'Code Fork' ],
			[ 'fa fa-solid fa-code-merge' => 'Code Merge' ],
			[ 'fa fa-solid fa-code-pull-request' => 'Code Pull Request' ],
			[ 'fa fa-solid fa-comment-nodes' => 'Comment Nodes' ],
			[ 'fa fa-brands fa-css' => 'Css' ],
			[ 'fa fa-solid fa-cube' => 'Cube' ],
			[ 'fa fa-solid fa-cubes' => 'Cubes' ],
			[ 'fa fa-solid fa-diagram-project' => 'Diagram Project' ],
			[ 'fa fa-solid fa-file' => 'File' ],
			[ 'fa fa-regular fa-file' => 'File' ],
			[ 'fa fa-solid fa-file-code' => 'File Code' ],
			[ 'fa fa-regular fa-file-code' => 'File Code' ],
			[ 'fa fa-solid fa-file-lines' => 'File Lines' ],
			[ 'fa fa-regular fa-file-lines' => 'File Lines' ],
			[ 'fa fa-solid fa-filter' => 'Filter' ],
			[ 'fa fa-solid fa-fire-extinguisher' => 'Fire Extinguisher' ],
			[ 'fa fa-solid fa-folder' => 'Folder' ],
			[ 'fa fa-regular fa-folder' => 'Folder' ],
			[ 'fa fa-solid fa-folder-open' => 'Folder Open' ],
			[ 'fa fa-regular fa-folder-open' => 'Folder Open' ],
			[ 'fa fa-solid fa-font-awesome' => 'Font Awesome' ],
			[ 'fa fa-regular fa-font-awesome' => 'Font Awesome' ],
			[ 'fa fa-brands fa-font-awesome' => 'Font Awesome' ],
			[ 'fa fa-solid fa-gear' => 'Gear' ],
			[ 'fa fa-solid fa-gears' => 'Gears' ],
			[ 'fa fa-solid fa-hexagon-nodes' => 'Hexagon Nodes' ],
			[ 'fa fa-solid fa-hexagon-nodes-bolt' => 'Hexagon Nodes Bolt' ],
			[ 'fa fa-solid fa-keyboard' => 'Keyboard' ],
			[ 'fa fa-regular fa-keyboard' => 'Keyboard' ],
			[ 'fa fa-solid fa-laptop-code' => 'Laptop Code' ],
			[ 'fa fa-solid fa-microchip' => 'Microchip' ],
			[ 'fa fa-solid fa-mug-saucer' => 'Mug Saucer' ],
			[ 'fa fa-solid fa-network-wired' => 'Network Wired' ],
			[ 'fa fa-solid fa-notdef' => 'Notdef' ],
			[ 'fa fa-solid fa-qrcode' => 'Qrcode' ],
			[ 'fa fa-solid fa-rectangle-xmark' => 'Rectangle Xmark' ],
			[ 'fa fa-regular fa-rectangle-xmark' => 'Rectangle Xmark' ],
			[ 'fa fa-solid fa-shield' => 'Shield' ],
			[ 'fa fa-solid fa-shield-halved' => 'Shield Halved' ],
			[ 'fa fa-solid fa-sitemap' => 'Sitemap' ],
			[ 'fa fa-solid fa-square-binary' => 'Square Binary' ],
			[ 'fa fa-solid fa-terminal' => 'Terminal' ],
			[ 'fa fa-solid fa-user-secret' => 'User Secret' ],
			[ 'fa fa-solid fa-web-awesome' => 'Web Awesome' ],
			[ 'fa fa-brands fa-web-awesome' => 'Web Awesome' ],
			[ 'fa fa-solid fa-window-maximize' => 'Window Maximize' ],
			[ 'fa fa-regular fa-window-maximize' => 'Window Maximize' ],
			[ 'fa fa-solid fa-window-minimize' => 'Window Minimize' ],
			[ 'fa fa-regular fa-window-minimize' => 'Window Minimize' ],
			[ 'fa fa-solid fa-window-restore' => 'Window Restore' ],
			[ 'fa fa-regular fa-window-restore' => 'Window Restore' ],
		],
		'Communication' => [
			[ 'fa fa-solid fa-address-book' => 'Address Book' ],
			[ 'fa fa-regular fa-address-book' => 'Address Book' ],
			[ 'fa fa-solid fa-address-card' => 'Address Card' ],
			[ 'fa fa-regular fa-address-card' => 'Address Card' ],
			[ 'fa fa-solid fa-at' => 'At' ],
			[ 'fa fa-solid fa-blender-phone' => 'Blender Phone' ],
			[ 'fa fa-brands fa-bluetooth-b' => 'Bluetooth B' ],
			[ 'fa fa-solid fa-bullhorn' => 'Bullhorn' ],
			[ 'fa fa-solid fa-comment' => 'Comment' ],
			[ 'fa fa-regular fa-comment' => 'Comment' ],
			[ 'fa fa-solid fa-comment-dots' => 'Comment Dots' ],
			[ 'fa fa-regular fa-comment-dots' => 'Comment Dots' ],
			[ 'fa fa-solid fa-comment-medical' => 'Comment Medical' ],
			[ 'fa fa-solid fa-comment-nodes' => 'Comment Nodes' ],
			[ 'fa fa-solid fa-comment-slash' => 'Comment Slash' ],
			[ 'fa fa-solid fa-comment-sms' => 'Comment Sms' ],
			[ 'fa fa-solid fa-comments' => 'Comments' ],
			[ 'fa fa-regular fa-comments' => 'Comments' ],
			[ 'fa fa-solid fa-ear-listen' => 'Ear Listen' ],
			[ 'fa fa-solid fa-envelope' => 'Envelope' ],
			[ 'fa fa-regular fa-envelope' => 'Envelope' ],
			[ 'fa fa-solid fa-envelope-circle-check' => 'Envelope Circle Check' ],
			[ 'fa fa-solid fa-envelope-open' => 'Envelope Open' ],
			[ 'fa fa-regular fa-envelope-open' => 'Envelope Open' ],
			[ 'fa fa-solid fa-face-frown' => 'Face Frown' ],
			[ 'fa fa-regular fa-face-frown' => 'Face Frown' ],
			[ 'fa fa-solid fa-face-meh' => 'Face Meh' ],
			[ 'fa fa-regular fa-face-meh' => 'Face Meh' ],
			[ 'fa fa-solid fa-face-smile' => 'Face Smile' ],
			[ 'fa fa-regular fa-face-smile' => 'Face Smile' ],
			[ 'fa fa-solid fa-fax' => 'Fax' ],
			[ 'fa fa-solid fa-hands-asl-interpreting' => 'Hands Asl Interpreting' ],
			[ 'fa fa-solid fa-icons' => 'Icons' ],
			[ 'fa fa-solid fa-inbox' => 'Inbox' ],
			[ 'fa fa-solid fa-language' => 'Language' ],
			[ 'fa fa-solid fa-message' => 'Message' ],
			[ 'fa fa-regular fa-message' => 'Message' ],
			[ 'fa fa-solid fa-microphone' => 'Microphone' ],
			[ 'fa fa-solid fa-microphone-lines' => 'Microphone Lines' ],
			[ 'fa fa-solid fa-microphone-lines-slash' => 'Microphone Lines Slash' ],
			[ 'fa fa-solid fa-microphone-slash' => 'Microphone Slash' ],
			[ 'fa fa-solid fa-mobile' => 'Mobile' ],
			[ 'fa fa-solid fa-mobile-button' => 'Mobile Button' ],
			[ 'fa fa-solid fa-mobile-retro' => 'Mobile Retro' ],
			[ 'fa fa-solid fa-mobile-screen' => 'Mobile Screen' ],
			[ 'fa fa-solid fa-mobile-screen-button' => 'Mobile Screen Button' ],
			[ 'fa fa-solid fa-paper-plane' => 'Paper Plane' ],
			[ 'fa fa-regular fa-paper-plane' => 'Paper Plane' ],
			[ 'fa fa-solid fa-phone' => 'Phone' ],
			[ 'fa fa-solid fa-phone-flip' => 'Phone Flip' ],
			[ 'fa fa-solid fa-phone-slash' => 'Phone Slash' ],
			[ 'fa fa-solid fa-phone-volume' => 'Phone Volume' ],
			[ 'fa fa-solid fa-poo' => 'Poo' ],
			[ 'fa fa-solid fa-quote-left' => 'Quote Left' ],
			[ 'fa fa-solid fa-quote-right' => 'Quote Right' ],
			[ 'fa fa-solid fa-square-envelope' => 'Square Envelope' ],
			[ 'fa fa-solid fa-square-phone' => 'Square Phone' ],
			[ 'fa fa-solid fa-square-phone-flip' => 'Square Phone Flip' ],
			[ 'fa fa-solid fa-square-rss' => 'Square Rss' ],
			[ 'fa fa-solid fa-tower-cell' => 'Tower Cell' ],
			[ 'fa fa-solid fa-tty' => 'Tty' ],
			[ 'fa fa-solid fa-video' => 'Video' ],
			[ 'fa fa-solid fa-video-slash' => 'Video Slash' ],
			[ 'fa fa-solid fa-voicemail' => 'Voicemail' ],
			[ 'fa fa-solid fa-walkie-talkie' => 'Walkie Talkie' ],
		],
		'Connectivity' => [
			[ 'fa fa-brands fa-bluetooth' => 'Bluetooth' ],
			[ 'fa fa-solid fa-circle-nodes' => 'Circle Nodes' ],
			[ 'fa fa-solid fa-cloud' => 'Cloud' ],
			[ 'fa fa-solid fa-cloud-arrow-down' => 'Cloud Arrow Down' ],
			[ 'fa fa-solid fa-cloud-arrow-up' => 'Cloud Arrow Up' ],
			[ 'fa fa-solid fa-ethernet' => 'Ethernet' ],
			[ 'fa fa-solid fa-globe' => 'Globe' ],
			[ 'fa fa-solid fa-house-signal' => 'House Signal' ],
			[ 'fa fa-solid fa-rss' => 'Rss' ],
			[ 'fa fa-solid fa-satellite-dish' => 'Satellite Dish' ],
			[ 'fa fa-solid fa-signal' => 'Signal' ],
			[ 'fa fa-solid fa-tower-broadcast' => 'Tower Broadcast' ],
			[ 'fa fa-solid fa-tower-cell' => 'Tower Cell' ],
			[ 'fa fa-solid fa-wifi' => 'Wifi' ],
		],
		'Construction' => [
			[ 'fa fa-solid fa-arrow-up-from-ground-water' => 'Arrow Up From Ground Water' ],
			[ 'fa fa-solid fa-bore-hole' => 'Bore Hole' ],
			[ 'fa fa-solid fa-brush' => 'Brush' ],
			[ 'fa fa-solid fa-bucket' => 'Bucket' ],
			[ 'fa fa-solid fa-compass-drafting' => 'Compass Drafting' ],
			[ 'fa fa-solid fa-dumpster' => 'Dumpster' ],
			[ 'fa fa-solid fa-dumpster-fire' => 'Dumpster Fire' ],
			[ 'fa fa-solid fa-hammer' => 'Hammer' ],
			[ 'fa fa-solid fa-helmet-safety' => 'Helmet Safety' ],
			[ 'fa fa-solid fa-mound' => 'Mound' ],
			[ 'fa fa-solid fa-paint-roller' => 'Paint Roller' ],
			[ 'fa fa-solid fa-pen-ruler' => 'Pen Ruler' ],
			[ 'fa fa-solid fa-pencil' => 'Pencil' ],
			[ 'fa fa-solid fa-person-digging' => 'Person Digging' ],
			[ 'fa fa-solid fa-ruler' => 'Ruler' ],
			[ 'fa fa-solid fa-ruler-combined' => 'Ruler Combined' ],
			[ 'fa fa-solid fa-ruler-horizontal' => 'Ruler Horizontal' ],
			[ 'fa fa-solid fa-ruler-vertical' => 'Ruler Vertical' ],
			[ 'fa fa-solid fa-screwdriver' => 'Screwdriver' ],
			[ 'fa fa-solid fa-screwdriver-wrench' => 'Screwdriver Wrench' ],
			[ 'fa fa-solid fa-sheet-plastic' => 'Sheet Plastic' ],
			[ 'fa fa-solid fa-tarp' => 'Tarp' ],
			[ 'fa fa-solid fa-tarp-droplet' => 'Tarp Droplet' ],
			[ 'fa fa-solid fa-toilet-portable' => 'Toilet Portable' ],
			[ 'fa fa-solid fa-toilets-portable' => 'Toilets Portable' ],
			[ 'fa fa-solid fa-toolbox' => 'Toolbox' ],
			[ 'fa fa-solid fa-trowel' => 'Trowel' ],
			[ 'fa fa-solid fa-trowel-bricks' => 'Trowel Bricks' ],
			[ 'fa fa-solid fa-truck-pickup' => 'Truck Pickup' ],
			[ 'fa fa-solid fa-wrench' => 'Wrench' ],
		],
		'Design' => [
			[ 'fa fa-solid fa-bezier-curve' => 'Bezier Curve' ],
			[ 'fa fa-solid fa-brush' => 'Brush' ],
			[ 'fa fa-solid fa-circle-half-stroke' => 'Circle Half Stroke' ],
			[ 'fa fa-solid fa-circle-nodes' => 'Circle Nodes' ],
			[ 'fa fa-solid fa-clone' => 'Clone' ],
			[ 'fa fa-regular fa-clone' => 'Clone' ],
			[ 'fa fa-solid fa-compass-drafting' => 'Compass Drafting' ],
			[ 'fa fa-solid fa-copy' => 'Copy' ],
			[ 'fa fa-regular fa-copy' => 'Copy' ],
			[ 'fa fa-solid fa-crop' => 'Crop' ],
			[ 'fa fa-solid fa-crop-simple' => 'Crop Simple' ],
			[ 'fa fa-solid fa-crosshairs' => 'Crosshairs' ],
			[ 'fa fa-solid fa-cube' => 'Cube' ],
			[ 'fa fa-solid fa-cubes' => 'Cubes' ],
			[ 'fa fa-solid fa-draw-polygon' => 'Draw Polygon' ],
			[ 'fa fa-solid fa-droplet' => 'Droplet' ],
			[ 'fa fa-solid fa-droplet-slash' => 'Droplet Slash' ],
			[ 'fa fa-solid fa-eraser' => 'Eraser' ],
			[ 'fa fa-solid fa-eye' => 'Eye' ],
			[ 'fa fa-regular fa-eye' => 'Eye' ],
			[ 'fa fa-solid fa-eye-dropper' => 'Eye Dropper' ],
			[ 'fa fa-solid fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-regular fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-solid fa-fill' => 'Fill' ],
			[ 'fa fa-solid fa-fill-drip' => 'Fill Drip' ],
			[ 'fa fa-solid fa-floppy-disk' => 'Floppy Disk' ],
			[ 'fa fa-regular fa-floppy-disk' => 'Floppy Disk' ],
			[ 'fa fa-solid fa-font-awesome' => 'Font Awesome' ],
			[ 'fa fa-regular fa-font-awesome' => 'Font Awesome' ],
			[ 'fa fa-brands fa-font-awesome' => 'Font Awesome' ],
			[ 'fa fa-solid fa-highlighter' => 'Highlighter' ],
			[ 'fa fa-solid fa-icons' => 'Icons' ],
			[ 'fa fa-solid fa-layer-group' => 'Layer Group' ],
			[ 'fa fa-solid fa-lines-leaning' => 'Lines Leaning' ],
			[ 'fa fa-solid fa-marker' => 'Marker' ],
			[ 'fa fa-solid fa-object-group' => 'Object Group' ],
			[ 'fa fa-regular fa-object-group' => 'Object Group' ],
			[ 'fa fa-solid fa-object-ungroup' => 'Object Ungroup' ],
			[ 'fa fa-regular fa-object-ungroup' => 'Object Ungroup' ],
			[ 'fa fa-solid fa-paint-roller' => 'Paint Roller' ],
			[ 'fa fa-solid fa-paintbrush' => 'Paintbrush' ],
			[ 'fa fa-solid fa-palette' => 'Palette' ],
			[ 'fa fa-solid fa-paste' => 'Paste' ],
			[ 'fa fa-regular fa-paste' => 'Paste' ],
			[ 'fa fa-solid fa-pen' => 'Pen' ],
			[ 'fa fa-solid fa-pen-clip' => 'Pen Clip' ],
			[ 'fa fa-solid fa-pen-fancy' => 'Pen Fancy' ],
			[ 'fa fa-solid fa-pen-nib' => 'Pen Nib' ],
			[ 'fa fa-solid fa-pen-ruler' => 'Pen Ruler' ],
			[ 'fa fa-solid fa-pen-to-square' => 'Pen To Square' ],
			[ 'fa fa-regular fa-pen-to-square' => 'Pen To Square' ],
			[ 'fa fa-solid fa-pencil' => 'Pencil' ],
			[ 'fa fa-solid fa-ruler-combined' => 'Ruler Combined' ],
			[ 'fa fa-solid fa-ruler-horizontal' => 'Ruler Horizontal' ],
			[ 'fa fa-solid fa-ruler-vertical' => 'Ruler Vertical' ],
			[ 'fa fa-solid fa-scissors' => 'Scissors' ],
			[ 'fa fa-solid fa-splotch' => 'Splotch' ],
			[ 'fa fa-solid fa-spray-can' => 'Spray Can' ],
			[ 'fa fa-solid fa-stamp' => 'Stamp' ],
			[ 'fa fa-solid fa-stapler' => 'Stapler' ],
			[ 'fa fa-solid fa-swatchbook' => 'Swatchbook' ],
			[ 'fa fa-solid fa-vector-square' => 'Vector Square' ],
			[ 'fa fa-solid fa-wand-magic' => 'Wand Magic' ],
			[ 'fa fa-solid fa-wand-magic-sparkles' => 'Wand Magic Sparkles' ],
			[ 'fa fa-solid fa-web-awesome' => 'Web Awesome' ],
			[ 'fa fa-brands fa-web-awesome' => 'Web Awesome' ],
		],
		'Devices-hardware' => [
			[ 'fa fa-solid fa-blender-phone' => 'Blender Phone' ],
			[ 'fa fa-solid fa-camera' => 'Camera' ],
			[ 'fa fa-solid fa-camera-retro' => 'Camera Retro' ],
			[ 'fa fa-solid fa-car-battery' => 'Car Battery' ],
			[ 'fa fa-solid fa-compact-disc' => 'Compact Disc' ],
			[ 'fa fa-solid fa-computer' => 'Computer' ],
			[ 'fa fa-solid fa-computer-mouse' => 'Computer Mouse' ],
			[ 'fa fa-solid fa-database' => 'Database' ],
			[ 'fa fa-solid fa-desktop' => 'Desktop' ],
			[ 'fa fa-solid fa-display' => 'Display' ],
			[ 'fa fa-solid fa-download' => 'Download' ],
			[ 'fa fa-solid fa-ethernet' => 'Ethernet' ],
			[ 'fa fa-solid fa-fax' => 'Fax' ],
			[ 'fa fa-solid fa-floppy-disk' => 'Floppy Disk' ],
			[ 'fa fa-regular fa-floppy-disk' => 'Floppy Disk' ],
			[ 'fa fa-solid fa-gamepad' => 'Gamepad' ],
			[ 'fa fa-solid fa-hard-drive' => 'Hard Drive' ],
			[ 'fa fa-regular fa-hard-drive' => 'Hard Drive' ],
			[ 'fa fa-solid fa-headphones' => 'Headphones' ],
			[ 'fa fa-solid fa-house-laptop' => 'House Laptop' ],
			[ 'fa fa-solid fa-keyboard' => 'Keyboard' ],
			[ 'fa fa-regular fa-keyboard' => 'Keyboard' ],
			[ 'fa fa-solid fa-laptop' => 'Laptop' ],
			[ 'fa fa-solid fa-laptop-file' => 'Laptop File' ],
			[ 'fa fa-solid fa-memory' => 'Memory' ],
			[ 'fa fa-solid fa-microchip' => 'Microchip' ],
			[ 'fa fa-solid fa-mobile' => 'Mobile' ],
			[ 'fa fa-solid fa-mobile-button' => 'Mobile Button' ],
			[ 'fa fa-solid fa-mobile-retro' => 'Mobile Retro' ],
			[ 'fa fa-solid fa-mobile-screen' => 'Mobile Screen' ],
			[ 'fa fa-solid fa-mobile-screen-button' => 'Mobile Screen Button' ],
			[ 'fa fa-solid fa-plug' => 'Plug' ],
			[ 'fa fa-solid fa-power-off' => 'Power Off' ],
			[ 'fa fa-solid fa-print' => 'Print' ],
			[ 'fa fa-solid fa-satellite' => 'Satellite' ],
			[ 'fa fa-solid fa-satellite-dish' => 'Satellite Dish' ],
			[ 'fa fa-solid fa-sd-card' => 'Sd Card' ],
			[ 'fa fa-solid fa-server' => 'Server' ],
			[ 'fa fa-solid fa-sim-card' => 'Sim Card' ],
			[ 'fa fa-solid fa-tablet' => 'Tablet' ],
			[ 'fa fa-solid fa-tablet-button' => 'Tablet Button' ],
			[ 'fa fa-solid fa-tablet-screen-button' => 'Tablet Screen Button' ],
			[ 'fa fa-solid fa-tachograph-digital' => 'Tachograph Digital' ],
			[ 'fa fa-solid fa-tv' => 'Tv' ],
			[ 'fa fa-solid fa-upload' => 'Upload' ],
			[ 'fa fa-solid fa-walkie-talkie' => 'Walkie Talkie' ],
		],
		'Disaster' => [
			[ 'fa fa-solid fa-biohazard' => 'Biohazard' ],
			[ 'fa fa-solid fa-bugs' => 'Bugs' ],
			[ 'fa fa-solid fa-burst' => 'Burst' ],
			[ 'fa fa-solid fa-child-combatant' => 'Child Combatant' ],
			[ 'fa fa-solid fa-circle-radiation' => 'Circle Radiation' ],
			[ 'fa fa-solid fa-cloud-bolt' => 'Cloud Bolt' ],
			[ 'fa fa-solid fa-cloud-showers-heavy' => 'Cloud Showers Heavy' ],
			[ 'fa fa-solid fa-cloud-showers-water' => 'Cloud Showers Water' ],
			[ 'fa fa-solid fa-helmet-un' => 'Helmet Un' ],
			[ 'fa fa-solid fa-hill-avalanche' => 'Hill Avalanche' ],
			[ 'fa fa-solid fa-hill-rockslide' => 'Hill Rockslide' ],
			[ 'fa fa-solid fa-house-chimney-crack' => 'House Chimney Crack' ],
			[ 'fa fa-solid fa-house-crack' => 'House Crack' ],
			[ 'fa fa-solid fa-house-fire' => 'House Fire' ],
			[ 'fa fa-solid fa-house-flood-water' => 'House Flood Water' ],
			[ 'fa fa-solid fa-house-flood-water-circle-arrow-right' => 'House Flood Water Circle Arrow Right' ],
			[ 'fa fa-solid fa-house-tsunami' => 'House Tsunami' ],
			[ 'fa fa-solid fa-hurricane' => 'Hurricane' ],
			[ 'fa fa-solid fa-locust' => 'Locust' ],
			[ 'fa fa-solid fa-mosquito' => 'Mosquito' ],
			[ 'fa fa-solid fa-person-drowning' => 'Person Drowning' ],
			[ 'fa fa-solid fa-person-rifle' => 'Person Rifle' ],
			[ 'fa fa-solid fa-person-walking-arrow-loop-left' => 'Person Walking Arrow Loop Left' ],
			[ 'fa fa-solid fa-person-walking-arrow-right' => 'Person Walking Arrow Right' ],
			[ 'fa fa-solid fa-person-walking-dashed-line-arrow-right' => 'Person Walking Dashed Line Arrow Right' ],
			[ 'fa fa-solid fa-plant-wilt' => 'Plant Wilt' ],
			[ 'fa fa-solid fa-radiation' => 'Radiation' ],
			[ 'fa fa-solid fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-regular fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-solid fa-sun-plant-wilt' => 'Sun Plant Wilt' ],
			[ 'fa fa-solid fa-temperature-arrow-down' => 'Temperature Arrow Down' ],
			[ 'fa fa-solid fa-temperature-arrow-up' => 'Temperature Arrow Up' ],
			[ 'fa fa-solid fa-tornado' => 'Tornado' ],
			[ 'fa fa-solid fa-volcano' => 'Volcano' ],
			[ 'fa fa-solid fa-wheat-awn-circle-exclamation' => 'Wheat Awn Circle Exclamation' ],
			[ 'fa fa-solid fa-wind' => 'Wind' ],
			[ 'fa fa-solid fa-worm' => 'Worm' ],
			[ 'fa fa-solid fa-xmarks-lines' => 'Xmarks Lines' ],
		],
		'Editing' => [
			[ 'fa fa-solid fa-arrows-rotate' => 'Arrows Rotate' ],
			[ 'fa fa-solid fa-bandage' => 'Bandage' ],
			[ 'fa fa-solid fa-bars' => 'Bars' ],
			[ 'fa fa-solid fa-brush' => 'Brush' ],
			[ 'fa fa-solid fa-chart-simple' => 'Chart Simple' ],
			[ 'fa fa-solid fa-check' => 'Check' ],
			[ 'fa fa-solid fa-check-double' => 'Check Double' ],
			[ 'fa fa-solid fa-circle-check' => 'Circle Check' ],
			[ 'fa fa-regular fa-circle-check' => 'Circle Check' ],
			[ 'fa fa-solid fa-circle-half-stroke' => 'Circle Half Stroke' ],
			[ 'fa fa-solid fa-crop' => 'Crop' ],
			[ 'fa fa-solid fa-crop-simple' => 'Crop Simple' ],
			[ 'fa fa-solid fa-cube' => 'Cube' ],
			[ 'fa fa-solid fa-delete-left' => 'Delete Left' ],
			[ 'fa fa-solid fa-ellipsis' => 'Ellipsis' ],
			[ 'fa fa-solid fa-ellipsis-vertical' => 'Ellipsis Vertical' ],
			[ 'fa fa-solid fa-eye-dropper' => 'Eye Dropper' ],
			[ 'fa fa-solid fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-regular fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-solid fa-gear' => 'Gear' ],
			[ 'fa fa-solid fa-grip' => 'Grip' ],
			[ 'fa fa-solid fa-grip-lines' => 'Grip Lines' ],
			[ 'fa fa-solid fa-grip-lines-vertical' => 'Grip Lines Vertical' ],
			[ 'fa fa-solid fa-grip-vertical' => 'Grip Vertical' ],
			[ 'fa fa-solid fa-link' => 'Link' ],
			[ 'fa fa-solid fa-link-slash' => 'Link Slash' ],
			[ 'fa fa-solid fa-minus' => 'Minus' ],
			[ 'fa fa-solid fa-paintbrush' => 'Paintbrush' ],
			[ 'fa fa-solid fa-pen' => 'Pen' ],
			[ 'fa fa-solid fa-pen-clip' => 'Pen Clip' ],
			[ 'fa fa-solid fa-pen-fancy' => 'Pen Fancy' ],
			[ 'fa fa-solid fa-pen-nib' => 'Pen Nib' ],
			[ 'fa fa-solid fa-pen-ruler' => 'Pen Ruler' ],
			[ 'fa fa-solid fa-pen-to-square' => 'Pen To Square' ],
			[ 'fa fa-regular fa-pen-to-square' => 'Pen To Square' ],
			[ 'fa fa-solid fa-pencil' => 'Pencil' ],
			[ 'fa fa-solid fa-plus' => 'Plus' ],
			[ 'fa fa-solid fa-rotate' => 'Rotate' ],
			[ 'fa fa-solid fa-scissors' => 'Scissors' ],
			[ 'fa fa-solid fa-signature' => 'Signature' ],
			[ 'fa fa-solid fa-sliders' => 'Sliders' ],
			[ 'fa fa-solid fa-square-check' => 'Square Check' ],
			[ 'fa fa-regular fa-square-check' => 'Square Check' ],
			[ 'fa fa-solid fa-square-pen' => 'Square Pen' ],
			[ 'fa fa-solid fa-trash' => 'Trash' ],
			[ 'fa fa-solid fa-trash-arrow-up' => 'Trash Arrow Up' ],
			[ 'fa fa-solid fa-trash-can' => 'Trash Can' ],
			[ 'fa fa-regular fa-trash-can' => 'Trash Can' ],
			[ 'fa fa-solid fa-trash-can-arrow-up' => 'Trash Can Arrow Up' ],
			[ 'fa fa-solid fa-wand-magic' => 'Wand Magic' ],
			[ 'fa fa-solid fa-wand-magic-sparkles' => 'Wand Magic Sparkles' ],
			[ 'fa fa-solid fa-xmark' => 'Xmark' ],
		],
		'Education' => [
			[ 'fa fa-solid fa-apple-whole' => 'Apple Whole' ],
			[ 'fa fa-solid fa-atom' => 'Atom' ],
			[ 'fa fa-solid fa-award' => 'Award' ],
			[ 'fa fa-solid fa-bell' => 'Bell' ],
			[ 'fa fa-regular fa-bell' => 'Bell' ],
			[ 'fa fa-solid fa-bell-slash' => 'Bell Slash' ],
			[ 'fa fa-regular fa-bell-slash' => 'Bell Slash' ],
			[ 'fa fa-solid fa-book-open' => 'Book Open' ],
			[ 'fa fa-solid fa-book-open-reader' => 'Book Open Reader' ],
			[ 'fa fa-solid fa-chalkboard' => 'Chalkboard' ],
			[ 'fa fa-solid fa-chalkboard-user' => 'Chalkboard User' ],
			[ 'fa fa-solid fa-graduation-cap' => 'Graduation Cap' ],
			[ 'fa fa-solid fa-laptop-code' => 'Laptop Code' ],
			[ 'fa fa-solid fa-laptop-file' => 'Laptop File' ],
			[ 'fa fa-solid fa-masks-theater' => 'Masks Theater' ],
			[ 'fa fa-solid fa-microscope' => 'Microscope' ],
			[ 'fa fa-solid fa-music' => 'Music' ],
			[ 'fa fa-solid fa-person-chalkboard' => 'Person Chalkboard' ],
			[ 'fa fa-solid fa-school' => 'School' ],
			[ 'fa fa-solid fa-school-circle-check' => 'School Circle Check' ],
			[ 'fa fa-solid fa-school-circle-exclamation' => 'School Circle Exclamation' ],
			[ 'fa fa-solid fa-school-circle-xmark' => 'School Circle Xmark' ],
			[ 'fa fa-solid fa-school-flag' => 'School Flag' ],
			[ 'fa fa-solid fa-school-lock' => 'School Lock' ],
			[ 'fa fa-solid fa-shapes' => 'Shapes' ],
			[ 'fa fa-solid fa-user-graduate' => 'User Graduate' ],
		],
		'Emoji' => [
			[ 'fa fa-solid fa-face-angry' => 'Face Angry' ],
			[ 'fa fa-regular fa-face-angry' => 'Face Angry' ],
			[ 'fa fa-solid fa-face-dizzy' => 'Face Dizzy' ],
			[ 'fa fa-regular fa-face-dizzy' => 'Face Dizzy' ],
			[ 'fa fa-solid fa-face-flushed' => 'Face Flushed' ],
			[ 'fa fa-regular fa-face-flushed' => 'Face Flushed' ],
			[ 'fa fa-solid fa-face-frown' => 'Face Frown' ],
			[ 'fa fa-regular fa-face-frown' => 'Face Frown' ],
			[ 'fa fa-solid fa-face-frown-open' => 'Face Frown Open' ],
			[ 'fa fa-regular fa-face-frown-open' => 'Face Frown Open' ],
			[ 'fa fa-solid fa-face-grimace' => 'Face Grimace' ],
			[ 'fa fa-regular fa-face-grimace' => 'Face Grimace' ],
			[ 'fa fa-solid fa-face-grin' => 'Face Grin' ],
			[ 'fa fa-regular fa-face-grin' => 'Face Grin' ],
			[ 'fa fa-solid fa-face-grin-beam' => 'Face Grin Beam' ],
			[ 'fa fa-regular fa-face-grin-beam' => 'Face Grin Beam' ],
			[ 'fa fa-solid fa-face-grin-beam-sweat' => 'Face Grin Beam Sweat' ],
			[ 'fa fa-regular fa-face-grin-beam-sweat' => 'Face Grin Beam Sweat' ],
			[ 'fa fa-solid fa-face-grin-hearts' => 'Face Grin Hearts' ],
			[ 'fa fa-regular fa-face-grin-hearts' => 'Face Grin Hearts' ],
			[ 'fa fa-solid fa-face-grin-squint' => 'Face Grin Squint' ],
			[ 'fa fa-regular fa-face-grin-squint' => 'Face Grin Squint' ],
			[ 'fa fa-solid fa-face-grin-squint-tears' => 'Face Grin Squint Tears' ],
			[ 'fa fa-regular fa-face-grin-squint-tears' => 'Face Grin Squint Tears' ],
			[ 'fa fa-solid fa-face-grin-stars' => 'Face Grin Stars' ],
			[ 'fa fa-regular fa-face-grin-stars' => 'Face Grin Stars' ],
			[ 'fa fa-solid fa-face-grin-tears' => 'Face Grin Tears' ],
			[ 'fa fa-regular fa-face-grin-tears' => 'Face Grin Tears' ],
			[ 'fa fa-solid fa-face-grin-tongue' => 'Face Grin Tongue' ],
			[ 'fa fa-regular fa-face-grin-tongue' => 'Face Grin Tongue' ],
			[ 'fa fa-solid fa-face-grin-tongue-squint' => 'Face Grin Tongue Squint' ],
			[ 'fa fa-regular fa-face-grin-tongue-squint' => 'Face Grin Tongue Squint' ],
			[ 'fa fa-solid fa-face-grin-tongue-wink' => 'Face Grin Tongue Wink' ],
			[ 'fa fa-regular fa-face-grin-tongue-wink' => 'Face Grin Tongue Wink' ],
			[ 'fa fa-solid fa-face-grin-wide' => 'Face Grin Wide' ],
			[ 'fa fa-regular fa-face-grin-wide' => 'Face Grin Wide' ],
			[ 'fa fa-solid fa-face-grin-wink' => 'Face Grin Wink' ],
			[ 'fa fa-regular fa-face-grin-wink' => 'Face Grin Wink' ],
			[ 'fa fa-solid fa-face-kiss' => 'Face Kiss' ],
			[ 'fa fa-regular fa-face-kiss' => 'Face Kiss' ],
			[ 'fa fa-solid fa-face-kiss-beam' => 'Face Kiss Beam' ],
			[ 'fa fa-regular fa-face-kiss-beam' => 'Face Kiss Beam' ],
			[ 'fa fa-solid fa-face-kiss-wink-heart' => 'Face Kiss Wink Heart' ],
			[ 'fa fa-regular fa-face-kiss-wink-heart' => 'Face Kiss Wink Heart' ],
			[ 'fa fa-solid fa-face-laugh' => 'Face Laugh' ],
			[ 'fa fa-regular fa-face-laugh' => 'Face Laugh' ],
			[ 'fa fa-solid fa-face-laugh-beam' => 'Face Laugh Beam' ],
			[ 'fa fa-regular fa-face-laugh-beam' => 'Face Laugh Beam' ],
			[ 'fa fa-solid fa-face-laugh-squint' => 'Face Laugh Squint' ],
			[ 'fa fa-regular fa-face-laugh-squint' => 'Face Laugh Squint' ],
			[ 'fa fa-solid fa-face-laugh-wink' => 'Face Laugh Wink' ],
			[ 'fa fa-regular fa-face-laugh-wink' => 'Face Laugh Wink' ],
			[ 'fa fa-solid fa-face-meh' => 'Face Meh' ],
			[ 'fa fa-regular fa-face-meh' => 'Face Meh' ],
			[ 'fa fa-solid fa-face-meh-blank' => 'Face Meh Blank' ],
			[ 'fa fa-regular fa-face-meh-blank' => 'Face Meh Blank' ],
			[ 'fa fa-solid fa-face-rolling-eyes' => 'Face Rolling Eyes' ],
			[ 'fa fa-regular fa-face-rolling-eyes' => 'Face Rolling Eyes' ],
			[ 'fa fa-solid fa-face-sad-cry' => 'Face Sad Cry' ],
			[ 'fa fa-regular fa-face-sad-cry' => 'Face Sad Cry' ],
			[ 'fa fa-solid fa-face-sad-tear' => 'Face Sad Tear' ],
			[ 'fa fa-regular fa-face-sad-tear' => 'Face Sad Tear' ],
			[ 'fa fa-solid fa-face-smile' => 'Face Smile' ],
			[ 'fa fa-regular fa-face-smile' => 'Face Smile' ],
			[ 'fa fa-solid fa-face-smile-beam' => 'Face Smile Beam' ],
			[ 'fa fa-regular fa-face-smile-beam' => 'Face Smile Beam' ],
			[ 'fa fa-solid fa-face-smile-wink' => 'Face Smile Wink' ],
			[ 'fa fa-regular fa-face-smile-wink' => 'Face Smile Wink' ],
			[ 'fa fa-solid fa-face-surprise' => 'Face Surprise' ],
			[ 'fa fa-regular fa-face-surprise' => 'Face Surprise' ],
			[ 'fa fa-solid fa-face-tired' => 'Face Tired' ],
			[ 'fa fa-regular fa-face-tired' => 'Face Tired' ],
		],
		'Energy' => [
			[ 'fa fa-solid fa-arrow-up-from-ground-water' => 'Arrow Up From Ground Water' ],
			[ 'fa fa-solid fa-atom' => 'Atom' ],
			[ 'fa fa-solid fa-battery-empty' => 'Battery Empty' ],
			[ 'fa fa-solid fa-battery-full' => 'Battery Full' ],
			[ 'fa fa-solid fa-battery-half' => 'Battery Half' ],
			[ 'fa fa-solid fa-battery-quarter' => 'Battery Quarter' ],
			[ 'fa fa-solid fa-battery-three-quarters' => 'Battery Three Quarters' ],
			[ 'fa fa-solid fa-bolt' => 'Bolt' ],
			[ 'fa fa-solid fa-car-battery' => 'Car Battery' ],
			[ 'fa fa-solid fa-charging-station' => 'Charging Station' ],
			[ 'fa fa-solid fa-circle-radiation' => 'Circle Radiation' ],
			[ 'fa fa-solid fa-explosion' => 'Explosion' ],
			[ 'fa fa-solid fa-fan' => 'Fan' ],
			[ 'fa fa-solid fa-fire' => 'Fire' ],
			[ 'fa fa-solid fa-fire-flame-curved' => 'Fire Flame Curved' ],
			[ 'fa fa-solid fa-fire-flame-simple' => 'Fire Flame Simple' ],
			[ 'fa fa-solid fa-gas-pump' => 'Gas Pump' ],
			[ 'fa fa-solid fa-industry' => 'Industry' ],
			[ 'fa fa-solid fa-leaf' => 'Leaf' ],
			[ 'fa fa-solid fa-lightbulb' => 'Lightbulb' ],
			[ 'fa fa-regular fa-lightbulb' => 'Lightbulb' ],
			[ 'fa fa-solid fa-oil-well' => 'Oil Well' ],
			[ 'fa fa-solid fa-plug' => 'Plug' ],
			[ 'fa fa-solid fa-plug-circle-bolt' => 'Plug Circle Bolt' ],
			[ 'fa fa-solid fa-plug-circle-check' => 'Plug Circle Check' ],
			[ 'fa fa-solid fa-plug-circle-exclamation' => 'Plug Circle Exclamation' ],
			[ 'fa fa-solid fa-plug-circle-minus' => 'Plug Circle Minus' ],
			[ 'fa fa-solid fa-plug-circle-plus' => 'Plug Circle Plus' ],
			[ 'fa fa-solid fa-plug-circle-xmark' => 'Plug Circle Xmark' ],
			[ 'fa fa-solid fa-poop' => 'Poop' ],
			[ 'fa fa-solid fa-power-off' => 'Power Off' ],
			[ 'fa fa-solid fa-radiation' => 'Radiation' ],
			[ 'fa fa-solid fa-seedling' => 'Seedling' ],
			[ 'fa fa-solid fa-solar-panel' => 'Solar Panel' ],
			[ 'fa fa-solid fa-sun' => 'Sun' ],
			[ 'fa fa-regular fa-sun' => 'Sun' ],
			[ 'fa fa-solid fa-tower-broadcast' => 'Tower Broadcast' ],
			[ 'fa fa-solid fa-water' => 'Water' ],
			[ 'fa fa-solid fa-wind' => 'Wind' ],
		],
		'Files' => [
			[ 'fa fa-solid fa-box-archive' => 'Box Archive' ],
			[ 'fa fa-solid fa-clone' => 'Clone' ],
			[ 'fa fa-regular fa-clone' => 'Clone' ],
			[ 'fa fa-solid fa-copy' => 'Copy' ],
			[ 'fa fa-regular fa-copy' => 'Copy' ],
			[ 'fa fa-solid fa-file' => 'File' ],
			[ 'fa fa-regular fa-file' => 'File' ],
			[ 'fa fa-solid fa-file-arrow-down' => 'File Arrow Down' ],
			[ 'fa fa-solid fa-file-arrow-up' => 'File Arrow Up' ],
			[ 'fa fa-solid fa-file-audio' => 'File Audio' ],
			[ 'fa fa-regular fa-file-audio' => 'File Audio' ],
			[ 'fa fa-solid fa-file-circle-check' => 'File Circle Check' ],
			[ 'fa fa-solid fa-file-circle-exclamation' => 'File Circle Exclamation' ],
			[ 'fa fa-solid fa-file-circle-minus' => 'File Circle Minus' ],
			[ 'fa fa-solid fa-file-circle-plus' => 'File Circle Plus' ],
			[ 'fa fa-solid fa-file-circle-question' => 'File Circle Question' ],
			[ 'fa fa-solid fa-file-circle-xmark' => 'File Circle Xmark' ],
			[ 'fa fa-solid fa-file-code' => 'File Code' ],
			[ 'fa fa-regular fa-file-code' => 'File Code' ],
			[ 'fa fa-solid fa-file-csv' => 'File Csv' ],
			[ 'fa fa-solid fa-file-excel' => 'File Excel' ],
			[ 'fa fa-regular fa-file-excel' => 'File Excel' ],
			[ 'fa fa-solid fa-file-export' => 'File Export' ],
			[ 'fa fa-solid fa-file-fragment' => 'File Fragment' ],
			[ 'fa fa-solid fa-file-half-dashed' => 'File Half Dashed' ],
			[ 'fa fa-solid fa-file-image' => 'File Image' ],
			[ 'fa fa-regular fa-file-image' => 'File Image' ],
			[ 'fa fa-solid fa-file-import' => 'File Import' ],
			[ 'fa fa-solid fa-file-lines' => 'File Lines' ],
			[ 'fa fa-regular fa-file-lines' => 'File Lines' ],
			[ 'fa fa-solid fa-file-pdf' => 'File Pdf' ],
			[ 'fa fa-regular fa-file-pdf' => 'File Pdf' ],
			[ 'fa fa-solid fa-file-pen' => 'File Pen' ],
			[ 'fa fa-solid fa-file-powerpoint' => 'File Powerpoint' ],
			[ 'fa fa-regular fa-file-powerpoint' => 'File Powerpoint' ],
			[ 'fa fa-solid fa-file-shield' => 'File Shield' ],
			[ 'fa fa-solid fa-file-video' => 'File Video' ],
			[ 'fa fa-regular fa-file-video' => 'File Video' ],
			[ 'fa fa-solid fa-file-word' => 'File Word' ],
			[ 'fa fa-regular fa-file-word' => 'File Word' ],
			[ 'fa fa-solid fa-file-zipper' => 'File Zipper' ],
			[ 'fa fa-regular fa-file-zipper' => 'File Zipper' ],
			[ 'fa fa-solid fa-floppy-disk' => 'Floppy Disk' ],
			[ 'fa fa-regular fa-floppy-disk' => 'Floppy Disk' ],
			[ 'fa fa-solid fa-folder' => 'Folder' ],
			[ 'fa fa-regular fa-folder' => 'Folder' ],
			[ 'fa fa-solid fa-folder-closed' => 'Folder Closed' ],
			[ 'fa fa-regular fa-folder-closed' => 'Folder Closed' ],
			[ 'fa fa-solid fa-folder-open' => 'Folder Open' ],
			[ 'fa fa-regular fa-folder-open' => 'Folder Open' ],
			[ 'fa fa-solid fa-note-sticky' => 'Note Sticky' ],
			[ 'fa fa-regular fa-note-sticky' => 'Note Sticky' ],
			[ 'fa fa-solid fa-paste' => 'Paste' ],
			[ 'fa fa-regular fa-paste' => 'Paste' ],
			[ 'fa fa-solid fa-photo-film' => 'Photo Film' ],
			[ 'fa fa-solid fa-scissors' => 'Scissors' ],
		],
		'Film-video' => [
			[ 'fa fa-solid fa-audio-description' => 'Audio Description' ],
			[ 'fa fa-solid fa-circle' => 'Circle' ],
			[ 'fa fa-regular fa-circle' => 'Circle' ],
			[ 'fa fa-solid fa-clapperboard' => 'Clapperboard' ],
			[ 'fa fa-solid fa-closed-captioning' => 'Closed Captioning' ],
			[ 'fa fa-regular fa-closed-captioning' => 'Closed Captioning' ],
			[ 'fa fa-solid fa-compact-disc' => 'Compact Disc' ],
			[ 'fa fa-solid fa-file-audio' => 'File Audio' ],
			[ 'fa fa-regular fa-file-audio' => 'File Audio' ],
			[ 'fa fa-solid fa-file-video' => 'File Video' ],
			[ 'fa fa-regular fa-file-video' => 'File Video' ],
			[ 'fa fa-solid fa-film' => 'Film' ],
			[ 'fa fa-solid fa-headphones' => 'Headphones' ],
			[ 'fa fa-solid fa-microphone' => 'Microphone' ],
			[ 'fa fa-solid fa-microphone-lines' => 'Microphone Lines' ],
			[ 'fa fa-solid fa-microphone-lines-slash' => 'Microphone Lines Slash' ],
			[ 'fa fa-solid fa-microphone-slash' => 'Microphone Slash' ],
			[ 'fa fa-solid fa-photo-film' => 'Photo Film' ],
			[ 'fa fa-solid fa-podcast' => 'Podcast' ],
			[ 'fa fa-solid fa-square-rss' => 'Square Rss' ],
			[ 'fa fa-solid fa-ticket' => 'Ticket' ],
			[ 'fa fa-solid fa-tower-broadcast' => 'Tower Broadcast' ],
			[ 'fa fa-solid fa-tower-cell' => 'Tower Cell' ],
			[ 'fa fa-solid fa-tv' => 'Tv' ],
			[ 'fa fa-solid fa-video' => 'Video' ],
			[ 'fa fa-solid fa-video-slash' => 'Video Slash' ],
			[ 'fa fa-brands fa-youtube' => 'Youtube' ],
		],
		'Food-beverage' => [
			[ 'fa fa-solid fa-apple-whole' => 'Apple Whole' ],
			[ 'fa fa-solid fa-bacon' => 'Bacon' ],
			[ 'fa fa-solid fa-beer-mug-empty' => 'Beer Mug Empty' ],
			[ 'fa fa-solid fa-blender' => 'Blender' ],
			[ 'fa fa-solid fa-bone' => 'Bone' ],
			[ 'fa fa-solid fa-bottle-droplet' => 'Bottle Droplet' ],
			[ 'fa fa-solid fa-bottle-water' => 'Bottle Water' ],
			[ 'fa fa-solid fa-bowl-food' => 'Bowl Food' ],
			[ 'fa fa-solid fa-bowl-rice' => 'Bowl Rice' ],
			[ 'fa fa-solid fa-bread-slice' => 'Bread Slice' ],
			[ 'fa fa-solid fa-burger' => 'Burger' ],
			[ 'fa fa-solid fa-cake-candles' => 'Cake Candles' ],
			[ 'fa fa-solid fa-candy-cane' => 'Candy Cane' ],
			[ 'fa fa-solid fa-carrot' => 'Carrot' ],
			[ 'fa fa-solid fa-champagne-glasses' => 'Champagne Glasses' ],
			[ 'fa fa-solid fa-cheese' => 'Cheese' ],
			[ 'fa fa-solid fa-cloud-meatball' => 'Cloud Meatball' ],
			[ 'fa fa-solid fa-cookie' => 'Cookie' ],
			[ 'fa fa-solid fa-cubes-stacked' => 'Cubes Stacked' ],
			[ 'fa fa-solid fa-drumstick-bite' => 'Drumstick Bite' ],
			[ 'fa fa-solid fa-egg' => 'Egg' ],
			[ 'fa fa-solid fa-fish' => 'Fish' ],
			[ 'fa fa-solid fa-fish-fins' => 'Fish Fins' ],
			[ 'fa fa-solid fa-flask' => 'Flask' ],
			[ 'fa fa-solid fa-glass-water' => 'Glass Water' ],
			[ 'fa fa-solid fa-glass-water-droplet' => 'Glass Water Droplet' ],
			[ 'fa fa-solid fa-hotdog' => 'Hotdog' ],
			[ 'fa fa-solid fa-ice-cream' => 'Ice Cream' ],
			[ 'fa fa-solid fa-jar' => 'Jar' ],
			[ 'fa fa-solid fa-jar-wheat' => 'Jar Wheat' ],
			[ 'fa fa-solid fa-lemon' => 'Lemon' ],
			[ 'fa fa-regular fa-lemon' => 'Lemon' ],
			[ 'fa fa-solid fa-martini-glass' => 'Martini Glass' ],
			[ 'fa fa-solid fa-martini-glass-citrus' => 'Martini Glass Citrus' ],
			[ 'fa fa-solid fa-martini-glass-empty' => 'Martini Glass Empty' ],
			[ 'fa fa-solid fa-mug-hot' => 'Mug Hot' ],
			[ 'fa fa-solid fa-mug-saucer' => 'Mug Saucer' ],
			[ 'fa fa-solid fa-pepper-hot' => 'Pepper Hot' ],
			[ 'fa fa-solid fa-pizza-slice' => 'Pizza Slice' ],
			[ 'fa fa-solid fa-plate-wheat' => 'Plate Wheat' ],
			[ 'fa fa-solid fa-seedling' => 'Seedling' ],
			[ 'fa fa-solid fa-shrimp' => 'Shrimp' ],
			[ 'fa fa-solid fa-stroopwafel' => 'Stroopwafel' ],
			[ 'fa fa-solid fa-wheat-awn' => 'Wheat Awn' ],
			[ 'fa fa-solid fa-wheat-awn-circle-exclamation' => 'Wheat Awn Circle Exclamation' ],
			[ 'fa fa-solid fa-whiskey-glass' => 'Whiskey Glass' ],
			[ 'fa fa-solid fa-wine-bottle' => 'Wine Bottle' ],
			[ 'fa fa-solid fa-wine-glass' => 'Wine Glass' ],
			[ 'fa fa-solid fa-wine-glass-empty' => 'Wine Glass Empty' ],
		],
		'Fruits-vegetables' => [
			[ 'fa fa-solid fa-apple-whole' => 'Apple Whole' ],
			[ 'fa fa-solid fa-carrot' => 'Carrot' ],
			[ 'fa fa-solid fa-leaf' => 'Leaf' ],
			[ 'fa fa-solid fa-lemon' => 'Lemon' ],
			[ 'fa fa-regular fa-lemon' => 'Lemon' ],
			[ 'fa fa-solid fa-pepper-hot' => 'Pepper Hot' ],
			[ 'fa fa-solid fa-seedling' => 'Seedling' ],
		],
		'Gaming' => [
			[ 'fa fa-solid fa-book-skull' => 'Book Skull' ],
			[ 'fa fa-solid fa-chess' => 'Chess' ],
			[ 'fa fa-solid fa-chess-bishop' => 'Chess Bishop' ],
			[ 'fa fa-regular fa-chess-bishop' => 'Chess Bishop' ],
			[ 'fa fa-solid fa-chess-board' => 'Chess Board' ],
			[ 'fa fa-solid fa-chess-king' => 'Chess King' ],
			[ 'fa fa-regular fa-chess-king' => 'Chess King' ],
			[ 'fa fa-solid fa-chess-knight' => 'Chess Knight' ],
			[ 'fa fa-regular fa-chess-knight' => 'Chess Knight' ],
			[ 'fa fa-solid fa-chess-pawn' => 'Chess Pawn' ],
			[ 'fa fa-regular fa-chess-pawn' => 'Chess Pawn' ],
			[ 'fa fa-solid fa-chess-queen' => 'Chess Queen' ],
			[ 'fa fa-regular fa-chess-queen' => 'Chess Queen' ],
			[ 'fa fa-solid fa-chess-rook' => 'Chess Rook' ],
			[ 'fa fa-regular fa-chess-rook' => 'Chess Rook' ],
			[ 'fa fa-brands fa-critical-role' => 'Critical Role' ],
			[ 'fa fa-brands fa-d-and-d' => 'D And D' ],
			[ 'fa fa-brands fa-d-and-d-beyond' => 'D And D Beyond' ],
			[ 'fa fa-solid fa-diamond' => 'Diamond' ],
			[ 'fa fa-solid fa-dice' => 'Dice' ],
			[ 'fa fa-solid fa-dice-d20' => 'Dice D20' ],
			[ 'fa fa-solid fa-dice-d6' => 'Dice D6' ],
			[ 'fa fa-solid fa-dice-five' => 'Dice Five' ],
			[ 'fa fa-solid fa-dice-four' => 'Dice Four' ],
			[ 'fa fa-solid fa-dice-one' => 'Dice One' ],
			[ 'fa fa-solid fa-dice-six' => 'Dice Six' ],
			[ 'fa fa-solid fa-dice-three' => 'Dice Three' ],
			[ 'fa fa-solid fa-dice-two' => 'Dice Two' ],
			[ 'fa fa-solid fa-dragon' => 'Dragon' ],
			[ 'fa fa-solid fa-dungeon' => 'Dungeon' ],
			[ 'fa fa-brands fa-fantasy-flight-games' => 'Fantasy Flight Games' ],
			[ 'fa fa-solid fa-gamepad' => 'Gamepad' ],
			[ 'fa fa-solid fa-ghost' => 'Ghost' ],
			[ 'fa fa-solid fa-hand-fist' => 'Hand Fist' ],
			[ 'fa fa-solid fa-hat-wizard' => 'Hat Wizard' ],
			[ 'fa fa-solid fa-headset' => 'Headset' ],
			[ 'fa fa-solid fa-heart' => 'Heart' ],
			[ 'fa fa-regular fa-heart' => 'Heart' ],
			[ 'fa fa-brands fa-playstation' => 'Playstation' ],
			[ 'fa fa-solid fa-puzzle-piece' => 'Puzzle Piece' ],
			[ 'fa fa-solid fa-ring' => 'Ring' ],
			[ 'fa fa-solid fa-scroll' => 'Scroll' ],
			[ 'fa fa-solid fa-shield-halved' => 'Shield Halved' ],
			[ 'fa fa-solid fa-skull-crossbones' => 'Skull Crossbones' ],
			[ 'fa fa-solid fa-square-full' => 'Square Full' ],
			[ 'fa fa-regular fa-square-full' => 'Square Full' ],
			[ 'fa fa-brands fa-square-steam' => 'Square Steam' ],
			[ 'fa fa-brands fa-steam' => 'Steam' ],
			[ 'fa fa-brands fa-steam-symbol' => 'Steam Symbol' ],
			[ 'fa fa-brands fa-twitch' => 'Twitch' ],
			[ 'fa fa-solid fa-vr-cardboard' => 'Vr Cardboard' ],
			[ 'fa fa-solid fa-wand-sparkles' => 'Wand Sparkles' ],
			[ 'fa fa-brands fa-wizards-of-the-coast' => 'Wizards Of The Coast' ],
			[ 'fa fa-brands fa-xbox' => 'Xbox' ],
		],
		'Gender' => [
			[ 'fa fa-solid fa-genderless' => 'Genderless' ],
			[ 'fa fa-solid fa-mars' => 'Mars' ],
			[ 'fa fa-solid fa-mars-and-venus' => 'Mars And Venus' ],
			[ 'fa fa-solid fa-mars-double' => 'Mars Double' ],
			[ 'fa fa-solid fa-mars-stroke' => 'Mars Stroke' ],
			[ 'fa fa-solid fa-mars-stroke-right' => 'Mars Stroke Right' ],
			[ 'fa fa-solid fa-mars-stroke-up' => 'Mars Stroke Up' ],
			[ 'fa fa-solid fa-mercury' => 'Mercury' ],
			[ 'fa fa-solid fa-neuter' => 'Neuter' ],
			[ 'fa fa-solid fa-person-half-dress' => 'Person Half Dress' ],
			[ 'fa fa-solid fa-transgender' => 'Transgender' ],
			[ 'fa fa-solid fa-venus' => 'Venus' ],
			[ 'fa fa-solid fa-venus-double' => 'Venus Double' ],
			[ 'fa fa-solid fa-venus-mars' => 'Venus Mars' ],
		],
		'Halloween' => [
			[ 'fa fa-solid fa-book-skull' => 'Book Skull' ],
			[ 'fa fa-solid fa-broom' => 'Broom' ],
			[ 'fa fa-solid fa-cat' => 'Cat' ],
			[ 'fa fa-solid fa-cloud-moon' => 'Cloud Moon' ],
			[ 'fa fa-solid fa-crow' => 'Crow' ],
			[ 'fa fa-solid fa-ghost' => 'Ghost' ],
			[ 'fa fa-solid fa-hat-wizard' => 'Hat Wizard' ],
			[ 'fa fa-solid fa-mask' => 'Mask' ],
			[ 'fa fa-solid fa-skull' => 'Skull' ],
			[ 'fa fa-solid fa-skull-crossbones' => 'Skull Crossbones' ],
			[ 'fa fa-solid fa-spider' => 'Spider' ],
			[ 'fa fa-solid fa-toilet-paper' => 'Toilet Paper' ],
			[ 'fa fa-solid fa-wand-sparkles' => 'Wand Sparkles' ],
		],
		'Hands' => [
			[ 'fa fa-solid fa-hand' => 'Hand' ],
			[ 'fa fa-regular fa-hand' => 'Hand' ],
			[ 'fa fa-solid fa-hand-back-fist' => 'Hand Back Fist' ],
			[ 'fa fa-regular fa-hand-back-fist' => 'Hand Back Fist' ],
			[ 'fa fa-solid fa-hand-dots' => 'Hand Dots' ],
			[ 'fa fa-solid fa-hand-fist' => 'Hand Fist' ],
			[ 'fa fa-solid fa-hand-holding' => 'Hand Holding' ],
			[ 'fa fa-solid fa-hand-holding-dollar' => 'Hand Holding Dollar' ],
			[ 'fa fa-solid fa-hand-holding-droplet' => 'Hand Holding Droplet' ],
			[ 'fa fa-solid fa-hand-holding-hand' => 'Hand Holding Hand' ],
			[ 'fa fa-solid fa-hand-holding-heart' => 'Hand Holding Heart' ],
			[ 'fa fa-solid fa-hand-holding-medical' => 'Hand Holding Medical' ],
			[ 'fa fa-solid fa-hand-lizard' => 'Hand Lizard' ],
			[ 'fa fa-regular fa-hand-lizard' => 'Hand Lizard' ],
			[ 'fa fa-solid fa-hand-middle-finger' => 'Hand Middle Finger' ],
			[ 'fa fa-solid fa-hand-peace' => 'Hand Peace' ],
			[ 'fa fa-regular fa-hand-peace' => 'Hand Peace' ],
			[ 'fa fa-solid fa-hand-point-down' => 'Hand Point Down' ],
			[ 'fa fa-regular fa-hand-point-down' => 'Hand Point Down' ],
			[ 'fa fa-solid fa-hand-point-left' => 'Hand Point Left' ],
			[ 'fa fa-regular fa-hand-point-left' => 'Hand Point Left' ],
			[ 'fa fa-solid fa-hand-point-right' => 'Hand Point Right' ],
			[ 'fa fa-regular fa-hand-point-right' => 'Hand Point Right' ],
			[ 'fa fa-solid fa-hand-point-up' => 'Hand Point Up' ],
			[ 'fa fa-regular fa-hand-point-up' => 'Hand Point Up' ],
			[ 'fa fa-solid fa-hand-pointer' => 'Hand Pointer' ],
			[ 'fa fa-regular fa-hand-pointer' => 'Hand Pointer' ],
			[ 'fa fa-solid fa-hand-scissors' => 'Hand Scissors' ],
			[ 'fa fa-regular fa-hand-scissors' => 'Hand Scissors' ],
			[ 'fa fa-solid fa-hand-sparkles' => 'Hand Sparkles' ],
			[ 'fa fa-solid fa-hand-spock' => 'Hand Spock' ],
			[ 'fa fa-regular fa-hand-spock' => 'Hand Spock' ],
			[ 'fa fa-solid fa-hands-bound' => 'Hands Bound' ],
			[ 'fa fa-solid fa-hands-bubbles' => 'Hands Bubbles' ],
			[ 'fa fa-solid fa-hands-clapping' => 'Hands Clapping' ],
			[ 'fa fa-solid fa-hands-holding' => 'Hands Holding' ],
			[ 'fa fa-solid fa-hands-holding-child' => 'Hands Holding Child' ],
			[ 'fa fa-solid fa-hands-holding-circle' => 'Hands Holding Circle' ],
			[ 'fa fa-solid fa-hands-praying' => 'Hands Praying' ],
			[ 'fa fa-solid fa-handshake' => 'Handshake' ],
			[ 'fa fa-regular fa-handshake' => 'Handshake' ],
			[ 'fa fa-solid fa-handshake-angle' => 'Handshake Angle' ],
			[ 'fa fa-solid fa-handshake-simple' => 'Handshake Simple' ],
			[ 'fa fa-solid fa-handshake-simple-slash' => 'Handshake Simple Slash' ],
			[ 'fa fa-solid fa-handshake-slash' => 'Handshake Slash' ],
			[ 'fa fa-solid fa-thumbs-down' => 'Thumbs Down' ],
			[ 'fa fa-regular fa-thumbs-down' => 'Thumbs Down' ],
			[ 'fa fa-solid fa-thumbs-up' => 'Thumbs Up' ],
			[ 'fa fa-regular fa-thumbs-up' => 'Thumbs Up' ],
		],
		'Holidays' => [
			[ 'fa fa-solid fa-candy-cane' => 'Candy Cane' ],
			[ 'fa fa-solid fa-carrot' => 'Carrot' ],
			[ 'fa fa-solid fa-champagne-glasses' => 'Champagne Glasses' ],
			[ 'fa fa-solid fa-cookie-bite' => 'Cookie Bite' ],
			[ 'fa fa-solid fa-face-grin-hearts' => 'Face Grin Hearts' ],
			[ 'fa fa-regular fa-face-grin-hearts' => 'Face Grin Hearts' ],
			[ 'fa fa-solid fa-face-kiss-wink-heart' => 'Face Kiss Wink Heart' ],
			[ 'fa fa-regular fa-face-kiss-wink-heart' => 'Face Kiss Wink Heart' ],
			[ 'fa fa-solid fa-gift' => 'Gift' ],
			[ 'fa fa-solid fa-gifts' => 'Gifts' ],
			[ 'fa fa-solid fa-heart' => 'Heart' ],
			[ 'fa fa-regular fa-heart' => 'Heart' ],
			[ 'fa fa-solid fa-holly-berry' => 'Holly Berry' ],
			[ 'fa fa-solid fa-menorah' => 'Menorah' ],
			[ 'fa fa-solid fa-mug-hot' => 'Mug Hot' ],
			[ 'fa fa-solid fa-sleigh' => 'Sleigh' ],
			[ 'fa fa-solid fa-snowman' => 'Snowman' ],
		],
		'Household' => [
			[ 'fa fa-solid fa-arrow-up-from-water-pump' => 'Arrow Up From Water Pump' ],
			[ 'fa fa-solid fa-bath' => 'Bath' ],
			[ 'fa fa-solid fa-bed' => 'Bed' ],
			[ 'fa fa-solid fa-bell' => 'Bell' ],
			[ 'fa fa-regular fa-bell' => 'Bell' ],
			[ 'fa fa-solid fa-blender' => 'Blender' ],
			[ 'fa fa-solid fa-box-tissue' => 'Box Tissue' ],
			[ 'fa fa-solid fa-chair' => 'Chair' ],
			[ 'fa fa-solid fa-computer' => 'Computer' ],
			[ 'fa fa-solid fa-couch' => 'Couch' ],
			[ 'fa fa-solid fa-door-closed' => 'Door Closed' ],
			[ 'fa fa-solid fa-door-open' => 'Door Open' ],
			[ 'fa fa-solid fa-dungeon' => 'Dungeon' ],
			[ 'fa fa-solid fa-fan' => 'Fan' ],
			[ 'fa fa-solid fa-faucet' => 'Faucet' ],
			[ 'fa fa-solid fa-faucet-drip' => 'Faucet Drip' ],
			[ 'fa fa-solid fa-fire-burner' => 'Fire Burner' ],
			[ 'fa fa-solid fa-house-chimney-user' => 'House Chimney User' ],
			[ 'fa fa-solid fa-house-chimney-window' => 'House Chimney Window' ],
			[ 'fa fa-solid fa-house-fire' => 'House Fire' ],
			[ 'fa fa-solid fa-house-laptop' => 'House Laptop' ],
			[ 'fa fa-solid fa-house-lock' => 'House Lock' ],
			[ 'fa fa-solid fa-house-signal' => 'House Signal' ],
			[ 'fa fa-solid fa-house-user' => 'House User' ],
			[ 'fa fa-solid fa-jar' => 'Jar' ],
			[ 'fa fa-solid fa-jar-wheat' => 'Jar Wheat' ],
			[ 'fa fa-solid fa-jug-detergent' => 'Jug Detergent' ],
			[ 'fa fa-solid fa-kitchen-set' => 'Kitchen Set' ],
			[ 'fa fa-solid fa-lightbulb' => 'Lightbulb' ],
			[ 'fa fa-regular fa-lightbulb' => 'Lightbulb' ],
			[ 'fa fa-solid fa-mattress-pillow' => 'Mattress Pillow' ],
			[ 'fa fa-solid fa-mug-saucer' => 'Mug Saucer' ],
			[ 'fa fa-solid fa-people-roof' => 'People Roof' ],
			[ 'fa fa-solid fa-plug' => 'Plug' ],
			[ 'fa fa-solid fa-pump-soap' => 'Pump Soap' ],
			[ 'fa fa-solid fa-rug' => 'Rug' ],
			[ 'fa fa-solid fa-sheet-plastic' => 'Sheet Plastic' ],
			[ 'fa fa-solid fa-shower' => 'Shower' ],
			[ 'fa fa-solid fa-sink' => 'Sink' ],
			[ 'fa fa-solid fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-regular fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-solid fa-soap' => 'Soap' ],
			[ 'fa fa-solid fa-spoon' => 'Spoon' ],
			[ 'fa fa-solid fa-stairs' => 'Stairs' ],
			[ 'fa fa-solid fa-temperature-arrow-down' => 'Temperature Arrow Down' ],
			[ 'fa fa-solid fa-temperature-arrow-up' => 'Temperature Arrow Up' ],
			[ 'fa fa-solid fa-toilet' => 'Toilet' ],
			[ 'fa fa-solid fa-toilet-paper' => 'Toilet Paper' ],
			[ 'fa fa-solid fa-toilet-paper-slash' => 'Toilet Paper Slash' ],
			[ 'fa fa-solid fa-tv' => 'Tv' ],
			[ 'fa fa-solid fa-utensils' => 'Utensils' ],
		],
		'Humanitarian' => [
			[ 'fa fa-solid fa-anchor' => 'Anchor' ],
			[ 'fa fa-solid fa-anchor-circle-check' => 'Anchor Circle Check' ],
			[ 'fa fa-solid fa-anchor-circle-exclamation' => 'Anchor Circle Exclamation' ],
			[ 'fa fa-solid fa-anchor-circle-xmark' => 'Anchor Circle Xmark' ],
			[ 'fa fa-solid fa-anchor-lock' => 'Anchor Lock' ],
			[ 'fa fa-solid fa-arrow-down-up-across-line' => 'Arrow Down Up Across Line' ],
			[ 'fa fa-solid fa-arrow-down-up-lock' => 'Arrow Down Up Lock' ],
			[ 'fa fa-solid fa-arrow-right-to-city' => 'Arrow Right To City' ],
			[ 'fa fa-solid fa-arrow-up-from-ground-water' => 'Arrow Up From Ground Water' ],
			[ 'fa fa-solid fa-arrow-up-from-water-pump' => 'Arrow Up From Water Pump' ],
			[ 'fa fa-solid fa-arrow-up-right-dots' => 'Arrow Up Right Dots' ],
			[ 'fa fa-solid fa-arrow-up-right-from-square' => 'Arrow Up Right From Square' ],
			[ 'fa fa-solid fa-arrows-down-to-line' => 'Arrows Down To Line' ],
			[ 'fa fa-solid fa-arrows-down-to-people' => 'Arrows Down To People' ],
			[ 'fa fa-solid fa-arrows-left-right-to-line' => 'Arrows Left Right To Line' ],
			[ 'fa fa-solid fa-arrows-spin' => 'Arrows Spin' ],
			[ 'fa fa-solid fa-arrows-split-up-and-left' => 'Arrows Split Up And Left' ],
			[ 'fa fa-solid fa-arrows-to-circle' => 'Arrows To Circle' ],
			[ 'fa fa-solid fa-arrows-to-dot' => 'Arrows To Dot' ],
			[ 'fa fa-solid fa-arrows-to-eye' => 'Arrows To Eye' ],
			[ 'fa fa-solid fa-arrows-turn-right' => 'Arrows Turn Right' ],
			[ 'fa fa-solid fa-arrows-turn-to-dots' => 'Arrows Turn To Dots' ],
			[ 'fa fa-solid fa-arrows-up-to-line' => 'Arrows Up To Line' ],
			[ 'fa fa-solid fa-baby' => 'Baby' ],
			[ 'fa fa-solid fa-bacterium' => 'Bacterium' ],
			[ 'fa fa-solid fa-ban' => 'Ban' ],
			[ 'fa fa-solid fa-bed' => 'Bed' ],
			[ 'fa fa-solid fa-biohazard' => 'Biohazard' ],
			[ 'fa fa-solid fa-book-bookmark' => 'Book Bookmark' ],
			[ 'fa fa-solid fa-bore-hole' => 'Bore Hole' ],
			[ 'fa fa-solid fa-bottle-droplet' => 'Bottle Droplet' ],
			[ 'fa fa-solid fa-bottle-water' => 'Bottle Water' ],
			[ 'fa fa-solid fa-bowl-food' => 'Bowl Food' ],
			[ 'fa fa-solid fa-bowl-rice' => 'Bowl Rice' ],
			[ 'fa fa-solid fa-boxes-packing' => 'Boxes Packing' ],
			[ 'fa fa-solid fa-bridge' => 'Bridge' ],
			[ 'fa fa-solid fa-bridge-circle-check' => 'Bridge Circle Check' ],
			[ 'fa fa-solid fa-bridge-circle-exclamation' => 'Bridge Circle Exclamation' ],
			[ 'fa fa-solid fa-bridge-circle-xmark' => 'Bridge Circle Xmark' ],
			[ 'fa fa-solid fa-bridge-lock' => 'Bridge Lock' ],
			[ 'fa fa-solid fa-bridge-water' => 'Bridge Water' ],
			[ 'fa fa-solid fa-bucket' => 'Bucket' ],
			[ 'fa fa-solid fa-bugs' => 'Bugs' ],
			[ 'fa fa-solid fa-building' => 'Building' ],
			[ 'fa fa-regular fa-building' => 'Building' ],
			[ 'fa fa-solid fa-building-circle-arrow-right' => 'Building Circle Arrow Right' ],
			[ 'fa fa-solid fa-building-circle-check' => 'Building Circle Check' ],
			[ 'fa fa-solid fa-building-circle-exclamation' => 'Building Circle Exclamation' ],
			[ 'fa fa-solid fa-building-circle-xmark' => 'Building Circle Xmark' ],
			[ 'fa fa-solid fa-building-columns' => 'Building Columns' ],
			[ 'fa fa-solid fa-building-flag' => 'Building Flag' ],
			[ 'fa fa-solid fa-building-lock' => 'Building Lock' ],
			[ 'fa fa-solid fa-building-ngo' => 'Building Ngo' ],
			[ 'fa fa-solid fa-building-shield' => 'Building Shield' ],
			[ 'fa fa-solid fa-building-un' => 'Building Un' ],
			[ 'fa fa-solid fa-building-user' => 'Building User' ],
			[ 'fa fa-solid fa-building-wheat' => 'Building Wheat' ],
			[ 'fa fa-solid fa-burst' => 'Burst' ],
			[ 'fa fa-solid fa-bus' => 'Bus' ],
			[ 'fa fa-solid fa-car' => 'Car' ],
			[ 'fa fa-solid fa-car-on' => 'Car On' ],
			[ 'fa fa-solid fa-car-tunnel' => 'Car Tunnel' ],
			[ 'fa fa-solid fa-child-combatant' => 'Child Combatant' ],
			[ 'fa fa-solid fa-children' => 'Children' ],
			[ 'fa fa-solid fa-church' => 'Church' ],
			[ 'fa fa-solid fa-circle-h' => 'Circle H' ],
			[ 'fa fa-solid fa-circle-nodes' => 'Circle Nodes' ],
			[ 'fa fa-solid fa-clipboard-question' => 'Clipboard Question' ],
			[ 'fa fa-solid fa-clipboard-user' => 'Clipboard User' ],
			[ 'fa fa-solid fa-cloud-bolt' => 'Cloud Bolt' ],
			[ 'fa fa-solid fa-cloud-showers-heavy' => 'Cloud Showers Heavy' ],
			[ 'fa fa-solid fa-cloud-showers-water' => 'Cloud Showers Water' ],
			[ 'fa fa-solid fa-computer' => 'Computer' ],
			[ 'fa fa-solid fa-cow' => 'Cow' ],
			[ 'fa fa-solid fa-cubes-stacked' => 'Cubes Stacked' ],
			[ 'fa fa-solid fa-display' => 'Display' ],
			[ 'fa fa-solid fa-droplet' => 'Droplet' ],
			[ 'fa fa-solid fa-envelope' => 'Envelope' ],
			[ 'fa fa-regular fa-envelope' => 'Envelope' ],
			[ 'fa fa-solid fa-envelope-circle-check' => 'Envelope Circle Check' ],
			[ 'fa fa-solid fa-explosion' => 'Explosion' ],
			[ 'fa fa-solid fa-faucet-drip' => 'Faucet Drip' ],
			[ 'fa fa-solid fa-fax' => 'Fax' ],
			[ 'fa fa-solid fa-ferry' => 'Ferry' ],
			[ 'fa fa-solid fa-file' => 'File' ],
			[ 'fa fa-regular fa-file' => 'File' ],
			[ 'fa fa-solid fa-file-circle-check' => 'File Circle Check' ],
			[ 'fa fa-solid fa-file-circle-exclamation' => 'File Circle Exclamation' ],
			[ 'fa fa-solid fa-file-circle-minus' => 'File Circle Minus' ],
			[ 'fa fa-solid fa-file-circle-plus' => 'File Circle Plus' ],
			[ 'fa fa-solid fa-file-circle-question' => 'File Circle Question' ],
			[ 'fa fa-solid fa-file-circle-xmark' => 'File Circle Xmark' ],
			[ 'fa fa-solid fa-file-csv' => 'File Csv' ],
			[ 'fa fa-solid fa-file-pdf' => 'File Pdf' ],
			[ 'fa fa-regular fa-file-pdf' => 'File Pdf' ],
			[ 'fa fa-solid fa-file-pen' => 'File Pen' ],
			[ 'fa fa-solid fa-file-shield' => 'File Shield' ],
			[ 'fa fa-solid fa-fire-burner' => 'Fire Burner' ],
			[ 'fa fa-solid fa-fire-flame-simple' => 'Fire Flame Simple' ],
			[ 'fa fa-solid fa-fish-fins' => 'Fish Fins' ],
			[ 'fa fa-solid fa-flag' => 'Flag' ],
			[ 'fa fa-regular fa-flag' => 'Flag' ],
			[ 'fa fa-solid fa-flask-vial' => 'Flask Vial' ],
			[ 'fa fa-solid fa-gas-pump' => 'Gas Pump' ],
			[ 'fa fa-solid fa-glass-water' => 'Glass Water' ],
			[ 'fa fa-solid fa-glass-water-droplet' => 'Glass Water Droplet' ],
			[ 'fa fa-solid fa-gopuram' => 'Gopuram' ],
			[ 'fa fa-solid fa-group-arrows-rotate' => 'Group Arrows Rotate' ],
			[ 'fa fa-solid fa-hammer' => 'Hammer' ],
			[ 'fa fa-solid fa-hand-holding-hand' => 'Hand Holding Hand' ],
			[ 'fa fa-solid fa-handcuffs' => 'Handcuffs' ],
			[ 'fa fa-solid fa-hands-bound' => 'Hands Bound' ],
			[ 'fa fa-solid fa-hands-bubbles' => 'Hands Bubbles' ],
			[ 'fa fa-solid fa-hands-holding-child' => 'Hands Holding Child' ],
			[ 'fa fa-solid fa-hands-holding-circle' => 'Hands Holding Circle' ],
			[ 'fa fa-solid fa-handshake-simple' => 'Handshake Simple' ],
			[ 'fa fa-solid fa-headset' => 'Headset' ],
			[ 'fa fa-solid fa-heart-circle-bolt' => 'Heart Circle Bolt' ],
			[ 'fa fa-solid fa-heart-circle-check' => 'Heart Circle Check' ],
			[ 'fa fa-solid fa-heart-circle-exclamation' => 'Heart Circle Exclamation' ],
			[ 'fa fa-solid fa-heart-circle-minus' => 'Heart Circle Minus' ],
			[ 'fa fa-solid fa-heart-circle-plus' => 'Heart Circle Plus' ],
			[ 'fa fa-solid fa-heart-circle-xmark' => 'Heart Circle Xmark' ],
			[ 'fa fa-solid fa-helicopter' => 'Helicopter' ],
			[ 'fa fa-solid fa-helicopter-symbol' => 'Helicopter Symbol' ],
			[ 'fa fa-solid fa-helmet-un' => 'Helmet Un' ],
			[ 'fa fa-solid fa-hill-avalanche' => 'Hill Avalanche' ],
			[ 'fa fa-solid fa-hill-rockslide' => 'Hill Rockslide' ],
			[ 'fa fa-solid fa-hospital' => 'Hospital' ],
			[ 'fa fa-regular fa-hospital' => 'Hospital' ],
			[ 'fa fa-solid fa-hotel' => 'Hotel' ],
			[ 'fa fa-solid fa-house-chimney' => 'House Chimney' ],
			[ 'fa fa-solid fa-house-chimney-crack' => 'House Chimney Crack' ],
			[ 'fa fa-solid fa-house-circle-check' => 'House Circle Check' ],
			[ 'fa fa-solid fa-house-circle-exclamation' => 'House Circle Exclamation' ],
			[ 'fa fa-solid fa-house-circle-xmark' => 'House Circle Xmark' ],
			[ 'fa fa-solid fa-house-fire' => 'House Fire' ],
			[ 'fa fa-solid fa-house-flag' => 'House Flag' ],
			[ 'fa fa-solid fa-house-flood-water' => 'House Flood Water' ],
			[ 'fa fa-solid fa-house-flood-water-circle-arrow-right' => 'House Flood Water Circle Arrow Right' ],
			[ 'fa fa-solid fa-house-lock' => 'House Lock' ],
			[ 'fa fa-solid fa-house-medical' => 'House Medical' ],
			[ 'fa fa-solid fa-house-medical-circle-check' => 'House Medical Circle Check' ],
			[ 'fa fa-solid fa-house-medical-circle-exclamation' => 'House Medical Circle Exclamation' ],
			[ 'fa fa-solid fa-house-medical-circle-xmark' => 'House Medical Circle Xmark' ],
			[ 'fa fa-solid fa-house-medical-flag' => 'House Medical Flag' ],
			[ 'fa fa-solid fa-house-signal' => 'House Signal' ],
			[ 'fa fa-solid fa-house-tsunami' => 'House Tsunami' ],
			[ 'fa fa-solid fa-hurricane' => 'Hurricane' ],
			[ 'fa fa-solid fa-id-card' => 'Id Card' ],
			[ 'fa fa-regular fa-id-card' => 'Id Card' ],
			[ 'fa fa-solid fa-jar' => 'Jar' ],
			[ 'fa fa-solid fa-jar-wheat' => 'Jar Wheat' ],
			[ 'fa fa-solid fa-jet-fighter-up' => 'Jet Fighter Up' ],
			[ 'fa fa-solid fa-jug-detergent' => 'Jug Detergent' ],
			[ 'fa fa-solid fa-kitchen-set' => 'Kitchen Set' ],
			[ 'fa fa-solid fa-land-mine-on' => 'Land Mine On' ],
			[ 'fa fa-solid fa-landmark' => 'Landmark' ],
			[ 'fa fa-solid fa-landmark-dome' => 'Landmark Dome' ],
			[ 'fa fa-solid fa-landmark-flag' => 'Landmark Flag' ],
			[ 'fa fa-solid fa-laptop' => 'Laptop' ],
			[ 'fa fa-solid fa-laptop-file' => 'Laptop File' ],
			[ 'fa fa-solid fa-life-ring' => 'Life Ring' ],
			[ 'fa fa-regular fa-life-ring' => 'Life Ring' ],
			[ 'fa fa-solid fa-lines-leaning' => 'Lines Leaning' ],
			[ 'fa fa-solid fa-location-pin-lock' => 'Location Pin Lock' ],
			[ 'fa fa-solid fa-locust' => 'Locust' ],
			[ 'fa fa-solid fa-lungs' => 'Lungs' ],
			[ 'fa fa-solid fa-magnifying-glass-arrow-right' => 'Magnifying Glass Arrow Right' ],
			[ 'fa fa-solid fa-magnifying-glass-chart' => 'Magnifying Glass Chart' ],
			[ 'fa fa-solid fa-mars-and-venus' => 'Mars And Venus' ],
			[ 'fa fa-solid fa-mars-and-venus-burst' => 'Mars And Venus Burst' ],
			[ 'fa fa-solid fa-mask-face' => 'Mask Face' ],
			[ 'fa fa-solid fa-mask-ventilator' => 'Mask Ventilator' ],
			[ 'fa fa-solid fa-mattress-pillow' => 'Mattress Pillow' ],
			[ 'fa fa-solid fa-microscope' => 'Microscope' ],
			[ 'fa fa-solid fa-mobile-retro' => 'Mobile Retro' ],
			[ 'fa fa-solid fa-mobile-screen' => 'Mobile Screen' ],
			[ 'fa fa-solid fa-money-bill-transfer' => 'Money Bill Transfer' ],
			[ 'fa fa-solid fa-money-bill-trend-up' => 'Money Bill Trend Up' ],
			[ 'fa fa-solid fa-money-bill-wheat' => 'Money Bill Wheat' ],
			[ 'fa fa-solid fa-money-bills' => 'Money Bills' ],
			[ 'fa fa-solid fa-mosque' => 'Mosque' ],
			[ 'fa fa-solid fa-mosquito' => 'Mosquito' ],
			[ 'fa fa-solid fa-mosquito-net' => 'Mosquito Net' ],
			[ 'fa fa-solid fa-mound' => 'Mound' ],
			[ 'fa fa-solid fa-mountain-city' => 'Mountain City' ],
			[ 'fa fa-solid fa-mountain-sun' => 'Mountain Sun' ],
			[ 'fa fa-solid fa-oil-well' => 'Oil Well' ],
			[ 'fa fa-solid fa-parachute-box' => 'Parachute Box' ],
			[ 'fa fa-solid fa-people-arrows' => 'People Arrows' ],
			[ 'fa fa-solid fa-people-group' => 'People Group' ],
			[ 'fa fa-solid fa-people-line' => 'People Line' ],
			[ 'fa fa-solid fa-people-pulling' => 'People Pulling' ],
			[ 'fa fa-solid fa-people-robbery' => 'People Robbery' ],
			[ 'fa fa-solid fa-people-roof' => 'People Roof' ],
			[ 'fa fa-solid fa-person' => 'Person' ],
			[ 'fa fa-solid fa-person-arrow-down-to-line' => 'Person Arrow Down To Line' ],
			[ 'fa fa-solid fa-person-arrow-up-from-line' => 'Person Arrow Up From Line' ],
			[ 'fa fa-solid fa-person-breastfeeding' => 'Person Breastfeeding' ],
			[ 'fa fa-solid fa-person-burst' => 'Person Burst' ],
			[ 'fa fa-solid fa-person-cane' => 'Person Cane' ],
			[ 'fa fa-solid fa-person-chalkboard' => 'Person Chalkboard' ],
			[ 'fa fa-solid fa-person-circle-check' => 'Person Circle Check' ],
			[ 'fa fa-solid fa-person-circle-exclamation' => 'Person Circle Exclamation' ],
			[ 'fa fa-solid fa-person-circle-minus' => 'Person Circle Minus' ],
			[ 'fa fa-solid fa-person-circle-plus' => 'Person Circle Plus' ],
			[ 'fa fa-solid fa-person-circle-question' => 'Person Circle Question' ],
			[ 'fa fa-solid fa-person-circle-xmark' => 'Person Circle Xmark' ],
			[ 'fa fa-solid fa-person-digging' => 'Person Digging' ],
			[ 'fa fa-solid fa-person-dress' => 'Person Dress' ],
			[ 'fa fa-solid fa-person-dress-burst' => 'Person Dress Burst' ],
			[ 'fa fa-solid fa-person-drowning' => 'Person Drowning' ],
			[ 'fa fa-solid fa-person-falling' => 'Person Falling' ],
			[ 'fa fa-solid fa-person-falling-burst' => 'Person Falling Burst' ],
			[ 'fa fa-solid fa-person-half-dress' => 'Person Half Dress' ],
			[ 'fa fa-solid fa-person-harassing' => 'Person Harassing' ],
			[ 'fa fa-solid fa-person-military-pointing' => 'Person Military Pointing' ],
			[ 'fa fa-solid fa-person-military-rifle' => 'Person Military Rifle' ],
			[ 'fa fa-solid fa-person-military-to-person' => 'Person Military To Person' ],
			[ 'fa fa-solid fa-person-pregnant' => 'Person Pregnant' ],
			[ 'fa fa-solid fa-person-rays' => 'Person Rays' ],
			[ 'fa fa-solid fa-person-rifle' => 'Person Rifle' ],
			[ 'fa fa-solid fa-person-shelter' => 'Person Shelter' ],
			[ 'fa fa-solid fa-person-through-window' => 'Person Through Window' ],
			[ 'fa fa-solid fa-person-walking' => 'Person Walking' ],
			[ 'fa fa-solid fa-person-walking-arrow-loop-left' => 'Person Walking Arrow Loop Left' ],
			[ 'fa fa-solid fa-person-walking-arrow-right' => 'Person Walking Arrow Right' ],
			[ 'fa fa-solid fa-person-walking-dashed-line-arrow-right' => 'Person Walking Dashed Line Arrow Right' ],
			[ 'fa fa-solid fa-person-walking-luggage' => 'Person Walking Luggage' ],
			[ 'fa fa-solid fa-pills' => 'Pills' ],
			[ 'fa fa-solid fa-plane-circle-check' => 'Plane Circle Check' ],
			[ 'fa fa-solid fa-plane-circle-exclamation' => 'Plane Circle Exclamation' ],
			[ 'fa fa-solid fa-plane-circle-xmark' => 'Plane Circle Xmark' ],
			[ 'fa fa-solid fa-plane-lock' => 'Plane Lock' ],
			[ 'fa fa-solid fa-plane-up' => 'Plane Up' ],
			[ 'fa fa-solid fa-plant-wilt' => 'Plant Wilt' ],
			[ 'fa fa-solid fa-plate-wheat' => 'Plate Wheat' ],
			[ 'fa fa-solid fa-plug' => 'Plug' ],
			[ 'fa fa-solid fa-plug-circle-bolt' => 'Plug Circle Bolt' ],
			[ 'fa fa-solid fa-plug-circle-check' => 'Plug Circle Check' ],
			[ 'fa fa-solid fa-plug-circle-exclamation' => 'Plug Circle Exclamation' ],
			[ 'fa fa-solid fa-plug-circle-minus' => 'Plug Circle Minus' ],
			[ 'fa fa-solid fa-plug-circle-plus' => 'Plug Circle Plus' ],
			[ 'fa fa-solid fa-plug-circle-xmark' => 'Plug Circle Xmark' ],
			[ 'fa fa-solid fa-pump-soap' => 'Pump Soap' ],
			[ 'fa fa-solid fa-radiation' => 'Radiation' ],
			[ 'fa fa-solid fa-radio' => 'Radio' ],
			[ 'fa fa-solid fa-ranking-star' => 'Ranking Star' ],
			[ 'fa fa-solid fa-road' => 'Road' ],
			[ 'fa fa-solid fa-road-barrier' => 'Road Barrier' ],
			[ 'fa fa-solid fa-road-bridge' => 'Road Bridge' ],
			[ 'fa fa-solid fa-road-circle-check' => 'Road Circle Check' ],
			[ 'fa fa-solid fa-road-circle-exclamation' => 'Road Circle Exclamation' ],
			[ 'fa fa-solid fa-road-circle-xmark' => 'Road Circle Xmark' ],
			[ 'fa fa-solid fa-road-lock' => 'Road Lock' ],
			[ 'fa fa-solid fa-road-spikes' => 'Road Spikes' ],
			[ 'fa fa-solid fa-rug' => 'Rug' ],
			[ 'fa fa-solid fa-sack-dollar' => 'Sack Dollar' ],
			[ 'fa fa-solid fa-sack-xmark' => 'Sack Xmark' ],
			[ 'fa fa-solid fa-sailboat' => 'Sailboat' ],
			[ 'fa fa-solid fa-satellite-dish' => 'Satellite Dish' ],
			[ 'fa fa-solid fa-scale-balanced' => 'Scale Balanced' ],
			[ 'fa fa-solid fa-school' => 'School' ],
			[ 'fa fa-solid fa-school-circle-check' => 'School Circle Check' ],
			[ 'fa fa-solid fa-school-circle-exclamation' => 'School Circle Exclamation' ],
			[ 'fa fa-solid fa-school-circle-xmark' => 'School Circle Xmark' ],
			[ 'fa fa-solid fa-school-flag' => 'School Flag' ],
			[ 'fa fa-solid fa-school-lock' => 'School Lock' ],
			[ 'fa fa-solid fa-seedling' => 'Seedling' ],
			[ 'fa fa-solid fa-sheet-plastic' => 'Sheet Plastic' ],
			[ 'fa fa-solid fa-shield-cat' => 'Shield Cat' ],
			[ 'fa fa-solid fa-shield-dog' => 'Shield Dog' ],
			[ 'fa fa-solid fa-shield-heart' => 'Shield Heart' ],
			[ 'fa fa-solid fa-ship' => 'Ship' ],
			[ 'fa fa-solid fa-shirt' => 'Shirt' ],
			[ 'fa fa-solid fa-shop' => 'Shop' ],
			[ 'fa fa-solid fa-shop-lock' => 'Shop Lock' ],
			[ 'fa fa-solid fa-shower' => 'Shower' ],
			[ 'fa fa-solid fa-skull-crossbones' => 'Skull Crossbones' ],
			[ 'fa fa-solid fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-regular fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-solid fa-soap' => 'Soap' ],
			[ 'fa fa-solid fa-square-nfi' => 'Square Nfi' ],
			[ 'fa fa-solid fa-square-person-confined' => 'Square Person Confined' ],
			[ 'fa fa-solid fa-square-virus' => 'Square Virus' ],
			[ 'fa fa-solid fa-staff-snake' => 'Staff Snake' ],
			[ 'fa fa-solid fa-stethoscope' => 'Stethoscope' ],
			[ 'fa fa-solid fa-suitcase-medical' => 'Suitcase Medical' ],
			[ 'fa fa-solid fa-sun-plant-wilt' => 'Sun Plant Wilt' ],
			[ 'fa fa-solid fa-syringe' => 'Syringe' ],
			[ 'fa fa-solid fa-tarp' => 'Tarp' ],
			[ 'fa fa-solid fa-tarp-droplet' => 'Tarp Droplet' ],
			[ 'fa fa-solid fa-temperature-arrow-down' => 'Temperature Arrow Down' ],
			[ 'fa fa-solid fa-temperature-arrow-up' => 'Temperature Arrow Up' ],
			[ 'fa fa-solid fa-tent' => 'Tent' ],
			[ 'fa fa-solid fa-tent-arrow-down-to-line' => 'Tent Arrow Down To Line' ],
			[ 'fa fa-solid fa-tent-arrow-left-right' => 'Tent Arrow Left Right' ],
			[ 'fa fa-solid fa-tent-arrow-turn-left' => 'Tent Arrow Turn Left' ],
			[ 'fa fa-solid fa-tent-arrows-down' => 'Tent Arrows Down' ],
			[ 'fa fa-solid fa-tents' => 'Tents' ],
			[ 'fa fa-solid fa-toilet' => 'Toilet' ],
			[ 'fa fa-solid fa-toilet-portable' => 'Toilet Portable' ],
			[ 'fa fa-solid fa-toilets-portable' => 'Toilets Portable' ],
			[ 'fa fa-solid fa-tornado' => 'Tornado' ],
			[ 'fa fa-solid fa-tower-broadcast' => 'Tower Broadcast' ],
			[ 'fa fa-solid fa-tower-cell' => 'Tower Cell' ],
			[ 'fa fa-solid fa-tower-observation' => 'Tower Observation' ],
			[ 'fa fa-solid fa-train-subway' => 'Train Subway' ],
			[ 'fa fa-solid fa-trash-can' => 'Trash Can' ],
			[ 'fa fa-regular fa-trash-can' => 'Trash Can' ],
			[ 'fa fa-solid fa-tree-city' => 'Tree City' ],
			[ 'fa fa-solid fa-trowel' => 'Trowel' ],
			[ 'fa fa-solid fa-trowel-bricks' => 'Trowel Bricks' ],
			[ 'fa fa-solid fa-truck' => 'Truck' ],
			[ 'fa fa-solid fa-truck-arrow-right' => 'Truck Arrow Right' ],
			[ 'fa fa-solid fa-truck-droplet' => 'Truck Droplet' ],
			[ 'fa fa-solid fa-truck-field' => 'Truck Field' ],
			[ 'fa fa-solid fa-truck-field-un' => 'Truck Field Un' ],
			[ 'fa fa-solid fa-truck-front' => 'Truck Front' ],
			[ 'fa fa-solid fa-truck-medical' => 'Truck Medical' ],
			[ 'fa fa-solid fa-truck-plane' => 'Truck Plane' ],
			[ 'fa fa-solid fa-user-doctor' => 'User Doctor' ],
			[ 'fa fa-solid fa-user-injured' => 'User Injured' ],
			[ 'fa fa-solid fa-users-between-lines' => 'Users Between Lines' ],
			[ 'fa fa-solid fa-users-line' => 'Users Line' ],
			[ 'fa fa-solid fa-users-rays' => 'Users Rays' ],
			[ 'fa fa-solid fa-users-rectangle' => 'Users Rectangle' ],
			[ 'fa fa-solid fa-users-viewfinder' => 'Users Viewfinder' ],
			[ 'fa fa-solid fa-vial-circle-check' => 'Vial Circle Check' ],
			[ 'fa fa-solid fa-vial-virus' => 'Vial Virus' ],
			[ 'fa fa-solid fa-vihara' => 'Vihara' ],
			[ 'fa fa-solid fa-virus' => 'Virus' ],
			[ 'fa fa-solid fa-virus-covid' => 'Virus Covid' ],
			[ 'fa fa-solid fa-volcano' => 'Volcano' ],
			[ 'fa fa-solid fa-walkie-talkie' => 'Walkie Talkie' ],
			[ 'fa fa-solid fa-wheat-awn' => 'Wheat Awn' ],
			[ 'fa fa-solid fa-wheat-awn-circle-exclamation' => 'Wheat Awn Circle Exclamation' ],
			[ 'fa fa-solid fa-wheelchair-move' => 'Wheelchair Move' ],
			[ 'fa fa-solid fa-wifi' => 'Wifi' ],
			[ 'fa fa-solid fa-wind' => 'Wind' ],
			[ 'fa fa-solid fa-worm' => 'Worm' ],
			[ 'fa fa-solid fa-xmarks-lines' => 'Xmarks Lines' ],
		],
		'Logistics' => [
			[ 'fa fa-solid fa-anchor' => 'Anchor' ],
			[ 'fa fa-solid fa-anchor-circle-check' => 'Anchor Circle Check' ],
			[ 'fa fa-solid fa-anchor-circle-exclamation' => 'Anchor Circle Exclamation' ],
			[ 'fa fa-solid fa-anchor-circle-xmark' => 'Anchor Circle Xmark' ],
			[ 'fa fa-solid fa-anchor-lock' => 'Anchor Lock' ],
			[ 'fa fa-solid fa-box' => 'Box' ],
			[ 'fa fa-solid fa-boxes-packing' => 'Boxes Packing' ],
			[ 'fa fa-solid fa-boxes-stacked' => 'Boxes Stacked' ],
			[ 'fa fa-solid fa-bridge' => 'Bridge' ],
			[ 'fa fa-solid fa-bridge-circle-check' => 'Bridge Circle Check' ],
			[ 'fa fa-solid fa-bridge-circle-exclamation' => 'Bridge Circle Exclamation' ],
			[ 'fa fa-solid fa-bridge-circle-xmark' => 'Bridge Circle Xmark' ],
			[ 'fa fa-solid fa-bridge-lock' => 'Bridge Lock' ],
			[ 'fa fa-solid fa-bridge-water' => 'Bridge Water' ],
			[ 'fa fa-solid fa-bus' => 'Bus' ],
			[ 'fa fa-solid fa-bus-simple' => 'Bus Simple' ],
			[ 'fa fa-solid fa-car' => 'Car' ],
			[ 'fa fa-solid fa-car-tunnel' => 'Car Tunnel' ],
			[ 'fa fa-solid fa-cart-flatbed' => 'Cart Flatbed' ],
			[ 'fa fa-solid fa-chart-simple' => 'Chart Simple' ],
			[ 'fa fa-solid fa-clipboard-check' => 'Clipboard Check' ],
			[ 'fa fa-solid fa-clipboard-list' => 'Clipboard List' ],
			[ 'fa fa-solid fa-clipboard-question' => 'Clipboard Question' ],
			[ 'fa fa-solid fa-dolly' => 'Dolly' ],
			[ 'fa fa-solid fa-ferry' => 'Ferry' ],
			[ 'fa fa-solid fa-gas-pump' => 'Gas Pump' ],
			[ 'fa fa-solid fa-gears' => 'Gears' ],
			[ 'fa fa-solid fa-helicopter' => 'Helicopter' ],
			[ 'fa fa-solid fa-helicopter-symbol' => 'Helicopter Symbol' ],
			[ 'fa fa-solid fa-helmet-safety' => 'Helmet Safety' ],
			[ 'fa fa-solid fa-jet-fighter-up' => 'Jet Fighter Up' ],
			[ 'fa fa-solid fa-pallet' => 'Pallet' ],
			[ 'fa fa-solid fa-plane-circle-check' => 'Plane Circle Check' ],
			[ 'fa fa-solid fa-plane-circle-exclamation' => 'Plane Circle Exclamation' ],
			[ 'fa fa-solid fa-plane-circle-xmark' => 'Plane Circle Xmark' ],
			[ 'fa fa-solid fa-plane-lock' => 'Plane Lock' ],
			[ 'fa fa-solid fa-road' => 'Road' ],
			[ 'fa fa-solid fa-road-barrier' => 'Road Barrier' ],
			[ 'fa fa-solid fa-road-bridge' => 'Road Bridge' ],
			[ 'fa fa-solid fa-road-circle-check' => 'Road Circle Check' ],
			[ 'fa fa-solid fa-road-circle-exclamation' => 'Road Circle Exclamation' ],
			[ 'fa fa-solid fa-road-circle-xmark' => 'Road Circle Xmark' ],
			[ 'fa fa-solid fa-road-lock' => 'Road Lock' ],
			[ 'fa fa-solid fa-sailboat' => 'Sailboat' ],
			[ 'fa fa-solid fa-square-nfi' => 'Square Nfi' ],
			[ 'fa fa-solid fa-train' => 'Train' ],
			[ 'fa fa-solid fa-train-subway' => 'Train Subway' ],
			[ 'fa fa-solid fa-truck' => 'Truck' ],
			[ 'fa fa-solid fa-truck-arrow-right' => 'Truck Arrow Right' ],
			[ 'fa fa-solid fa-truck-fast' => 'Truck Fast' ],
			[ 'fa fa-solid fa-truck-field' => 'Truck Field' ],
			[ 'fa fa-solid fa-truck-field-un' => 'Truck Field Un' ],
			[ 'fa fa-solid fa-truck-front' => 'Truck Front' ],
			[ 'fa fa-solid fa-truck-plane' => 'Truck Plane' ],
			[ 'fa fa-solid fa-warehouse' => 'Warehouse' ],
			[ 'fa fa-solid fa-xmarks-lines' => 'Xmarks Lines' ],
		],
		'Maps' => [
			[ 'fa fa-solid fa-anchor' => 'Anchor' ],
			[ 'fa fa-solid fa-bag-shopping' => 'Bag Shopping' ],
			[ 'fa fa-solid fa-basket-shopping' => 'Basket Shopping' ],
			[ 'fa fa-solid fa-bath' => 'Bath' ],
			[ 'fa fa-solid fa-bed' => 'Bed' ],
			[ 'fa fa-solid fa-beer-mug-empty' => 'Beer Mug Empty' ],
			[ 'fa fa-solid fa-bell' => 'Bell' ],
			[ 'fa fa-regular fa-bell' => 'Bell' ],
			[ 'fa fa-solid fa-bell-slash' => 'Bell Slash' ],
			[ 'fa fa-regular fa-bell-slash' => 'Bell Slash' ],
			[ 'fa fa-solid fa-bicycle' => 'Bicycle' ],
			[ 'fa fa-solid fa-binoculars' => 'Binoculars' ],
			[ 'fa fa-solid fa-bomb' => 'Bomb' ],
			[ 'fa fa-solid fa-book' => 'Book' ],
			[ 'fa fa-solid fa-book-atlas' => 'Book Atlas' ],
			[ 'fa fa-solid fa-bookmark' => 'Bookmark' ],
			[ 'fa fa-regular fa-bookmark' => 'Bookmark' ],
			[ 'fa fa-solid fa-bridge' => 'Bridge' ],
			[ 'fa fa-solid fa-bridge-water' => 'Bridge Water' ],
			[ 'fa fa-solid fa-briefcase' => 'Briefcase' ],
			[ 'fa fa-solid fa-building' => 'Building' ],
			[ 'fa fa-regular fa-building' => 'Building' ],
			[ 'fa fa-solid fa-building-columns' => 'Building Columns' ],
			[ 'fa fa-solid fa-cake-candles' => 'Cake Candles' ],
			[ 'fa fa-solid fa-car' => 'Car' ],
			[ 'fa fa-solid fa-cart-shopping' => 'Cart Shopping' ],
			[ 'fa fa-solid fa-circle-info' => 'Circle Info' ],
			[ 'fa fa-solid fa-crosshairs' => 'Crosshairs' ],
			[ 'fa fa-solid fa-diamond-turn-right' => 'Diamond Turn Right' ],
			[ 'fa fa-solid fa-dollar-sign' => 'Dollar Sign' ],
			[ 'fa fa-solid fa-draw-polygon' => 'Draw Polygon' ],
			[ 'fa fa-solid fa-droplet' => 'Droplet' ],
			[ 'fa fa-solid fa-eye' => 'Eye' ],
			[ 'fa fa-regular fa-eye' => 'Eye' ],
			[ 'fa fa-solid fa-eye-low-vision' => 'Eye Low Vision' ],
			[ 'fa fa-solid fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-regular fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-solid fa-fire' => 'Fire' ],
			[ 'fa fa-solid fa-fire-extinguisher' => 'Fire Extinguisher' ],
			[ 'fa fa-solid fa-fire-flame-curved' => 'Fire Flame Curved' ],
			[ 'fa fa-solid fa-flag' => 'Flag' ],
			[ 'fa fa-regular fa-flag' => 'Flag' ],
			[ 'fa fa-solid fa-flag-checkered' => 'Flag Checkered' ],
			[ 'fa fa-solid fa-flask' => 'Flask' ],
			[ 'fa fa-solid fa-gamepad' => 'Gamepad' ],
			[ 'fa fa-solid fa-gavel' => 'Gavel' ],
			[ 'fa fa-solid fa-gift' => 'Gift' ],
			[ 'fa fa-solid fa-globe' => 'Globe' ],
			[ 'fa fa-solid fa-graduation-cap' => 'Graduation Cap' ],
			[ 'fa fa-solid fa-heart' => 'Heart' ],
			[ 'fa fa-regular fa-heart' => 'Heart' ],
			[ 'fa fa-solid fa-heart-pulse' => 'Heart Pulse' ],
			[ 'fa fa-solid fa-helicopter' => 'Helicopter' ],
			[ 'fa fa-solid fa-helicopter-symbol' => 'Helicopter Symbol' ],
			[ 'fa fa-solid fa-hospital' => 'Hospital' ],
			[ 'fa fa-regular fa-hospital' => 'Hospital' ],
			[ 'fa fa-solid fa-house' => 'House' ],
			[ 'fa fa-solid fa-image' => 'Image' ],
			[ 'fa fa-regular fa-image' => 'Image' ],
			[ 'fa fa-solid fa-images' => 'Images' ],
			[ 'fa fa-regular fa-images' => 'Images' ],
			[ 'fa fa-solid fa-industry' => 'Industry' ],
			[ 'fa fa-solid fa-info' => 'Info' ],
			[ 'fa fa-solid fa-jet-fighter' => 'Jet Fighter' ],
			[ 'fa fa-solid fa-key' => 'Key' ],
			[ 'fa fa-solid fa-landmark' => 'Landmark' ],
			[ 'fa fa-solid fa-landmark-flag' => 'Landmark Flag' ],
			[ 'fa fa-solid fa-layer-group' => 'Layer Group' ],
			[ 'fa fa-solid fa-leaf' => 'Leaf' ],
			[ 'fa fa-solid fa-lemon' => 'Lemon' ],
			[ 'fa fa-regular fa-lemon' => 'Lemon' ],
			[ 'fa fa-solid fa-life-ring' => 'Life Ring' ],
			[ 'fa fa-regular fa-life-ring' => 'Life Ring' ],
			[ 'fa fa-solid fa-lightbulb' => 'Lightbulb' ],
			[ 'fa fa-regular fa-lightbulb' => 'Lightbulb' ],
			[ 'fa fa-solid fa-location-arrow' => 'Location Arrow' ],
			[ 'fa fa-solid fa-location-crosshairs' => 'Location Crosshairs' ],
			[ 'fa fa-solid fa-location-dot' => 'Location Dot' ],
			[ 'fa fa-solid fa-location-pin' => 'Location Pin' ],
			[ 'fa fa-solid fa-location-pin-lock' => 'Location Pin Lock' ],
			[ 'fa fa-solid fa-magnet' => 'Magnet' ],
			[ 'fa fa-solid fa-magnifying-glass' => 'Magnifying Glass' ],
			[ 'fa fa-solid fa-magnifying-glass-location' => 'Magnifying Glass Location' ],
			[ 'fa fa-solid fa-magnifying-glass-minus' => 'Magnifying Glass Minus' ],
			[ 'fa fa-solid fa-magnifying-glass-plus' => 'Magnifying Glass Plus' ],
			[ 'fa fa-solid fa-map' => 'Map' ],
			[ 'fa fa-regular fa-map' => 'Map' ],
			[ 'fa fa-solid fa-map-pin' => 'Map Pin' ],
			[ 'fa fa-solid fa-martini-glass-empty' => 'Martini Glass Empty' ],
			[ 'fa fa-solid fa-money-bill' => 'Money Bill' ],
			[ 'fa fa-solid fa-money-bill-1' => 'Money Bill 1' ],
			[ 'fa fa-regular fa-money-bill-1' => 'Money Bill 1' ],
			[ 'fa fa-solid fa-monument' => 'Monument' ],
			[ 'fa fa-solid fa-motorcycle' => 'Motorcycle' ],
			[ 'fa fa-solid fa-mountain-sun' => 'Mountain Sun' ],
			[ 'fa fa-solid fa-mug-saucer' => 'Mug Saucer' ],
			[ 'fa fa-solid fa-music' => 'Music' ],
			[ 'fa fa-solid fa-newspaper' => 'Newspaper' ],
			[ 'fa fa-regular fa-newspaper' => 'Newspaper' ],
			[ 'fa fa-solid fa-paw' => 'Paw' ],
			[ 'fa fa-solid fa-person' => 'Person' ],
			[ 'fa fa-solid fa-person-walking-with-cane' => 'Person Walking With Cane' ],
			[ 'fa fa-solid fa-phone' => 'Phone' ],
			[ 'fa fa-solid fa-phone-flip' => 'Phone Flip' ],
			[ 'fa fa-solid fa-phone-volume' => 'Phone Volume' ],
			[ 'fa fa-solid fa-plane' => 'Plane' ],
			[ 'fa fa-solid fa-plug' => 'Plug' ],
			[ 'fa fa-solid fa-plus' => 'Plus' ],
			[ 'fa fa-solid fa-print' => 'Print' ],
			[ 'fa fa-solid fa-recycle' => 'Recycle' ],
			[ 'fa fa-solid fa-restroom' => 'Restroom' ],
			[ 'fa fa-solid fa-road' => 'Road' ],
			[ 'fa fa-solid fa-rocket' => 'Rocket' ],
			[ 'fa fa-solid fa-route' => 'Route' ],
			[ 'fa fa-solid fa-scale-balanced' => 'Scale Balanced' ],
			[ 'fa fa-solid fa-scale-unbalanced' => 'Scale Unbalanced' ],
			[ 'fa fa-solid fa-scale-unbalanced-flip' => 'Scale Unbalanced Flip' ],
			[ 'fa fa-solid fa-ship' => 'Ship' ],
			[ 'fa fa-solid fa-shoe-prints' => 'Shoe Prints' ],
			[ 'fa fa-solid fa-shower' => 'Shower' ],
			[ 'fa fa-solid fa-signs-post' => 'Signs Post' ],
			[ 'fa fa-solid fa-snowplow' => 'Snowplow' ],
			[ 'fa fa-solid fa-spoon' => 'Spoon' ],
			[ 'fa fa-solid fa-square-h' => 'Square H' ],
			[ 'fa fa-solid fa-square-parking' => 'Square Parking' ],
			[ 'fa fa-solid fa-square-phone' => 'Square Phone' ],
			[ 'fa fa-solid fa-square-phone-flip' => 'Square Phone Flip' ],
			[ 'fa fa-solid fa-square-plus' => 'Square Plus' ],
			[ 'fa fa-regular fa-square-plus' => 'Square Plus' ],
			[ 'fa fa-solid fa-street-view' => 'Street View' ],
			[ 'fa fa-solid fa-suitcase' => 'Suitcase' ],
			[ 'fa fa-solid fa-suitcase-medical' => 'Suitcase Medical' ],
			[ 'fa fa-solid fa-tag' => 'Tag' ],
			[ 'fa fa-solid fa-tags' => 'Tags' ],
			[ 'fa fa-solid fa-taxi' => 'Taxi' ],
			[ 'fa fa-solid fa-thumbtack' => 'Thumbtack' ],
			[ 'fa fa-solid fa-thumbtack-slash' => 'Thumbtack Slash' ],
			[ 'fa fa-solid fa-ticket' => 'Ticket' ],
			[ 'fa fa-solid fa-ticket-simple' => 'Ticket Simple' ],
			[ 'fa fa-solid fa-traffic-light' => 'Traffic Light' ],
			[ 'fa fa-solid fa-train' => 'Train' ],
			[ 'fa fa-solid fa-train-subway' => 'Train Subway' ],
			[ 'fa fa-solid fa-train-tram' => 'Train Tram' ],
			[ 'fa fa-solid fa-tree' => 'Tree' ],
			[ 'fa fa-solid fa-trophy' => 'Trophy' ],
			[ 'fa fa-solid fa-truck' => 'Truck' ],
			[ 'fa fa-solid fa-truck-medical' => 'Truck Medical' ],
			[ 'fa fa-solid fa-tty' => 'Tty' ],
			[ 'fa fa-solid fa-umbrella' => 'Umbrella' ],
			[ 'fa fa-solid fa-utensils' => 'Utensils' ],
			[ 'fa fa-solid fa-vest' => 'Vest' ],
			[ 'fa fa-solid fa-vest-patches' => 'Vest Patches' ],
			[ 'fa fa-solid fa-wheelchair' => 'Wheelchair' ],
			[ 'fa fa-solid fa-wheelchair-move' => 'Wheelchair Move' ],
			[ 'fa fa-solid fa-wifi' => 'Wifi' ],
			[ 'fa fa-solid fa-wine-glass' => 'Wine Glass' ],
			[ 'fa fa-solid fa-wrench' => 'Wrench' ],
		],
		'Maritime' => [
			[ 'fa fa-solid fa-anchor' => 'Anchor' ],
			[ 'fa fa-solid fa-anchor-circle-check' => 'Anchor Circle Check' ],
			[ 'fa fa-solid fa-anchor-circle-exclamation' => 'Anchor Circle Exclamation' ],
			[ 'fa fa-solid fa-anchor-circle-xmark' => 'Anchor Circle Xmark' ],
			[ 'fa fa-solid fa-anchor-lock' => 'Anchor Lock' ],
			[ 'fa fa-solid fa-ferry' => 'Ferry' ],
			[ 'fa fa-solid fa-fish' => 'Fish' ],
			[ 'fa fa-solid fa-fish-fins' => 'Fish Fins' ],
			[ 'fa fa-solid fa-otter' => 'Otter' ],
			[ 'fa fa-solid fa-person-swimming' => 'Person Swimming' ],
			[ 'fa fa-solid fa-sailboat' => 'Sailboat' ],
			[ 'fa fa-solid fa-ship' => 'Ship' ],
			[ 'fa fa-solid fa-shrimp' => 'Shrimp' ],
			[ 'fa fa-solid fa-water' => 'Water' ],
		],
		'Marketing' => [
			[ 'fa fa-solid fa-arrows-spin' => 'Arrows Spin' ],
			[ 'fa fa-solid fa-arrows-to-dot' => 'Arrows To Dot' ],
			[ 'fa fa-solid fa-arrows-to-eye' => 'Arrows To Eye' ],
			[ 'fa fa-solid fa-bullhorn' => 'Bullhorn' ],
			[ 'fa fa-solid fa-bullseye' => 'Bullseye' ],
			[ 'fa fa-solid fa-chart-simple' => 'Chart Simple' ],
			[ 'fa fa-solid fa-comment-dollar' => 'Comment Dollar' ],
			[ 'fa fa-solid fa-comments-dollar' => 'Comments Dollar' ],
			[ 'fa fa-solid fa-envelope-open-text' => 'Envelope Open Text' ],
			[ 'fa fa-solid fa-envelopes-bulk' => 'Envelopes Bulk' ],
			[ 'fa fa-solid fa-filter-circle-dollar' => 'Filter Circle Dollar' ],
			[ 'fa fa-solid fa-group-arrows-rotate' => 'Group Arrows Rotate' ],
			[ 'fa fa-solid fa-lightbulb' => 'Lightbulb' ],
			[ 'fa fa-regular fa-lightbulb' => 'Lightbulb' ],
			[ 'fa fa-solid fa-magnifying-glass-arrow-right' => 'Magnifying Glass Arrow Right' ],
			[ 'fa fa-solid fa-magnifying-glass-chart' => 'Magnifying Glass Chart' ],
			[ 'fa fa-solid fa-magnifying-glass-dollar' => 'Magnifying Glass Dollar' ],
			[ 'fa fa-solid fa-magnifying-glass-location' => 'Magnifying Glass Location' ],
			[ 'fa fa-solid fa-people-group' => 'People Group' ],
			[ 'fa fa-solid fa-person-rays' => 'Person Rays' ],
			[ 'fa fa-solid fa-ranking-star' => 'Ranking Star' ],
			[ 'fa fa-solid fa-rectangle-ad' => 'Rectangle Ad' ],
			[ 'fa fa-solid fa-square-poll-horizontal' => 'Square Poll Horizontal' ],
			[ 'fa fa-solid fa-square-poll-vertical' => 'Square Poll Vertical' ],
			[ 'fa fa-solid fa-timeline' => 'Timeline' ],
		],
		'Mathematics' => [
			[ 'fa fa-solid fa-calculator' => 'Calculator' ],
			[ 'fa fa-solid fa-circle-minus' => 'Circle Minus' ],
			[ 'fa fa-solid fa-circle-plus' => 'Circle Plus' ],
			[ 'fa fa-solid fa-circle-xmark' => 'Circle Xmark' ],
			[ 'fa fa-regular fa-circle-xmark' => 'Circle Xmark' ],
			[ 'fa fa-solid fa-divide' => 'Divide' ],
			[ 'fa fa-solid fa-equals' => 'Equals' ],
			[ 'fa fa-solid fa-greater-than' => 'Greater Than' ],
			[ 'fa fa-solid fa-greater-than-equal' => 'Greater Than Equal' ],
			[ 'fa fa-solid fa-infinity' => 'Infinity' ],
			[ 'fa fa-solid fa-less-than' => 'Less Than' ],
			[ 'fa fa-solid fa-less-than-equal' => 'Less Than Equal' ],
			[ 'fa fa-solid fa-minus' => 'Minus' ],
			[ 'fa fa-solid fa-not-equal' => 'Not Equal' ],
			[ 'fa fa-solid fa-percent' => 'Percent' ],
			[ 'fa fa-solid fa-plus' => 'Plus' ],
			[ 'fa fa-solid fa-plus-minus' => 'Plus Minus' ],
			[ 'fa fa-solid fa-square-minus' => 'Square Minus' ],
			[ 'fa fa-regular fa-square-minus' => 'Square Minus' ],
			[ 'fa fa-solid fa-square-root-variable' => 'Square Root Variable' ],
			[ 'fa fa-solid fa-square-xmark' => 'Square Xmark' ],
			[ 'fa fa-solid fa-subscript' => 'Subscript' ],
			[ 'fa fa-solid fa-superscript' => 'Superscript' ],
			[ 'fa fa-solid fa-wave-square' => 'Wave Square' ],
			[ 'fa fa-solid fa-xmark' => 'Xmark' ],
		],
		'Media-playback' => [
			[ 'fa fa-solid fa-arrow-rotate-left' => 'Arrow Rotate Left' ],
			[ 'fa fa-solid fa-arrow-rotate-right' => 'Arrow Rotate Right' ],
			[ 'fa fa-solid fa-arrows-rotate' => 'Arrows Rotate' ],
			[ 'fa fa-solid fa-backward' => 'Backward' ],
			[ 'fa fa-solid fa-backward-fast' => 'Backward Fast' ],
			[ 'fa fa-solid fa-backward-step' => 'Backward Step' ],
			[ 'fa fa-solid fa-circle-pause' => 'Circle Pause' ],
			[ 'fa fa-regular fa-circle-pause' => 'Circle Pause' ],
			[ 'fa fa-solid fa-circle-play' => 'Circle Play' ],
			[ 'fa fa-regular fa-circle-play' => 'Circle Play' ],
			[ 'fa fa-solid fa-circle-stop' => 'Circle Stop' ],
			[ 'fa fa-regular fa-circle-stop' => 'Circle Stop' ],
			[ 'fa fa-solid fa-compress' => 'Compress' ],
			[ 'fa fa-solid fa-down-left-and-up-right-to-center' => 'Down Left And Up Right To Center' ],
			[ 'fa fa-solid fa-eject' => 'Eject' ],
			[ 'fa fa-solid fa-expand' => 'Expand' ],
			[ 'fa fa-solid fa-forward' => 'Forward' ],
			[ 'fa fa-solid fa-forward-fast' => 'Forward Fast' ],
			[ 'fa fa-solid fa-forward-step' => 'Forward Step' ],
			[ 'fa fa-solid fa-hand' => 'Hand' ],
			[ 'fa fa-regular fa-hand' => 'Hand' ],
			[ 'fa fa-solid fa-maximize' => 'Maximize' ],
			[ 'fa fa-solid fa-minimize' => 'Minimize' ],
			[ 'fa fa-solid fa-music' => 'Music' ],
			[ 'fa fa-solid fa-pause' => 'Pause' ],
			[ 'fa fa-solid fa-phone-volume' => 'Phone Volume' ],
			[ 'fa fa-solid fa-play' => 'Play' ],
			[ 'fa fa-solid fa-plus-minus' => 'Plus Minus' ],
			[ 'fa fa-solid fa-repeat' => 'Repeat' ],
			[ 'fa fa-solid fa-rotate' => 'Rotate' ],
			[ 'fa fa-solid fa-rotate-left' => 'Rotate Left' ],
			[ 'fa fa-solid fa-rotate-right' => 'Rotate Right' ],
			[ 'fa fa-solid fa-rss' => 'Rss' ],
			[ 'fa fa-solid fa-shuffle' => 'Shuffle' ],
			[ 'fa fa-solid fa-sliders' => 'Sliders' ],
			[ 'fa fa-solid fa-stop' => 'Stop' ],
			[ 'fa fa-solid fa-up-right-and-down-left-from-center' => 'Up Right And Down Left From Center' ],
			[ 'fa fa-solid fa-volume-high' => 'Volume High' ],
			[ 'fa fa-solid fa-volume-low' => 'Volume Low' ],
			[ 'fa fa-solid fa-volume-off' => 'Volume Off' ],
			[ 'fa fa-solid fa-volume-xmark' => 'Volume Xmark' ],
		],
		'Medical-health' => [
			[ 'fa fa-brands fa-accessible-icon' => 'Accessible Icon' ],
			[ 'fa fa-solid fa-bacteria' => 'Bacteria' ],
			[ 'fa fa-solid fa-bacterium' => 'Bacterium' ],
			[ 'fa fa-solid fa-ban-smoking' => 'Ban Smoking' ],
			[ 'fa fa-solid fa-bandage' => 'Bandage' ],
			[ 'fa fa-solid fa-bed-pulse' => 'Bed Pulse' ],
			[ 'fa fa-solid fa-biohazard' => 'Biohazard' ],
			[ 'fa fa-solid fa-bone' => 'Bone' ],
			[ 'fa fa-solid fa-bong' => 'Bong' ],
			[ 'fa fa-solid fa-book-medical' => 'Book Medical' ],
			[ 'fa fa-solid fa-brain' => 'Brain' ],
			[ 'fa fa-solid fa-briefcase-medical' => 'Briefcase Medical' ],
			[ 'fa fa-solid fa-cannabis' => 'Cannabis' ],
			[ 'fa fa-solid fa-capsules' => 'Capsules' ],
			[ 'fa fa-solid fa-circle-h' => 'Circle H' ],
			[ 'fa fa-solid fa-circle-radiation' => 'Circle Radiation' ],
			[ 'fa fa-solid fa-clipboard-user' => 'Clipboard User' ],
			[ 'fa fa-solid fa-clock-rotate-left' => 'Clock Rotate Left' ],
			[ 'fa fa-solid fa-comment-medical' => 'Comment Medical' ],
			[ 'fa fa-solid fa-crutch' => 'Crutch' ],
			[ 'fa fa-solid fa-disease' => 'Disease' ],
			[ 'fa fa-solid fa-dna' => 'Dna' ],
			[ 'fa fa-solid fa-droplet' => 'Droplet' ],
			[ 'fa fa-solid fa-droplet-slash' => 'Droplet Slash' ],
			[ 'fa fa-solid fa-eye' => 'Eye' ],
			[ 'fa fa-regular fa-eye' => 'Eye' ],
			[ 'fa fa-solid fa-eye-dropper' => 'Eye Dropper' ],
			[ 'fa fa-solid fa-file-medical' => 'File Medical' ],
			[ 'fa fa-solid fa-file-prescription' => 'File Prescription' ],
			[ 'fa fa-solid fa-file-waveform' => 'File Waveform' ],
			[ 'fa fa-solid fa-fire-flame-simple' => 'Fire Flame Simple' ],
			[ 'fa fa-solid fa-flask' => 'Flask' ],
			[ 'fa fa-solid fa-flask-vial' => 'Flask Vial' ],
			[ 'fa fa-solid fa-hand-dots' => 'Hand Dots' ],
			[ 'fa fa-solid fa-hand-holding-droplet' => 'Hand Holding Droplet' ],
			[ 'fa fa-solid fa-hand-holding-medical' => 'Hand Holding Medical' ],
			[ 'fa fa-solid fa-head-side-cough' => 'Head Side Cough' ],
			[ 'fa fa-solid fa-head-side-cough-slash' => 'Head Side Cough Slash' ],
			[ 'fa fa-solid fa-head-side-mask' => 'Head Side Mask' ],
			[ 'fa fa-solid fa-head-side-virus' => 'Head Side Virus' ],
			[ 'fa fa-solid fa-heart' => 'Heart' ],
			[ 'fa fa-regular fa-heart' => 'Heart' ],
			[ 'fa fa-solid fa-heart-circle-bolt' => 'Heart Circle Bolt' ],
			[ 'fa fa-solid fa-heart-circle-check' => 'Heart Circle Check' ],
			[ 'fa fa-solid fa-heart-circle-exclamation' => 'Heart Circle Exclamation' ],
			[ 'fa fa-solid fa-heart-circle-minus' => 'Heart Circle Minus' ],
			[ 'fa fa-solid fa-heart-circle-plus' => 'Heart Circle Plus' ],
			[ 'fa fa-solid fa-heart-circle-xmark' => 'Heart Circle Xmark' ],
			[ 'fa fa-solid fa-heart-pulse' => 'Heart Pulse' ],
			[ 'fa fa-solid fa-hospital' => 'Hospital' ],
			[ 'fa fa-regular fa-hospital' => 'Hospital' ],
			[ 'fa fa-solid fa-hospital-user' => 'Hospital User' ],
			[ 'fa fa-solid fa-house-chimney-medical' => 'House Chimney Medical' ],
			[ 'fa fa-solid fa-house-medical' => 'House Medical' ],
			[ 'fa fa-solid fa-house-medical-circle-check' => 'House Medical Circle Check' ],
			[ 'fa fa-solid fa-house-medical-circle-exclamation' => 'House Medical Circle Exclamation' ],
			[ 'fa fa-solid fa-house-medical-circle-xmark' => 'House Medical Circle Xmark' ],
			[ 'fa fa-solid fa-house-medical-flag' => 'House Medical Flag' ],
			[ 'fa fa-solid fa-id-card-clip' => 'Id Card Clip' ],
			[ 'fa fa-solid fa-joint' => 'Joint' ],
			[ 'fa fa-solid fa-kit-medical' => 'Kit Medical' ],
			[ 'fa fa-solid fa-laptop-medical' => 'Laptop Medical' ],
			[ 'fa fa-solid fa-lungs' => 'Lungs' ],
			[ 'fa fa-solid fa-lungs-virus' => 'Lungs Virus' ],
			[ 'fa fa-solid fa-mask-face' => 'Mask Face' ],
			[ 'fa fa-solid fa-mask-ventilator' => 'Mask Ventilator' ],
			[ 'fa fa-solid fa-microscope' => 'Microscope' ],
			[ 'fa fa-solid fa-mortar-pestle' => 'Mortar Pestle' ],
			[ 'fa fa-solid fa-notes-medical' => 'Notes Medical' ],
			[ 'fa fa-solid fa-pager' => 'Pager' ],
			[ 'fa fa-solid fa-person-breastfeeding' => 'Person Breastfeeding' ],
			[ 'fa fa-solid fa-person-cane' => 'Person Cane' ],
			[ 'fa fa-solid fa-person-dots-from-line' => 'Person Dots From Line' ],
			[ 'fa fa-solid fa-person-half-dress' => 'Person Half Dress' ],
			[ 'fa fa-solid fa-pills' => 'Pills' ],
			[ 'fa fa-solid fa-plus' => 'Plus' ],
			[ 'fa fa-solid fa-poop' => 'Poop' ],
			[ 'fa fa-solid fa-prescription' => 'Prescription' ],
			[ 'fa fa-solid fa-prescription-bottle' => 'Prescription Bottle' ],
			[ 'fa fa-solid fa-prescription-bottle-medical' => 'Prescription Bottle Medical' ],
			[ 'fa fa-solid fa-pump-medical' => 'Pump Medical' ],
			[ 'fa fa-solid fa-radiation' => 'Radiation' ],
			[ 'fa fa-solid fa-receipt' => 'Receipt' ],
			[ 'fa fa-solid fa-shield-virus' => 'Shield Virus' ],
			[ 'fa fa-solid fa-skull' => 'Skull' ],
			[ 'fa fa-solid fa-skull-crossbones' => 'Skull Crossbones' ],
			[ 'fa fa-solid fa-smoking' => 'Smoking' ],
			[ 'fa fa-solid fa-square-h' => 'Square H' ],
			[ 'fa fa-solid fa-square-plus' => 'Square Plus' ],
			[ 'fa fa-regular fa-square-plus' => 'Square Plus' ],
			[ 'fa fa-solid fa-square-virus' => 'Square Virus' ],
			[ 'fa fa-solid fa-staff-snake' => 'Staff Snake' ],
			[ 'fa fa-solid fa-star-of-life' => 'Star Of Life' ],
			[ 'fa fa-solid fa-stethoscope' => 'Stethoscope' ],
			[ 'fa fa-solid fa-suitcase-medical' => 'Suitcase Medical' ],
			[ 'fa fa-solid fa-syringe' => 'Syringe' ],
			[ 'fa fa-solid fa-tablets' => 'Tablets' ],
			[ 'fa fa-solid fa-teeth' => 'Teeth' ],
			[ 'fa fa-solid fa-teeth-open' => 'Teeth Open' ],
			[ 'fa fa-solid fa-thermometer' => 'Thermometer' ],
			[ 'fa fa-solid fa-tooth' => 'Tooth' ],
			[ 'fa fa-solid fa-truck-droplet' => 'Truck Droplet' ],
			[ 'fa fa-solid fa-truck-medical' => 'Truck Medical' ],
			[ 'fa fa-solid fa-user-doctor' => 'User Doctor' ],
			[ 'fa fa-solid fa-user-nurse' => 'User Nurse' ],
			[ 'fa fa-solid fa-vial' => 'Vial' ],
			[ 'fa fa-solid fa-vial-circle-check' => 'Vial Circle Check' ],
			[ 'fa fa-solid fa-vial-virus' => 'Vial Virus' ],
			[ 'fa fa-solid fa-vials' => 'Vials' ],
			[ 'fa fa-solid fa-virus' => 'Virus' ],
			[ 'fa fa-solid fa-virus-covid' => 'Virus Covid' ],
			[ 'fa fa-solid fa-virus-covid-slash' => 'Virus Covid Slash' ],
			[ 'fa fa-solid fa-virus-slash' => 'Virus Slash' ],
			[ 'fa fa-solid fa-viruses' => 'Viruses' ],
			[ 'fa fa-solid fa-weight-scale' => 'Weight Scale' ],
			[ 'fa fa-solid fa-wheelchair' => 'Wheelchair' ],
			[ 'fa fa-solid fa-wheelchair-move' => 'Wheelchair Move' ],
			[ 'fa fa-solid fa-x-ray' => 'X Ray' ],
		],
		'Money' => [
			[ 'fa fa-solid fa-austral-sign' => 'Austral Sign' ],
			[ 'fa fa-solid fa-baht-sign' => 'Baht Sign' ],
			[ 'fa fa-solid fa-bangladeshi-taka-sign' => 'Bangladeshi Taka Sign' ],
			[ 'fa fa-brands fa-bitcoin' => 'Bitcoin' ],
			[ 'fa fa-solid fa-bitcoin-sign' => 'Bitcoin Sign' ],
			[ 'fa fa-solid fa-brazilian-real-sign' => 'Brazilian Real Sign' ],
			[ 'fa fa-brands fa-btc' => 'Btc' ],
			[ 'fa fa-solid fa-cash-register' => 'Cash Register' ],
			[ 'fa fa-solid fa-cedi-sign' => 'Cedi Sign' ],
			[ 'fa fa-solid fa-cent-sign' => 'Cent Sign' ],
			[ 'fa fa-solid fa-chart-line' => 'Chart Line' ],
			[ 'fa fa-solid fa-chart-pie' => 'Chart Pie' ],
			[ 'fa fa-solid fa-circle-dollar-to-slot' => 'Circle Dollar To Slot' ],
			[ 'fa fa-solid fa-coins' => 'Coins' ],
			[ 'fa fa-solid fa-colon-sign' => 'Colon Sign' ],
			[ 'fa fa-solid fa-comment-dollar' => 'Comment Dollar' ],
			[ 'fa fa-solid fa-comments-dollar' => 'Comments Dollar' ],
			[ 'fa fa-solid fa-credit-card' => 'Credit Card' ],
			[ 'fa fa-regular fa-credit-card' => 'Credit Card' ],
			[ 'fa fa-solid fa-cruzeiro-sign' => 'Cruzeiro Sign' ],
			[ 'fa fa-solid fa-dollar-sign' => 'Dollar Sign' ],
			[ 'fa fa-solid fa-dong-sign' => 'Dong Sign' ],
			[ 'fa fa-brands fa-ethereum' => 'Ethereum' ],
			[ 'fa fa-solid fa-euro-sign' => 'Euro Sign' ],
			[ 'fa fa-solid fa-file-invoice' => 'File Invoice' ],
			[ 'fa fa-solid fa-file-invoice-dollar' => 'File Invoice Dollar' ],
			[ 'fa fa-solid fa-florin-sign' => 'Florin Sign' ],
			[ 'fa fa-solid fa-franc-sign' => 'Franc Sign' ],
			[ 'fa fa-brands fa-gg' => 'Gg' ],
			[ 'fa fa-brands fa-gg-circle' => 'Gg Circle' ],
			[ 'fa fa-solid fa-guarani-sign' => 'Guarani Sign' ],
			[ 'fa fa-solid fa-hand-holding-dollar' => 'Hand Holding Dollar' ],
			[ 'fa fa-solid fa-hryvnia-sign' => 'Hryvnia Sign' ],
			[ 'fa fa-solid fa-indian-rupee-sign' => 'Indian Rupee Sign' ],
			[ 'fa fa-solid fa-kip-sign' => 'Kip Sign' ],
			[ 'fa fa-solid fa-landmark' => 'Landmark' ],
			[ 'fa fa-solid fa-lari-sign' => 'Lari Sign' ],
			[ 'fa fa-solid fa-lira-sign' => 'Lira Sign' ],
			[ 'fa fa-solid fa-litecoin-sign' => 'Litecoin Sign' ],
			[ 'fa fa-solid fa-manat-sign' => 'Manat Sign' ],
			[ 'fa fa-solid fa-mill-sign' => 'Mill Sign' ],
			[ 'fa fa-solid fa-money-bill' => 'Money Bill' ],
			[ 'fa fa-solid fa-money-bill-1' => 'Money Bill 1' ],
			[ 'fa fa-regular fa-money-bill-1' => 'Money Bill 1' ],
			[ 'fa fa-solid fa-money-bill-1-wave' => 'Money Bill 1 Wave' ],
			[ 'fa fa-solid fa-money-bill-transfer' => 'Money Bill Transfer' ],
			[ 'fa fa-solid fa-money-bill-trend-up' => 'Money Bill Trend Up' ],
			[ 'fa fa-solid fa-money-bill-wave' => 'Money Bill Wave' ],
			[ 'fa fa-solid fa-money-bill-wheat' => 'Money Bill Wheat' ],
			[ 'fa fa-solid fa-money-bills' => 'Money Bills' ],
			[ 'fa fa-solid fa-money-check' => 'Money Check' ],
			[ 'fa fa-solid fa-money-check-dollar' => 'Money Check Dollar' ],
			[ 'fa fa-solid fa-naira-sign' => 'Naira Sign' ],
			[ 'fa fa-solid fa-percent' => 'Percent' ],
			[ 'fa fa-solid fa-peseta-sign' => 'Peseta Sign' ],
			[ 'fa fa-solid fa-peso-sign' => 'Peso Sign' ],
			[ 'fa fa-solid fa-piggy-bank' => 'Piggy Bank' ],
			[ 'fa fa-solid fa-receipt' => 'Receipt' ],
			[ 'fa fa-solid fa-ruble-sign' => 'Ruble Sign' ],
			[ 'fa fa-solid fa-rupee-sign' => 'Rupee Sign' ],
			[ 'fa fa-solid fa-rupiah-sign' => 'Rupiah Sign' ],
			[ 'fa fa-solid fa-sack-dollar' => 'Sack Dollar' ],
			[ 'fa fa-solid fa-sack-xmark' => 'Sack Xmark' ],
			[ 'fa fa-solid fa-scale-balanced' => 'Scale Balanced' ],
			[ 'fa fa-solid fa-scale-unbalanced' => 'Scale Unbalanced' ],
			[ 'fa fa-solid fa-scale-unbalanced-flip' => 'Scale Unbalanced Flip' ],
			[ 'fa fa-solid fa-shekel-sign' => 'Shekel Sign' ],
			[ 'fa fa-solid fa-stamp' => 'Stamp' ],
			[ 'fa fa-solid fa-sterling-sign' => 'Sterling Sign' ],
			[ 'fa fa-solid fa-tenge-sign' => 'Tenge Sign' ],
			[ 'fa fa-solid fa-turkish-lira-sign' => 'Turkish Lira Sign' ],
			[ 'fa fa-solid fa-vault' => 'Vault' ],
			[ 'fa fa-solid fa-wallet' => 'Wallet' ],
			[ 'fa fa-solid fa-won-sign' => 'Won Sign' ],
			[ 'fa fa-solid fa-yen-sign' => 'Yen Sign' ],
		],
		'Moving' => [
			[ 'fa fa-solid fa-box-archive' => 'Box Archive' ],
			[ 'fa fa-solid fa-box-open' => 'Box Open' ],
			[ 'fa fa-solid fa-boxes-packing' => 'Boxes Packing' ],
			[ 'fa fa-solid fa-caravan' => 'Caravan' ],
			[ 'fa fa-solid fa-couch' => 'Couch' ],
			[ 'fa fa-solid fa-dolly' => 'Dolly' ],
			[ 'fa fa-solid fa-house-chimney' => 'House Chimney' ],
			[ 'fa fa-solid fa-people-carry-box' => 'People Carry Box' ],
			[ 'fa fa-solid fa-route' => 'Route' ],
			[ 'fa fa-solid fa-sign-hanging' => 'Sign Hanging' ],
			[ 'fa fa-solid fa-suitcase' => 'Suitcase' ],
			[ 'fa fa-solid fa-tape' => 'Tape' ],
			[ 'fa fa-solid fa-trailer' => 'Trailer' ],
			[ 'fa fa-solid fa-truck-moving' => 'Truck Moving' ],
			[ 'fa fa-solid fa-truck-ramp-box' => 'Truck Ramp Box' ],
			[ 'fa fa-solid fa-wine-glass' => 'Wine Glass' ],
		],
		'Music-audio' => [
			[ 'fa fa-solid fa-compact-disc' => 'Compact Disc' ],
			[ 'fa fa-solid fa-drum' => 'Drum' ],
			[ 'fa fa-solid fa-drum-steelpan' => 'Drum Steelpan' ],
			[ 'fa fa-solid fa-file-audio' => 'File Audio' ],
			[ 'fa fa-regular fa-file-audio' => 'File Audio' ],
			[ 'fa fa-solid fa-guitar' => 'Guitar' ],
			[ 'fa fa-solid fa-headphones' => 'Headphones' ],
			[ 'fa fa-solid fa-headphones-simple' => 'Headphones Simple' ],
			[ 'fa fa-solid fa-microphone' => 'Microphone' ],
			[ 'fa fa-solid fa-microphone-lines' => 'Microphone Lines' ],
			[ 'fa fa-solid fa-microphone-lines-slash' => 'Microphone Lines Slash' ],
			[ 'fa fa-solid fa-microphone-slash' => 'Microphone Slash' ],
			[ 'fa fa-solid fa-music' => 'Music' ],
			[ 'fa fa-brands fa-napster' => 'Napster' ],
			[ 'fa fa-solid fa-radio' => 'Radio' ],
			[ 'fa fa-solid fa-record-vinyl' => 'Record Vinyl' ],
			[ 'fa fa-solid fa-sliders' => 'Sliders' ],
			[ 'fa fa-brands fa-soundcloud' => 'Soundcloud' ],
			[ 'fa fa-brands fa-spotify' => 'Spotify' ],
			[ 'fa fa-solid fa-volume-high' => 'Volume High' ],
			[ 'fa fa-solid fa-volume-low' => 'Volume Low' ],
			[ 'fa fa-solid fa-volume-off' => 'Volume Off' ],
			[ 'fa fa-solid fa-volume-xmark' => 'Volume Xmark' ],
			[ 'fa fa-solid fa-wave-square' => 'Wave Square' ],
		],
		'Nature' => [
			[ 'fa fa-solid fa-binoculars' => 'Binoculars' ],
			[ 'fa fa-solid fa-bug' => 'Bug' ],
			[ 'fa fa-solid fa-bugs' => 'Bugs' ],
			[ 'fa fa-solid fa-cannabis' => 'Cannabis' ],
			[ 'fa fa-solid fa-cloud-sun' => 'Cloud Sun' ],
			[ 'fa fa-solid fa-clover' => 'Clover' ],
			[ 'fa fa-solid fa-feather' => 'Feather' ],
			[ 'fa fa-solid fa-feather-pointed' => 'Feather Pointed' ],
			[ 'fa fa-solid fa-fire' => 'Fire' ],
			[ 'fa fa-solid fa-frog' => 'Frog' ],
			[ 'fa fa-solid fa-icicles' => 'Icicles' ],
			[ 'fa fa-solid fa-leaf' => 'Leaf' ],
			[ 'fa fa-solid fa-locust' => 'Locust' ],
			[ 'fa fa-solid fa-mosquito' => 'Mosquito' ],
			[ 'fa fa-solid fa-mound' => 'Mound' ],
			[ 'fa fa-solid fa-mountain' => 'Mountain' ],
			[ 'fa fa-solid fa-mountain-city' => 'Mountain City' ],
			[ 'fa fa-solid fa-mountain-sun' => 'Mountain Sun' ],
			[ 'fa fa-solid fa-person-hiking' => 'Person Hiking' ],
			[ 'fa fa-solid fa-plant-wilt' => 'Plant Wilt' ],
			[ 'fa fa-solid fa-seedling' => 'Seedling' ],
			[ 'fa fa-solid fa-signs-post' => 'Signs Post' ],
			[ 'fa fa-solid fa-spider' => 'Spider' ],
			[ 'fa fa-solid fa-tree' => 'Tree' ],
			[ 'fa fa-solid fa-volcano' => 'Volcano' ],
			[ 'fa fa-solid fa-water' => 'Water' ],
			[ 'fa fa-solid fa-wind' => 'Wind' ],
			[ 'fa fa-solid fa-worm' => 'Worm' ],
		],
		'Numbers' => [],
		'Photos-images' => [
			[ 'fa fa-solid fa-bolt' => 'Bolt' ],
			[ 'fa fa-solid fa-bolt-lightning' => 'Bolt Lightning' ],
			[ 'fa fa-solid fa-camera' => 'Camera' ],
			[ 'fa fa-solid fa-camera-retro' => 'Camera Retro' ],
			[ 'fa fa-solid fa-camera-rotate' => 'Camera Rotate' ],
			[ 'fa fa-solid fa-chalkboard' => 'Chalkboard' ],
			[ 'fa fa-solid fa-circle-half-stroke' => 'Circle Half Stroke' ],
			[ 'fa fa-solid fa-clone' => 'Clone' ],
			[ 'fa fa-regular fa-clone' => 'Clone' ],
			[ 'fa fa-solid fa-droplet' => 'Droplet' ],
			[ 'fa fa-solid fa-eye' => 'Eye' ],
			[ 'fa fa-regular fa-eye' => 'Eye' ],
			[ 'fa fa-solid fa-eye-dropper' => 'Eye Dropper' ],
			[ 'fa fa-solid fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-regular fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-solid fa-file-image' => 'File Image' ],
			[ 'fa fa-regular fa-file-image' => 'File Image' ],
			[ 'fa fa-solid fa-film' => 'Film' ],
			[ 'fa fa-solid fa-id-badge' => 'Id Badge' ],
			[ 'fa fa-regular fa-id-badge' => 'Id Badge' ],
			[ 'fa fa-solid fa-id-card' => 'Id Card' ],
			[ 'fa fa-regular fa-id-card' => 'Id Card' ],
			[ 'fa fa-solid fa-image' => 'Image' ],
			[ 'fa fa-regular fa-image' => 'Image' ],
			[ 'fa fa-solid fa-image-portrait' => 'Image Portrait' ],
			[ 'fa fa-solid fa-images' => 'Images' ],
			[ 'fa fa-regular fa-images' => 'Images' ],
			[ 'fa fa-solid fa-minimize' => 'Minimize' ],
			[ 'fa fa-solid fa-panorama' => 'Panorama' ],
			[ 'fa fa-solid fa-photo-film' => 'Photo Film' ],
			[ 'fa fa-solid fa-sliders' => 'Sliders' ],
			[ 'fa fa-brands fa-unsplash' => 'Unsplash' ],
		],
		'Political' => [
			[ 'fa fa-solid fa-award' => 'Award' ],
			[ 'fa fa-solid fa-building-flag' => 'Building Flag' ],
			[ 'fa fa-solid fa-bullhorn' => 'Bullhorn' ],
			[ 'fa fa-solid fa-check-double' => 'Check Double' ],
			[ 'fa fa-solid fa-check-to-slot' => 'Check To Slot' ],
			[ 'fa fa-solid fa-circle-dollar-to-slot' => 'Circle Dollar To Slot' ],
			[ 'fa fa-solid fa-democrat' => 'Democrat' ],
			[ 'fa fa-solid fa-dove' => 'Dove' ],
			[ 'fa fa-solid fa-dumpster-fire' => 'Dumpster Fire' ],
			[ 'fa fa-solid fa-flag-usa' => 'Flag Usa' ],
			[ 'fa fa-solid fa-hand-fist' => 'Hand Fist' ],
			[ 'fa fa-solid fa-handshake' => 'Handshake' ],
			[ 'fa fa-regular fa-handshake' => 'Handshake' ],
			[ 'fa fa-solid fa-landmark-dome' => 'Landmark Dome' ],
			[ 'fa fa-solid fa-landmark-flag' => 'Landmark Flag' ],
			[ 'fa fa-solid fa-person-booth' => 'Person Booth' ],
			[ 'fa fa-solid fa-piggy-bank' => 'Piggy Bank' ],
			[ 'fa fa-solid fa-republican' => 'Republican' ],
			[ 'fa fa-solid fa-scale-balanced' => 'Scale Balanced' ],
			[ 'fa fa-solid fa-scale-unbalanced' => 'Scale Unbalanced' ],
			[ 'fa fa-solid fa-scale-unbalanced-flip' => 'Scale Unbalanced Flip' ],
		],
		'Punctuation-symbols' => [
			[ 'fa fa-solid fa-asterisk' => 'Asterisk' ],
			[ 'fa fa-solid fa-at' => 'At' ],
			[ 'fa fa-solid fa-check' => 'Check' ],
			[ 'fa fa-solid fa-check-double' => 'Check Double' ],
			[ 'fa fa-solid fa-circle-exclamation' => 'Circle Exclamation' ],
			[ 'fa fa-solid fa-circle-question' => 'Circle Question' ],
			[ 'fa fa-regular fa-circle-question' => 'Circle Question' ],
			[ 'fa fa-solid fa-equals' => 'Equals' ],
			[ 'fa fa-solid fa-exclamation' => 'Exclamation' ],
			[ 'fa fa-solid fa-greater-than' => 'Greater Than' ],
			[ 'fa fa-solid fa-hashtag' => 'Hashtag' ],
			[ 'fa fa-solid fa-less-than' => 'Less Than' ],
			[ 'fa fa-solid fa-minus' => 'Minus' ],
			[ 'fa fa-solid fa-percent' => 'Percent' ],
			[ 'fa fa-solid fa-plus' => 'Plus' ],
			[ 'fa fa-solid fa-question' => 'Question' ],
			[ 'fa fa-solid fa-quote-left' => 'Quote Left' ],
			[ 'fa fa-solid fa-quote-right' => 'Quote Right' ],
			[ 'fa fa-solid fa-section' => 'Section' ],
		],
		'Religion' => [
			[ 'fa fa-solid fa-ankh' => 'Ankh' ],
			[ 'fa fa-solid fa-atom' => 'Atom' ],
			[ 'fa fa-solid fa-bahai' => 'Bahai' ],
			[ 'fa fa-solid fa-book-bible' => 'Book Bible' ],
			[ 'fa fa-solid fa-book-journal-whills' => 'Book Journal Whills' ],
			[ 'fa fa-solid fa-book-quran' => 'Book Quran' ],
			[ 'fa fa-solid fa-book-tanakh' => 'Book Tanakh' ],
			[ 'fa fa-solid fa-church' => 'Church' ],
			[ 'fa fa-solid fa-cross' => 'Cross' ],
			[ 'fa fa-solid fa-dharmachakra' => 'Dharmachakra' ],
			[ 'fa fa-solid fa-dove' => 'Dove' ],
			[ 'fa fa-solid fa-gopuram' => 'Gopuram' ],
			[ 'fa fa-solid fa-hamsa' => 'Hamsa' ],
			[ 'fa fa-solid fa-hands-praying' => 'Hands Praying' ],
			[ 'fa fa-solid fa-hanukiah' => 'Hanukiah' ],
			[ 'fa fa-solid fa-jedi' => 'Jedi' ],
			[ 'fa fa-solid fa-kaaba' => 'Kaaba' ],
			[ 'fa fa-solid fa-khanda' => 'Khanda' ],
			[ 'fa fa-solid fa-menorah' => 'Menorah' ],
			[ 'fa fa-solid fa-mosque' => 'Mosque' ],
			[ 'fa fa-solid fa-om' => 'Om' ],
			[ 'fa fa-solid fa-peace' => 'Peace' ],
			[ 'fa fa-solid fa-person-praying' => 'Person Praying' ],
			[ 'fa fa-solid fa-place-of-worship' => 'Place Of Worship' ],
			[ 'fa fa-solid fa-scroll-torah' => 'Scroll Torah' ],
			[ 'fa fa-solid fa-spaghetti-monster-flying' => 'Spaghetti Monster Flying' ],
			[ 'fa fa-solid fa-star-and-crescent' => 'Star And Crescent' ],
			[ 'fa fa-solid fa-star-of-david' => 'Star Of David' ],
			[ 'fa fa-solid fa-synagogue' => 'Synagogue' ],
			[ 'fa fa-solid fa-torii-gate' => 'Torii Gate' ],
			[ 'fa fa-solid fa-vihara' => 'Vihara' ],
			[ 'fa fa-solid fa-yin-yang' => 'Yin Yang' ],
		],
		'Science' => [
			[ 'fa fa-solid fa-atom' => 'Atom' ],
			[ 'fa fa-solid fa-biohazard' => 'Biohazard' ],
			[ 'fa fa-solid fa-brain' => 'Brain' ],
			[ 'fa fa-solid fa-capsules' => 'Capsules' ],
			[ 'fa fa-solid fa-circle-radiation' => 'Circle Radiation' ],
			[ 'fa fa-solid fa-clipboard-check' => 'Clipboard Check' ],
			[ 'fa fa-solid fa-disease' => 'Disease' ],
			[ 'fa fa-solid fa-dna' => 'Dna' ],
			[ 'fa fa-solid fa-eye-dropper' => 'Eye Dropper' ],
			[ 'fa fa-solid fa-filter' => 'Filter' ],
			[ 'fa fa-solid fa-fire' => 'Fire' ],
			[ 'fa fa-solid fa-fire-flame-curved' => 'Fire Flame Curved' ],
			[ 'fa fa-solid fa-fire-flame-simple' => 'Fire Flame Simple' ],
			[ 'fa fa-solid fa-flask' => 'Flask' ],
			[ 'fa fa-solid fa-flask-vial' => 'Flask Vial' ],
			[ 'fa fa-solid fa-frog' => 'Frog' ],
			[ 'fa fa-solid fa-magnet' => 'Magnet' ],
			[ 'fa fa-solid fa-microscope' => 'Microscope' ],
			[ 'fa fa-solid fa-mortar-pestle' => 'Mortar Pestle' ],
			[ 'fa fa-solid fa-pills' => 'Pills' ],
			[ 'fa fa-solid fa-prescription-bottle' => 'Prescription Bottle' ],
			[ 'fa fa-solid fa-radiation' => 'Radiation' ],
			[ 'fa fa-solid fa-seedling' => 'Seedling' ],
			[ 'fa fa-solid fa-skull-crossbones' => 'Skull Crossbones' ],
			[ 'fa fa-solid fa-square-virus' => 'Square Virus' ],
			[ 'fa fa-solid fa-syringe' => 'Syringe' ],
			[ 'fa fa-solid fa-tablets' => 'Tablets' ],
			[ 'fa fa-solid fa-temperature-high' => 'Temperature High' ],
			[ 'fa fa-solid fa-temperature-low' => 'Temperature Low' ],
			[ 'fa fa-solid fa-vial' => 'Vial' ],
			[ 'fa fa-solid fa-vial-circle-check' => 'Vial Circle Check' ],
			[ 'fa fa-solid fa-vial-virus' => 'Vial Virus' ],
			[ 'fa fa-solid fa-vials' => 'Vials' ],
		],
		'Science-fiction' => [
			[ 'fa fa-solid fa-atom' => 'Atom' ],
			[ 'fa fa-solid fa-book-journal-whills' => 'Book Journal Whills' ],
			[ 'fa fa-solid fa-explosion' => 'Explosion' ],
			[ 'fa fa-brands fa-galactic-republic' => 'Galactic Republic' ],
			[ 'fa fa-brands fa-galactic-senate' => 'Galactic Senate' ],
			[ 'fa fa-solid fa-hand-spock' => 'Hand Spock' ],
			[ 'fa fa-regular fa-hand-spock' => 'Hand Spock' ],
			[ 'fa fa-solid fa-jedi' => 'Jedi' ],
			[ 'fa fa-brands fa-jedi-order' => 'Jedi Order' ],
			[ 'fa fa-brands fa-old-republic' => 'Old Republic' ],
			[ 'fa fa-solid fa-robot' => 'Robot' ],
			[ 'fa fa-solid fa-rocket' => 'Rocket' ],
			[ 'fa fa-brands fa-space-awesome' => 'Space Awesome' ],
			[ 'fa fa-solid fa-user-astronaut' => 'User Astronaut' ],
		],
		'Security' => [
			[ 'fa fa-solid fa-ban' => 'Ban' ],
			[ 'fa fa-solid fa-bug' => 'Bug' ],
			[ 'fa fa-solid fa-bug-slash' => 'Bug Slash' ],
			[ 'fa fa-solid fa-building-lock' => 'Building Lock' ],
			[ 'fa fa-solid fa-building-shield' => 'Building Shield' ],
			[ 'fa fa-solid fa-burst' => 'Burst' ],
			[ 'fa fa-solid fa-car-on' => 'Car On' ],
			[ 'fa fa-solid fa-door-closed' => 'Door Closed' ],
			[ 'fa fa-solid fa-door-open' => 'Door Open' ],
			[ 'fa fa-solid fa-dungeon' => 'Dungeon' ],
			[ 'fa fa-solid fa-explosion' => 'Explosion' ],
			[ 'fa fa-solid fa-eye' => 'Eye' ],
			[ 'fa fa-regular fa-eye' => 'Eye' ],
			[ 'fa fa-solid fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-regular fa-eye-slash' => 'Eye Slash' ],
			[ 'fa fa-solid fa-file-contract' => 'File Contract' ],
			[ 'fa fa-solid fa-file-shield' => 'File Shield' ],
			[ 'fa fa-solid fa-file-signature' => 'File Signature' ],
			[ 'fa fa-solid fa-fingerprint' => 'Fingerprint' ],
			[ 'fa fa-solid fa-gun' => 'Gun' ],
			[ 'fa fa-solid fa-handcuffs' => 'Handcuffs' ],
			[ 'fa fa-solid fa-hands-bound' => 'Hands Bound' ],
			[ 'fa fa-solid fa-hands-holding-child' => 'Hands Holding Child' ],
			[ 'fa fa-solid fa-hands-holding-circle' => 'Hands Holding Circle' ],
			[ 'fa fa-solid fa-house-fire' => 'House Fire' ],
			[ 'fa fa-solid fa-house-lock' => 'House Lock' ],
			[ 'fa fa-solid fa-id-badge' => 'Id Badge' ],
			[ 'fa fa-regular fa-id-badge' => 'Id Badge' ],
			[ 'fa fa-solid fa-id-card' => 'Id Card' ],
			[ 'fa fa-regular fa-id-card' => 'Id Card' ],
			[ 'fa fa-solid fa-id-card-clip' => 'Id Card Clip' ],
			[ 'fa fa-solid fa-key' => 'Key' ],
			[ 'fa fa-solid fa-land-mine-on' => 'Land Mine On' ],
			[ 'fa fa-solid fa-lock' => 'Lock' ],
			[ 'fa fa-solid fa-lock-open' => 'Lock Open' ],
			[ 'fa fa-solid fa-mars-and-venus-burst' => 'Mars And Venus Burst' ],
			[ 'fa fa-solid fa-mask' => 'Mask' ],
			[ 'fa fa-solid fa-passport' => 'Passport' ],
			[ 'fa fa-solid fa-people-pulling' => 'People Pulling' ],
			[ 'fa fa-solid fa-people-robbery' => 'People Robbery' ],
			[ 'fa fa-solid fa-person-burst' => 'Person Burst' ],
			[ 'fa fa-solid fa-person-dress-burst' => 'Person Dress Burst' ],
			[ 'fa fa-solid fa-person-falling-burst' => 'Person Falling Burst' ],
			[ 'fa fa-solid fa-person-harassing' => 'Person Harassing' ],
			[ 'fa fa-solid fa-person-military-pointing' => 'Person Military Pointing' ],
			[ 'fa fa-solid fa-person-military-rifle' => 'Person Military Rifle' ],
			[ 'fa fa-solid fa-person-military-to-person' => 'Person Military To Person' ],
			[ 'fa fa-solid fa-person-rifle' => 'Person Rifle' ],
			[ 'fa fa-solid fa-person-shelter' => 'Person Shelter' ],
			[ 'fa fa-solid fa-person-through-window' => 'Person Through Window' ],
			[ 'fa fa-solid fa-road-spikes' => 'Road Spikes' ],
			[ 'fa fa-solid fa-shield' => 'Shield' ],
			[ 'fa fa-solid fa-shield-cat' => 'Shield Cat' ],
			[ 'fa fa-solid fa-shield-dog' => 'Shield Dog' ],
			[ 'fa fa-solid fa-shield-halved' => 'Shield Halved' ],
			[ 'fa fa-solid fa-shield-heart' => 'Shield Heart' ],
			[ 'fa fa-solid fa-skull-crossbones' => 'Skull Crossbones' ],
			[ 'fa fa-solid fa-square-person-confined' => 'Square Person Confined' ],
			[ 'fa fa-solid fa-tower-observation' => 'Tower Observation' ],
			[ 'fa fa-solid fa-unlock' => 'Unlock' ],
			[ 'fa fa-solid fa-unlock-keyhole' => 'Unlock Keyhole' ],
			[ 'fa fa-solid fa-user-lock' => 'User Lock' ],
			[ 'fa fa-solid fa-user-secret' => 'User Secret' ],
			[ 'fa fa-solid fa-user-shield' => 'User Shield' ],
			[ 'fa fa-solid fa-vault' => 'Vault' ],
		],
		'Shapes' => [
			[ 'fa fa-solid fa-bookmark' => 'Bookmark' ],
			[ 'fa fa-regular fa-bookmark' => 'Bookmark' ],
			[ 'fa fa-solid fa-burst' => 'Burst' ],
			[ 'fa fa-solid fa-calendar' => 'Calendar' ],
			[ 'fa fa-regular fa-calendar' => 'Calendar' ],
			[ 'fa fa-solid fa-certificate' => 'Certificate' ],
			[ 'fa fa-solid fa-circle' => 'Circle' ],
			[ 'fa fa-regular fa-circle' => 'Circle' ],
			[ 'fa fa-solid fa-circle-half-stroke' => 'Circle Half Stroke' ],
			[ 'fa fa-solid fa-cloud' => 'Cloud' ],
			[ 'fa fa-solid fa-clover' => 'Clover' ],
			[ 'fa fa-solid fa-comment' => 'Comment' ],
			[ 'fa fa-regular fa-comment' => 'Comment' ],
			[ 'fa fa-solid fa-crown' => 'Crown' ],
			[ 'fa fa-solid fa-cubes-stacked' => 'Cubes Stacked' ],
			[ 'fa fa-solid fa-diamond' => 'Diamond' ],
			[ 'fa fa-solid fa-file' => 'File' ],
			[ 'fa fa-regular fa-file' => 'File' ],
			[ 'fa fa-solid fa-folder' => 'Folder' ],
			[ 'fa fa-regular fa-folder' => 'Folder' ],
			[ 'fa fa-solid fa-heart' => 'Heart' ],
			[ 'fa fa-regular fa-heart' => 'Heart' ],
			[ 'fa fa-solid fa-heart-crack' => 'Heart Crack' ],
			[ 'fa fa-solid fa-lines-leaning' => 'Lines Leaning' ],
			[ 'fa fa-solid fa-location-pin' => 'Location Pin' ],
			[ 'fa fa-solid fa-play' => 'Play' ],
			[ 'fa fa-solid fa-shapes' => 'Shapes' ],
			[ 'fa fa-solid fa-shield' => 'Shield' ],
			[ 'fa fa-solid fa-square' => 'Square' ],
			[ 'fa fa-regular fa-square' => 'Square' ],
			[ 'fa fa-solid fa-square-binary' => 'Square Binary' ],
			[ 'fa fa-solid fa-star' => 'Star' ],
			[ 'fa fa-regular fa-star' => 'Star' ],
			[ 'fa fa-solid fa-ticket-simple' => 'Ticket Simple' ],
		],
		'Shopping' => [
			[ 'fa fa-brands fa-alipay' => 'Alipay' ],
			[ 'fa fa-brands fa-amazon-pay' => 'Amazon Pay' ],
			[ 'fa fa-brands fa-apple-pay' => 'Apple Pay' ],
			[ 'fa fa-solid fa-bag-shopping' => 'Bag Shopping' ],
			[ 'fa fa-solid fa-barcode' => 'Barcode' ],
			[ 'fa fa-solid fa-basket-shopping' => 'Basket Shopping' ],
			[ 'fa fa-solid fa-bell' => 'Bell' ],
			[ 'fa fa-regular fa-bell' => 'Bell' ],
			[ 'fa fa-brands fa-bitcoin' => 'Bitcoin' ],
			[ 'fa fa-solid fa-bookmark' => 'Bookmark' ],
			[ 'fa fa-regular fa-bookmark' => 'Bookmark' ],
			[ 'fa fa-brands fa-btc' => 'Btc' ],
			[ 'fa fa-solid fa-bullhorn' => 'Bullhorn' ],
			[ 'fa fa-solid fa-camera' => 'Camera' ],
			[ 'fa fa-solid fa-camera-retro' => 'Camera Retro' ],
			[ 'fa fa-solid fa-cart-arrow-down' => 'Cart Arrow Down' ],
			[ 'fa fa-solid fa-cart-plus' => 'Cart Plus' ],
			[ 'fa fa-solid fa-cart-shopping' => 'Cart Shopping' ],
			[ 'fa fa-solid fa-cash-register' => 'Cash Register' ],
			[ 'fa fa-brands fa-cc-amazon-pay' => 'Cc Amazon Pay' ],
			[ 'fa fa-brands fa-cc-amex' => 'Cc Amex' ],
			[ 'fa fa-brands fa-cc-apple-pay' => 'Cc Apple Pay' ],
			[ 'fa fa-brands fa-cc-diners-club' => 'Cc Diners Club' ],
			[ 'fa fa-brands fa-cc-discover' => 'Cc Discover' ],
			[ 'fa fa-brands fa-cc-jcb' => 'Cc Jcb' ],
			[ 'fa fa-brands fa-cc-mastercard' => 'Cc Mastercard' ],
			[ 'fa fa-brands fa-cc-paypal' => 'Cc Paypal' ],
			[ 'fa fa-brands fa-cc-stripe' => 'Cc Stripe' ],
			[ 'fa fa-brands fa-cc-visa' => 'Cc Visa' ],
			[ 'fa fa-solid fa-certificate' => 'Certificate' ],
			[ 'fa fa-solid fa-credit-card' => 'Credit Card' ],
			[ 'fa fa-regular fa-credit-card' => 'Credit Card' ],
			[ 'fa fa-brands fa-ethereum' => 'Ethereum' ],
			[ 'fa fa-solid fa-gem' => 'Gem' ],
			[ 'fa fa-regular fa-gem' => 'Gem' ],
			[ 'fa fa-solid fa-gift' => 'Gift' ],
			[ 'fa fa-solid fa-gifts' => 'Gifts' ],
			[ 'fa fa-brands fa-google-pay' => 'Google Pay' ],
			[ 'fa fa-brands fa-google-wallet' => 'Google Wallet' ],
			[ 'fa fa-solid fa-handshake' => 'Handshake' ],
			[ 'fa fa-regular fa-handshake' => 'Handshake' ],
			[ 'fa fa-solid fa-heart' => 'Heart' ],
			[ 'fa fa-regular fa-heart' => 'Heart' ],
			[ 'fa fa-solid fa-key' => 'Key' ],
			[ 'fa fa-solid fa-money-check' => 'Money Check' ],
			[ 'fa fa-solid fa-money-check-dollar' => 'Money Check Dollar' ],
			[ 'fa fa-brands fa-nfc-directional' => 'Nfc Directional' ],
			[ 'fa fa-brands fa-nfc-symbol' => 'Nfc Symbol' ],
			[ 'fa fa-brands fa-paypal' => 'Paypal' ],
			[ 'fa fa-solid fa-person-booth' => 'Person Booth' ],
			[ 'fa fa-solid fa-receipt' => 'Receipt' ],
			[ 'fa fa-solid fa-shirt' => 'Shirt' ],
			[ 'fa fa-solid fa-shop' => 'Shop' ],
			[ 'fa fa-solid fa-shop-lock' => 'Shop Lock' ],
			[ 'fa fa-solid fa-shop-slash' => 'Shop Slash' ],
			[ 'fa fa-solid fa-star' => 'Star' ],
			[ 'fa fa-regular fa-star' => 'Star' ],
			[ 'fa fa-solid fa-store' => 'Store' ],
			[ 'fa fa-solid fa-store-slash' => 'Store Slash' ],
			[ 'fa fa-brands fa-stripe' => 'Stripe' ],
			[ 'fa fa-brands fa-stripe-s' => 'Stripe S' ],
			[ 'fa fa-solid fa-tag' => 'Tag' ],
			[ 'fa fa-solid fa-tags' => 'Tags' ],
			[ 'fa fa-solid fa-thumbs-down' => 'Thumbs Down' ],
			[ 'fa fa-regular fa-thumbs-down' => 'Thumbs Down' ],
			[ 'fa fa-solid fa-thumbs-up' => 'Thumbs Up' ],
			[ 'fa fa-regular fa-thumbs-up' => 'Thumbs Up' ],
			[ 'fa fa-solid fa-trophy' => 'Trophy' ],
			[ 'fa fa-solid fa-truck' => 'Truck' ],
			[ 'fa fa-solid fa-truck-fast' => 'Truck Fast' ],
		],
		'Social' => [
			[ 'fa fa-solid fa-bell' => 'Bell' ],
			[ 'fa fa-regular fa-bell' => 'Bell' ],
			[ 'fa fa-brands fa-bluesky' => 'Bluesky' ],
			[ 'fa fa-solid fa-cake-candles' => 'Cake Candles' ],
			[ 'fa fa-solid fa-camera' => 'Camera' ],
			[ 'fa fa-solid fa-circle-user' => 'Circle User' ],
			[ 'fa fa-regular fa-circle-user' => 'Circle User' ],
			[ 'fa fa-solid fa-comment' => 'Comment' ],
			[ 'fa fa-regular fa-comment' => 'Comment' ],
			[ 'fa fa-solid fa-envelope' => 'Envelope' ],
			[ 'fa fa-regular fa-envelope' => 'Envelope' ],
			[ 'fa fa-brands fa-facebook' => 'Facebook' ],
			[ 'fa fa-solid fa-hashtag' => 'Hashtag' ],
			[ 'fa fa-solid fa-heart' => 'Heart' ],
			[ 'fa fa-regular fa-heart' => 'Heart' ],
			[ 'fa fa-solid fa-icons' => 'Icons' ],
			[ 'fa fa-solid fa-image' => 'Image' ],
			[ 'fa fa-regular fa-image' => 'Image' ],
			[ 'fa fa-solid fa-images' => 'Images' ],
			[ 'fa fa-regular fa-images' => 'Images' ],
			[ 'fa fa-solid fa-location-dot' => 'Location Dot' ],
			[ 'fa fa-solid fa-location-pin' => 'Location Pin' ],
			[ 'fa fa-solid fa-message' => 'Message' ],
			[ 'fa fa-regular fa-message' => 'Message' ],
			[ 'fa fa-solid fa-photo-film' => 'Photo Film' ],
			[ 'fa fa-solid fa-retweet' => 'Retweet' ],
			[ 'fa fa-solid fa-share' => 'Share' ],
			[ 'fa fa-solid fa-share-from-square' => 'Share From Square' ],
			[ 'fa fa-regular fa-share-from-square' => 'Share From Square' ],
			[ 'fa fa-solid fa-share-nodes' => 'Share Nodes' ],
			[ 'fa fa-brands fa-square-bluesky' => 'Square Bluesky' ],
			[ 'fa fa-solid fa-square-poll-horizontal' => 'Square Poll Horizontal' ],
			[ 'fa fa-solid fa-square-poll-vertical' => 'Square Poll Vertical' ],
			[ 'fa fa-solid fa-square-share-nodes' => 'Square Share Nodes' ],
			[ 'fa fa-solid fa-star' => 'Star' ],
			[ 'fa fa-regular fa-star' => 'Star' ],
			[ 'fa fa-solid fa-thumbs-down' => 'Thumbs Down' ],
			[ 'fa fa-regular fa-thumbs-down' => 'Thumbs Down' ],
			[ 'fa fa-solid fa-thumbs-up' => 'Thumbs Up' ],
			[ 'fa fa-regular fa-thumbs-up' => 'Thumbs Up' ],
			[ 'fa fa-solid fa-thumbtack' => 'Thumbtack' ],
			[ 'fa fa-solid fa-thumbtack-slash' => 'Thumbtack Slash' ],
			[ 'fa fa-solid fa-user' => 'User' ],
			[ 'fa fa-regular fa-user' => 'User' ],
			[ 'fa fa-solid fa-user-group' => 'User Group' ],
			[ 'fa fa-solid fa-user-plus' => 'User Plus' ],
			[ 'fa fa-solid fa-users' => 'Users' ],
			[ 'fa fa-solid fa-video' => 'Video' ],
		],
		'Spinners' => [
			[ 'fa fa-solid fa-arrow-rotate-left' => 'Arrow Rotate Left' ],
			[ 'fa fa-solid fa-arrow-rotate-right' => 'Arrow Rotate Right' ],
			[ 'fa fa-solid fa-arrows-rotate' => 'Arrows Rotate' ],
			[ 'fa fa-solid fa-arrows-spin' => 'Arrows Spin' ],
			[ 'fa fa-solid fa-asterisk' => 'Asterisk' ],
			[ 'fa fa-solid fa-atom' => 'Atom' ],
			[ 'fa fa-solid fa-bahai' => 'Bahai' ],
			[ 'fa fa-solid fa-certificate' => 'Certificate' ],
			[ 'fa fa-solid fa-circle-notch' => 'Circle Notch' ],
			[ 'fa fa-solid fa-compact-disc' => 'Compact Disc' ],
			[ 'fa fa-solid fa-compass' => 'Compass' ],
			[ 'fa fa-regular fa-compass' => 'Compass' ],
			[ 'fa fa-solid fa-crosshairs' => 'Crosshairs' ],
			[ 'fa fa-solid fa-dharmachakra' => 'Dharmachakra' ],
			[ 'fa fa-solid fa-fan' => 'Fan' ],
			[ 'fa fa-solid fa-gear' => 'Gear' ],
			[ 'fa fa-solid fa-hurricane' => 'Hurricane' ],
			[ 'fa fa-solid fa-life-ring' => 'Life Ring' ],
			[ 'fa fa-regular fa-life-ring' => 'Life Ring' ],
			[ 'fa fa-solid fa-palette' => 'Palette' ],
			[ 'fa fa-solid fa-ring' => 'Ring' ],
			[ 'fa fa-solid fa-rotate' => 'Rotate' ],
			[ 'fa fa-solid fa-rotate-left' => 'Rotate Left' ],
			[ 'fa fa-solid fa-rotate-right' => 'Rotate Right' ],
			[ 'fa fa-solid fa-slash' => 'Slash' ],
			[ 'fa fa-solid fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-regular fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-solid fa-spinner' => 'Spinner' ],
			[ 'fa fa-solid fa-stroopwafel' => 'Stroopwafel' ],
			[ 'fa fa-solid fa-sun' => 'Sun' ],
			[ 'fa fa-regular fa-sun' => 'Sun' ],
			[ 'fa fa-solid fa-yin-yang' => 'Yin Yang' ],
		],
		'Sports-fitness' => [
			[ 'fa fa-solid fa-baseball' => 'Baseball' ],
			[ 'fa fa-solid fa-baseball-bat-ball' => 'Baseball Bat Ball' ],
			[ 'fa fa-solid fa-basketball' => 'Basketball' ],
			[ 'fa fa-solid fa-bicycle' => 'Bicycle' ],
			[ 'fa fa-solid fa-bowling-ball' => 'Bowling Ball' ],
			[ 'fa fa-solid fa-broom-ball' => 'Broom Ball' ],
			[ 'fa fa-solid fa-dumbbell' => 'Dumbbell' ],
			[ 'fa fa-solid fa-fire-flame-curved' => 'Fire Flame Curved' ],
			[ 'fa fa-solid fa-fire-flame-simple' => 'Fire Flame Simple' ],
			[ 'fa fa-solid fa-football' => 'Football' ],
			[ 'fa fa-solid fa-futbol' => 'Futbol' ],
			[ 'fa fa-regular fa-futbol' => 'Futbol' ],
			[ 'fa fa-solid fa-golf-ball-tee' => 'Golf Ball Tee' ],
			[ 'fa fa-solid fa-heart' => 'Heart' ],
			[ 'fa fa-regular fa-heart' => 'Heart' ],
			[ 'fa fa-solid fa-heart-pulse' => 'Heart Pulse' ],
			[ 'fa fa-solid fa-hockey-puck' => 'Hockey Puck' ],
			[ 'fa fa-solid fa-medal' => 'Medal' ],
			[ 'fa fa-solid fa-mound' => 'Mound' ],
			[ 'fa fa-solid fa-person-biking' => 'Person Biking' ],
			[ 'fa fa-solid fa-person-hiking' => 'Person Hiking' ],
			[ 'fa fa-solid fa-person-running' => 'Person Running' ],
			[ 'fa fa-solid fa-person-skating' => 'Person Skating' ],
			[ 'fa fa-solid fa-person-skiing' => 'Person Skiing' ],
			[ 'fa fa-solid fa-person-skiing-nordic' => 'Person Skiing Nordic' ],
			[ 'fa fa-solid fa-person-snowboarding' => 'Person Snowboarding' ],
			[ 'fa fa-solid fa-person-swimming' => 'Person Swimming' ],
			[ 'fa fa-solid fa-person-walking' => 'Person Walking' ],
			[ 'fa fa-solid fa-ranking-star' => 'Ranking Star' ],
			[ 'fa fa-solid fa-shoe-prints' => 'Shoe Prints' ],
			[ 'fa fa-solid fa-spa' => 'Spa' ],
			[ 'fa fa-solid fa-stopwatch-20' => 'Stopwatch 20' ],
			[ 'fa fa-solid fa-table-tennis-paddle-ball' => 'Table Tennis Paddle Ball' ],
			[ 'fa fa-solid fa-volleyball' => 'Volleyball' ],
			[ 'fa fa-solid fa-weight-hanging' => 'Weight Hanging' ],
		],
		'Text-formatting' => [
			[ 'fa fa-solid fa-align-center' => 'Align Center' ],
			[ 'fa fa-solid fa-align-justify' => 'Align Justify' ],
			[ 'fa fa-solid fa-align-left' => 'Align Left' ],
			[ 'fa fa-solid fa-align-right' => 'Align Right' ],
			[ 'fa fa-solid fa-bold' => 'Bold' ],
			[ 'fa fa-solid fa-border-all' => 'Border All' ],
			[ 'fa fa-solid fa-border-none' => 'Border None' ],
			[ 'fa fa-solid fa-border-top-left' => 'Border Top Left' ],
			[ 'fa fa-solid fa-check' => 'Check' ],
			[ 'fa fa-solid fa-check-double' => 'Check Double' ],
			[ 'fa fa-solid fa-circle-check' => 'Circle Check' ],
			[ 'fa fa-regular fa-circle-check' => 'Circle Check' ],
			[ 'fa fa-solid fa-filter-circle-xmark' => 'Filter Circle Xmark' ],
			[ 'fa fa-solid fa-font' => 'Font' ],
			[ 'fa fa-solid fa-heading' => 'Heading' ],
			[ 'fa fa-solid fa-highlighter' => 'Highlighter' ],
			[ 'fa fa-solid fa-i-cursor' => 'I Cursor' ],
			[ 'fa fa-solid fa-icons' => 'Icons' ],
			[ 'fa fa-solid fa-indent' => 'Indent' ],
			[ 'fa fa-solid fa-italic' => 'Italic' ],
			[ 'fa fa-solid fa-list' => 'List' ],
			[ 'fa fa-solid fa-list-check' => 'List Check' ],
			[ 'fa fa-solid fa-list-ol' => 'List Ol' ],
			[ 'fa fa-solid fa-list-ul' => 'List Ul' ],
			[ 'fa fa-solid fa-outdent' => 'Outdent' ],
			[ 'fa fa-solid fa-paragraph' => 'Paragraph' ],
			[ 'fa fa-solid fa-rectangle-list' => 'Rectangle List' ],
			[ 'fa fa-regular fa-rectangle-list' => 'Rectangle List' ],
			[ 'fa fa-solid fa-spell-check' => 'Spell Check' ],
			[ 'fa fa-solid fa-square-check' => 'Square Check' ],
			[ 'fa fa-regular fa-square-check' => 'Square Check' ],
			[ 'fa fa-solid fa-strikethrough' => 'Strikethrough' ],
			[ 'fa fa-solid fa-subscript' => 'Subscript' ],
			[ 'fa fa-solid fa-superscript' => 'Superscript' ],
			[ 'fa fa-solid fa-table' => 'Table' ],
			[ 'fa fa-solid fa-table-cells' => 'Table Cells' ],
			[ 'fa fa-solid fa-table-cells-column-lock' => 'Table Cells Column Lock' ],
			[ 'fa fa-solid fa-table-cells-large' => 'Table Cells Large' ],
			[ 'fa fa-solid fa-table-cells-row-lock' => 'Table Cells Row Lock' ],
			[ 'fa fa-solid fa-table-cells-row-unlock' => 'Table Cells Row Unlock' ],
			[ 'fa fa-solid fa-table-columns' => 'Table Columns' ],
			[ 'fa fa-solid fa-table-list' => 'Table List' ],
			[ 'fa fa-solid fa-text-height' => 'Text Height' ],
			[ 'fa fa-solid fa-text-slash' => 'Text Slash' ],
			[ 'fa fa-solid fa-text-width' => 'Text Width' ],
			[ 'fa fa-solid fa-underline' => 'Underline' ],
		],
		'Time' => [
			[ 'fa fa-solid fa-bell' => 'Bell' ],
			[ 'fa fa-regular fa-bell' => 'Bell' ],
			[ 'fa fa-solid fa-bell-slash' => 'Bell Slash' ],
			[ 'fa fa-regular fa-bell-slash' => 'Bell Slash' ],
			[ 'fa fa-solid fa-calendar' => 'Calendar' ],
			[ 'fa fa-regular fa-calendar' => 'Calendar' ],
			[ 'fa fa-solid fa-calendar-check' => 'Calendar Check' ],
			[ 'fa fa-regular fa-calendar-check' => 'Calendar Check' ],
			[ 'fa fa-solid fa-calendar-day' => 'Calendar Day' ],
			[ 'fa fa-solid fa-calendar-days' => 'Calendar Days' ],
			[ 'fa fa-regular fa-calendar-days' => 'Calendar Days' ],
			[ 'fa fa-solid fa-calendar-minus' => 'Calendar Minus' ],
			[ 'fa fa-regular fa-calendar-minus' => 'Calendar Minus' ],
			[ 'fa fa-solid fa-calendar-plus' => 'Calendar Plus' ],
			[ 'fa fa-regular fa-calendar-plus' => 'Calendar Plus' ],
			[ 'fa fa-solid fa-calendar-week' => 'Calendar Week' ],
			[ 'fa fa-solid fa-calendar-xmark' => 'Calendar Xmark' ],
			[ 'fa fa-regular fa-calendar-xmark' => 'Calendar Xmark' ],
			[ 'fa fa-solid fa-clock' => 'Clock' ],
			[ 'fa fa-regular fa-clock' => 'Clock' ],
			[ 'fa fa-solid fa-hourglass' => 'Hourglass' ],
			[ 'fa fa-regular fa-hourglass' => 'Hourglass' ],
			[ 'fa fa-solid fa-hourglass-end' => 'Hourglass End' ],
			[ 'fa fa-solid fa-hourglass-half' => 'Hourglass Half' ],
			[ 'fa fa-regular fa-hourglass-half' => 'Hourglass Half' ],
			[ 'fa fa-solid fa-hourglass-start' => 'Hourglass Start' ],
			[ 'fa fa-solid fa-stopwatch' => 'Stopwatch' ],
			[ 'fa fa-solid fa-stopwatch-20' => 'Stopwatch 20' ],
		],
		'Toggle' => [
			[ 'fa fa-solid fa-bullseye' => 'Bullseye' ],
			[ 'fa fa-solid fa-circle' => 'Circle' ],
			[ 'fa fa-regular fa-circle' => 'Circle' ],
			[ 'fa fa-solid fa-circle-check' => 'Circle Check' ],
			[ 'fa fa-regular fa-circle-check' => 'Circle Check' ],
			[ 'fa fa-solid fa-circle-dot' => 'Circle Dot' ],
			[ 'fa fa-regular fa-circle-dot' => 'Circle Dot' ],
			[ 'fa fa-solid fa-location-crosshairs' => 'Location Crosshairs' ],
			[ 'fa fa-solid fa-microphone' => 'Microphone' ],
			[ 'fa fa-solid fa-microphone-slash' => 'Microphone Slash' ],
			[ 'fa fa-solid fa-plane-up' => 'Plane Up' ],
			[ 'fa fa-solid fa-signal' => 'Signal' ],
			[ 'fa fa-solid fa-star' => 'Star' ],
			[ 'fa fa-regular fa-star' => 'Star' ],
			[ 'fa fa-solid fa-star-half' => 'Star Half' ],
			[ 'fa fa-regular fa-star-half' => 'Star Half' ],
			[ 'fa fa-solid fa-star-half-stroke' => 'Star Half Stroke' ],
			[ 'fa fa-regular fa-star-half-stroke' => 'Star Half Stroke' ],
			[ 'fa fa-solid fa-toggle-off' => 'Toggle Off' ],
			[ 'fa fa-solid fa-toggle-on' => 'Toggle On' ],
			[ 'fa fa-solid fa-wifi' => 'Wifi' ],
		],
		'Transportation' => [
			[ 'fa fa-brands fa-accessible-icon' => 'Accessible Icon' ],
			[ 'fa fa-solid fa-baby-carriage' => 'Baby Carriage' ],
			[ 'fa fa-solid fa-bicycle' => 'Bicycle' ],
			[ 'fa fa-solid fa-bus' => 'Bus' ],
			[ 'fa fa-solid fa-bus-simple' => 'Bus Simple' ],
			[ 'fa fa-solid fa-cable-car' => 'Cable Car' ],
			[ 'fa fa-solid fa-car' => 'Car' ],
			[ 'fa fa-solid fa-car-burst' => 'Car Burst' ],
			[ 'fa fa-solid fa-car-rear' => 'Car Rear' ],
			[ 'fa fa-solid fa-car-side' => 'Car Side' ],
			[ 'fa fa-solid fa-car-tunnel' => 'Car Tunnel' ],
			[ 'fa fa-solid fa-cart-shopping' => 'Cart Shopping' ],
			[ 'fa fa-solid fa-ferry' => 'Ferry' ],
			[ 'fa fa-solid fa-helicopter' => 'Helicopter' ],
			[ 'fa fa-solid fa-horse' => 'Horse' ],
			[ 'fa fa-solid fa-jet-fighter' => 'Jet Fighter' ],
			[ 'fa fa-solid fa-jet-fighter-up' => 'Jet Fighter Up' ],
			[ 'fa fa-solid fa-motorcycle' => 'Motorcycle' ],
			[ 'fa fa-solid fa-mound' => 'Mound' ],
			[ 'fa fa-solid fa-paper-plane' => 'Paper Plane' ],
			[ 'fa fa-regular fa-paper-plane' => 'Paper Plane' ],
			[ 'fa fa-solid fa-plane' => 'Plane' ],
			[ 'fa fa-solid fa-plane-slash' => 'Plane Slash' ],
			[ 'fa fa-solid fa-plane-up' => 'Plane Up' ],
			[ 'fa fa-solid fa-road' => 'Road' ],
			[ 'fa fa-solid fa-road-barrier' => 'Road Barrier' ],
			[ 'fa fa-solid fa-road-spikes' => 'Road Spikes' ],
			[ 'fa fa-solid fa-rocket' => 'Rocket' ],
			[ 'fa fa-solid fa-sailboat' => 'Sailboat' ],
			[ 'fa fa-solid fa-ship' => 'Ship' ],
			[ 'fa fa-solid fa-shuttle-space' => 'Shuttle Space' ],
			[ 'fa fa-solid fa-sleigh' => 'Sleigh' ],
			[ 'fa fa-solid fa-snowplow' => 'Snowplow' ],
			[ 'fa fa-solid fa-taxi' => 'Taxi' ],
			[ 'fa fa-solid fa-tractor' => 'Tractor' ],
			[ 'fa fa-solid fa-train' => 'Train' ],
			[ 'fa fa-solid fa-train-subway' => 'Train Subway' ],
			[ 'fa fa-solid fa-train-tram' => 'Train Tram' ],
			[ 'fa fa-solid fa-truck' => 'Truck' ],
			[ 'fa fa-solid fa-truck-arrow-right' => 'Truck Arrow Right' ],
			[ 'fa fa-solid fa-truck-droplet' => 'Truck Droplet' ],
			[ 'fa fa-solid fa-truck-field' => 'Truck Field' ],
			[ 'fa fa-solid fa-truck-field-un' => 'Truck Field Un' ],
			[ 'fa fa-solid fa-truck-front' => 'Truck Front' ],
			[ 'fa fa-solid fa-truck-medical' => 'Truck Medical' ],
			[ 'fa fa-solid fa-truck-monster' => 'Truck Monster' ],
			[ 'fa fa-solid fa-truck-pickup' => 'Truck Pickup' ],
			[ 'fa fa-solid fa-truck-plane' => 'Truck Plane' ],
			[ 'fa fa-solid fa-van-shuttle' => 'Van Shuttle' ],
			[ 'fa fa-solid fa-wheelchair' => 'Wheelchair' ],
			[ 'fa fa-solid fa-wheelchair-move' => 'Wheelchair Move' ],
		],
		'Travel-hotel' => [
			[ 'fa fa-solid fa-archway' => 'Archway' ],
			[ 'fa fa-solid fa-baby-carriage' => 'Baby Carriage' ],
			[ 'fa fa-solid fa-ban-smoking' => 'Ban Smoking' ],
			[ 'fa fa-solid fa-bath' => 'Bath' ],
			[ 'fa fa-solid fa-bed' => 'Bed' ],
			[ 'fa fa-solid fa-bell-concierge' => 'Bell Concierge' ],
			[ 'fa fa-solid fa-book-atlas' => 'Book Atlas' ],
			[ 'fa fa-solid fa-briefcase' => 'Briefcase' ],
			[ 'fa fa-solid fa-bus' => 'Bus' ],
			[ 'fa fa-solid fa-bus-simple' => 'Bus Simple' ],
			[ 'fa fa-solid fa-cable-car' => 'Cable Car' ],
			[ 'fa fa-solid fa-car' => 'Car' ],
			[ 'fa fa-solid fa-caravan' => 'Caravan' ],
			[ 'fa fa-solid fa-cart-flatbed-suitcase' => 'Cart Flatbed Suitcase' ],
			[ 'fa fa-solid fa-dice' => 'Dice' ],
			[ 'fa fa-solid fa-dice-five' => 'Dice Five' ],
			[ 'fa fa-solid fa-door-closed' => 'Door Closed' ],
			[ 'fa fa-solid fa-door-open' => 'Door Open' ],
			[ 'fa fa-solid fa-dumbbell' => 'Dumbbell' ],
			[ 'fa fa-solid fa-earth-africa' => 'Earth Africa' ],
			[ 'fa fa-solid fa-earth-americas' => 'Earth Americas' ],
			[ 'fa fa-solid fa-earth-asia' => 'Earth Asia' ],
			[ 'fa fa-solid fa-earth-europe' => 'Earth Europe' ],
			[ 'fa fa-solid fa-earth-oceania' => 'Earth Oceania' ],
			[ 'fa fa-solid fa-elevator' => 'Elevator' ],
			[ 'fa fa-solid fa-hot-tub-person' => 'Hot Tub Person' ],
			[ 'fa fa-solid fa-hotel' => 'Hotel' ],
			[ 'fa fa-solid fa-infinity' => 'Infinity' ],
			[ 'fa fa-solid fa-key' => 'Key' ],
			[ 'fa fa-solid fa-kitchen-set' => 'Kitchen Set' ],
			[ 'fa fa-solid fa-map' => 'Map' ],
			[ 'fa fa-regular fa-map' => 'Map' ],
			[ 'fa fa-solid fa-map-location' => 'Map Location' ],
			[ 'fa fa-solid fa-map-location-dot' => 'Map Location Dot' ],
			[ 'fa fa-solid fa-martini-glass' => 'Martini Glass' ],
			[ 'fa fa-solid fa-martini-glass-citrus' => 'Martini Glass Citrus' ],
			[ 'fa fa-solid fa-martini-glass-empty' => 'Martini Glass Empty' ],
			[ 'fa fa-solid fa-monument' => 'Monument' ],
			[ 'fa fa-solid fa-mountain-city' => 'Mountain City' ],
			[ 'fa fa-solid fa-mug-saucer' => 'Mug Saucer' ],
			[ 'fa fa-solid fa-passport' => 'Passport' ],
			[ 'fa fa-solid fa-person-swimming' => 'Person Swimming' ],
			[ 'fa fa-solid fa-person-walking-luggage' => 'Person Walking Luggage' ],
			[ 'fa fa-solid fa-plane' => 'Plane' ],
			[ 'fa fa-solid fa-plane-arrival' => 'Plane Arrival' ],
			[ 'fa fa-solid fa-plane-circle-check' => 'Plane Circle Check' ],
			[ 'fa fa-solid fa-plane-circle-exclamation' => 'Plane Circle Exclamation' ],
			[ 'fa fa-solid fa-plane-circle-xmark' => 'Plane Circle Xmark' ],
			[ 'fa fa-solid fa-plane-departure' => 'Plane Departure' ],
			[ 'fa fa-solid fa-plane-lock' => 'Plane Lock' ],
			[ 'fa fa-solid fa-plane-slash' => 'Plane Slash' ],
			[ 'fa fa-solid fa-plane-up' => 'Plane Up' ],
			[ 'fa fa-solid fa-shower' => 'Shower' ],
			[ 'fa fa-solid fa-smoking' => 'Smoking' ],
			[ 'fa fa-solid fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-regular fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-solid fa-spa' => 'Spa' ],
			[ 'fa fa-solid fa-stairs' => 'Stairs' ],
			[ 'fa fa-solid fa-suitcase' => 'Suitcase' ],
			[ 'fa fa-solid fa-suitcase-rolling' => 'Suitcase Rolling' ],
			[ 'fa fa-solid fa-taxi' => 'Taxi' ],
			[ 'fa fa-solid fa-toilet' => 'Toilet' ],
			[ 'fa fa-solid fa-toilet-paper' => 'Toilet Paper' ],
			[ 'fa fa-solid fa-train-tram' => 'Train Tram' ],
			[ 'fa fa-solid fa-tree-city' => 'Tree City' ],
			[ 'fa fa-solid fa-tv' => 'Tv' ],
			[ 'fa fa-solid fa-umbrella-beach' => 'Umbrella Beach' ],
			[ 'fa fa-solid fa-utensils' => 'Utensils' ],
			[ 'fa fa-solid fa-van-shuttle' => 'Van Shuttle' ],
			[ 'fa fa-solid fa-water-ladder' => 'Water Ladder' ],
			[ 'fa fa-solid fa-wheelchair' => 'Wheelchair' ],
			[ 'fa fa-solid fa-wheelchair-move' => 'Wheelchair Move' ],
			[ 'fa fa-solid fa-wifi' => 'Wifi' ],
			[ 'fa fa-solid fa-wine-glass' => 'Wine Glass' ],
			[ 'fa fa-solid fa-wine-glass-empty' => 'Wine Glass Empty' ],
		],
		'Users-people' => [
			[ 'fa fa-brands fa-accessible-icon' => 'Accessible Icon' ],
			[ 'fa fa-solid fa-address-book' => 'Address Book' ],
			[ 'fa fa-regular fa-address-book' => 'Address Book' ],
			[ 'fa fa-solid fa-address-card' => 'Address Card' ],
			[ 'fa fa-regular fa-address-card' => 'Address Card' ],
			[ 'fa fa-solid fa-arrows-down-to-people' => 'Arrows Down To People' ],
			[ 'fa fa-solid fa-baby' => 'Baby' ],
			[ 'fa fa-solid fa-bed' => 'Bed' ],
			[ 'fa fa-solid fa-chalkboard-user' => 'Chalkboard User' ],
			[ 'fa fa-solid fa-child' => 'Child' ],
			[ 'fa fa-solid fa-child-dress' => 'Child Dress' ],
			[ 'fa fa-solid fa-child-reaching' => 'Child Reaching' ],
			[ 'fa fa-solid fa-children' => 'Children' ],
			[ 'fa fa-solid fa-circle-user' => 'Circle User' ],
			[ 'fa fa-regular fa-circle-user' => 'Circle User' ],
			[ 'fa fa-solid fa-clipboard-user' => 'Clipboard User' ],
			[ 'fa fa-solid fa-elevator' => 'Elevator' ],
			[ 'fa fa-solid fa-face-frown' => 'Face Frown' ],
			[ 'fa fa-regular fa-face-frown' => 'Face Frown' ],
			[ 'fa fa-solid fa-face-meh' => 'Face Meh' ],
			[ 'fa fa-regular fa-face-meh' => 'Face Meh' ],
			[ 'fa fa-solid fa-face-smile' => 'Face Smile' ],
			[ 'fa fa-regular fa-face-smile' => 'Face Smile' ],
			[ 'fa fa-solid fa-head-side-cough' => 'Head Side Cough' ],
			[ 'fa fa-solid fa-head-side-cough-slash' => 'Head Side Cough Slash' ],
			[ 'fa fa-solid fa-head-side-mask' => 'Head Side Mask' ],
			[ 'fa fa-solid fa-head-side-virus' => 'Head Side Virus' ],
			[ 'fa fa-solid fa-hospital-user' => 'Hospital User' ],
			[ 'fa fa-solid fa-hot-tub-person' => 'Hot Tub Person' ],
			[ 'fa fa-solid fa-house-chimney-user' => 'House Chimney User' ],
			[ 'fa fa-solid fa-house-user' => 'House User' ],
			[ 'fa fa-solid fa-id-badge' => 'Id Badge' ],
			[ 'fa fa-regular fa-id-badge' => 'Id Badge' ],
			[ 'fa fa-solid fa-id-card' => 'Id Card' ],
			[ 'fa fa-regular fa-id-card' => 'Id Card' ],
			[ 'fa fa-solid fa-id-card-clip' => 'Id Card Clip' ],
			[ 'fa fa-solid fa-image-portrait' => 'Image Portrait' ],
			[ 'fa fa-solid fa-mars-and-venus-burst' => 'Mars And Venus Burst' ],
			[ 'fa fa-solid fa-people-arrows' => 'People Arrows' ],
			[ 'fa fa-solid fa-people-carry-box' => 'People Carry Box' ],
			[ 'fa fa-solid fa-people-group' => 'People Group' ],
			[ 'fa fa-solid fa-people-line' => 'People Line' ],
			[ 'fa fa-solid fa-people-pulling' => 'People Pulling' ],
			[ 'fa fa-solid fa-people-robbery' => 'People Robbery' ],
			[ 'fa fa-solid fa-people-roof' => 'People Roof' ],
			[ 'fa fa-solid fa-person' => 'Person' ],
			[ 'fa fa-solid fa-person-arrow-down-to-line' => 'Person Arrow Down To Line' ],
			[ 'fa fa-solid fa-person-arrow-up-from-line' => 'Person Arrow Up From Line' ],
			[ 'fa fa-solid fa-person-biking' => 'Person Biking' ],
			[ 'fa fa-solid fa-person-booth' => 'Person Booth' ],
			[ 'fa fa-solid fa-person-breastfeeding' => 'Person Breastfeeding' ],
			[ 'fa fa-solid fa-person-burst' => 'Person Burst' ],
			[ 'fa fa-solid fa-person-cane' => 'Person Cane' ],
			[ 'fa fa-solid fa-person-chalkboard' => 'Person Chalkboard' ],
			[ 'fa fa-solid fa-person-circle-check' => 'Person Circle Check' ],
			[ 'fa fa-solid fa-person-circle-exclamation' => 'Person Circle Exclamation' ],
			[ 'fa fa-solid fa-person-circle-minus' => 'Person Circle Minus' ],
			[ 'fa fa-solid fa-person-circle-plus' => 'Person Circle Plus' ],
			[ 'fa fa-solid fa-person-circle-question' => 'Person Circle Question' ],
			[ 'fa fa-solid fa-person-circle-xmark' => 'Person Circle Xmark' ],
			[ 'fa fa-solid fa-person-digging' => 'Person Digging' ],
			[ 'fa fa-solid fa-person-dots-from-line' => 'Person Dots From Line' ],
			[ 'fa fa-solid fa-person-dress' => 'Person Dress' ],
			[ 'fa fa-solid fa-person-dress-burst' => 'Person Dress Burst' ],
			[ 'fa fa-solid fa-person-drowning' => 'Person Drowning' ],
			[ 'fa fa-solid fa-person-falling' => 'Person Falling' ],
			[ 'fa fa-solid fa-person-falling-burst' => 'Person Falling Burst' ],
			[ 'fa fa-solid fa-person-half-dress' => 'Person Half Dress' ],
			[ 'fa fa-solid fa-person-harassing' => 'Person Harassing' ],
			[ 'fa fa-solid fa-person-hiking' => 'Person Hiking' ],
			[ 'fa fa-solid fa-person-military-pointing' => 'Person Military Pointing' ],
			[ 'fa fa-solid fa-person-military-rifle' => 'Person Military Rifle' ],
			[ 'fa fa-solid fa-person-military-to-person' => 'Person Military To Person' ],
			[ 'fa fa-solid fa-person-praying' => 'Person Praying' ],
			[ 'fa fa-solid fa-person-pregnant' => 'Person Pregnant' ],
			[ 'fa fa-solid fa-person-rays' => 'Person Rays' ],
			[ 'fa fa-solid fa-person-rifle' => 'Person Rifle' ],
			[ 'fa fa-solid fa-person-running' => 'Person Running' ],
			[ 'fa fa-solid fa-person-shelter' => 'Person Shelter' ],
			[ 'fa fa-solid fa-person-skating' => 'Person Skating' ],
			[ 'fa fa-solid fa-person-skiing' => 'Person Skiing' ],
			[ 'fa fa-solid fa-person-skiing-nordic' => 'Person Skiing Nordic' ],
			[ 'fa fa-solid fa-person-snowboarding' => 'Person Snowboarding' ],
			[ 'fa fa-solid fa-person-swimming' => 'Person Swimming' ],
			[ 'fa fa-solid fa-person-through-window' => 'Person Through Window' ],
			[ 'fa fa-solid fa-person-walking' => 'Person Walking' ],
			[ 'fa fa-solid fa-person-walking-arrow-loop-left' => 'Person Walking Arrow Loop Left' ],
			[ 'fa fa-solid fa-person-walking-arrow-right' => 'Person Walking Arrow Right' ],
			[ 'fa fa-solid fa-person-walking-dashed-line-arrow-right' => 'Person Walking Dashed Line Arrow Right' ],
			[ 'fa fa-solid fa-person-walking-luggage' => 'Person Walking Luggage' ],
			[ 'fa fa-solid fa-person-walking-with-cane' => 'Person Walking With Cane' ],
			[ 'fa fa-solid fa-poo' => 'Poo' ],
			[ 'fa fa-solid fa-restroom' => 'Restroom' ],
			[ 'fa fa-solid fa-skull' => 'Skull' ],
			[ 'fa fa-solid fa-square-person-confined' => 'Square Person Confined' ],
			[ 'fa fa-solid fa-street-view' => 'Street View' ],
			[ 'fa fa-solid fa-user' => 'User' ],
			[ 'fa fa-regular fa-user' => 'User' ],
			[ 'fa fa-solid fa-user-astronaut' => 'User Astronaut' ],
			[ 'fa fa-solid fa-user-check' => 'User Check' ],
			[ 'fa fa-solid fa-user-clock' => 'User Clock' ],
			[ 'fa fa-solid fa-user-doctor' => 'User Doctor' ],
			[ 'fa fa-solid fa-user-gear' => 'User Gear' ],
			[ 'fa fa-solid fa-user-graduate' => 'User Graduate' ],
			[ 'fa fa-solid fa-user-group' => 'User Group' ],
			[ 'fa fa-solid fa-user-injured' => 'User Injured' ],
			[ 'fa fa-solid fa-user-large' => 'User Large' ],
			[ 'fa fa-solid fa-user-large-slash' => 'User Large Slash' ],
			[ 'fa fa-solid fa-user-lock' => 'User Lock' ],
			[ 'fa fa-solid fa-user-minus' => 'User Minus' ],
			[ 'fa fa-solid fa-user-ninja' => 'User Ninja' ],
			[ 'fa fa-solid fa-user-nurse' => 'User Nurse' ],
			[ 'fa fa-solid fa-user-pen' => 'User Pen' ],
			[ 'fa fa-solid fa-user-plus' => 'User Plus' ],
			[ 'fa fa-solid fa-user-secret' => 'User Secret' ],
			[ 'fa fa-solid fa-user-shield' => 'User Shield' ],
			[ 'fa fa-solid fa-user-slash' => 'User Slash' ],
			[ 'fa fa-solid fa-user-tag' => 'User Tag' ],
			[ 'fa fa-solid fa-user-tie' => 'User Tie' ],
			[ 'fa fa-solid fa-user-xmark' => 'User Xmark' ],
			[ 'fa fa-solid fa-users' => 'Users' ],
			[ 'fa fa-solid fa-users-between-lines' => 'Users Between Lines' ],
			[ 'fa fa-solid fa-users-gear' => 'Users Gear' ],
			[ 'fa fa-solid fa-users-line' => 'Users Line' ],
			[ 'fa fa-solid fa-users-rays' => 'Users Rays' ],
			[ 'fa fa-solid fa-users-rectangle' => 'Users Rectangle' ],
			[ 'fa fa-solid fa-users-slash' => 'Users Slash' ],
			[ 'fa fa-solid fa-users-viewfinder' => 'Users Viewfinder' ],
			[ 'fa fa-solid fa-wheelchair' => 'Wheelchair' ],
			[ 'fa fa-solid fa-wheelchair-move' => 'Wheelchair Move' ],
		],
		'Weather' => [
			[ 'fa fa-solid fa-bolt' => 'Bolt' ],
			[ 'fa fa-solid fa-bolt-lightning' => 'Bolt Lightning' ],
			[ 'fa fa-solid fa-cloud' => 'Cloud' ],
			[ 'fa fa-solid fa-cloud-bolt' => 'Cloud Bolt' ],
			[ 'fa fa-solid fa-cloud-meatball' => 'Cloud Meatball' ],
			[ 'fa fa-solid fa-cloud-moon' => 'Cloud Moon' ],
			[ 'fa fa-solid fa-cloud-moon-rain' => 'Cloud Moon Rain' ],
			[ 'fa fa-solid fa-cloud-rain' => 'Cloud Rain' ],
			[ 'fa fa-solid fa-cloud-showers-heavy' => 'Cloud Showers Heavy' ],
			[ 'fa fa-solid fa-cloud-showers-water' => 'Cloud Showers Water' ],
			[ 'fa fa-solid fa-cloud-sun' => 'Cloud Sun' ],
			[ 'fa fa-solid fa-cloud-sun-rain' => 'Cloud Sun Rain' ],
			[ 'fa fa-solid fa-house-tsunami' => 'House Tsunami' ],
			[ 'fa fa-solid fa-hurricane' => 'Hurricane' ],
			[ 'fa fa-solid fa-icicles' => 'Icicles' ],
			[ 'fa fa-solid fa-meteor' => 'Meteor' ],
			[ 'fa fa-solid fa-moon' => 'Moon' ],
			[ 'fa fa-regular fa-moon' => 'Moon' ],
			[ 'fa fa-solid fa-poo-storm' => 'Poo Storm' ],
			[ 'fa fa-solid fa-rainbow' => 'Rainbow' ],
			[ 'fa fa-solid fa-smog' => 'Smog' ],
			[ 'fa fa-solid fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-regular fa-snowflake' => 'Snowflake' ],
			[ 'fa fa-solid fa-sun' => 'Sun' ],
			[ 'fa fa-regular fa-sun' => 'Sun' ],
			[ 'fa fa-solid fa-sun-plant-wilt' => 'Sun Plant Wilt' ],
			[ 'fa fa-solid fa-temperature-arrow-down' => 'Temperature Arrow Down' ],
			[ 'fa fa-solid fa-temperature-arrow-up' => 'Temperature Arrow Up' ],
			[ 'fa fa-solid fa-temperature-empty' => 'Temperature Empty' ],
			[ 'fa fa-solid fa-temperature-full' => 'Temperature Full' ],
			[ 'fa fa-solid fa-temperature-half' => 'Temperature Half' ],
			[ 'fa fa-solid fa-temperature-high' => 'Temperature High' ],
			[ 'fa fa-solid fa-temperature-low' => 'Temperature Low' ],
			[ 'fa fa-solid fa-temperature-quarter' => 'Temperature Quarter' ],
			[ 'fa fa-solid fa-temperature-three-quarters' => 'Temperature Three Quarters' ],
			[ 'fa fa-solid fa-tornado' => 'Tornado' ],
			[ 'fa fa-solid fa-umbrella' => 'Umbrella' ],
			[ 'fa fa-solid fa-volcano' => 'Volcano' ],
			[ 'fa fa-solid fa-water' => 'Water' ],
			[ 'fa fa-solid fa-wind' => 'Wind' ],
		],
		'Writing' => [
			[ 'fa fa-solid fa-blog' => 'Blog' ],
			[ 'fa fa-solid fa-book' => 'Book' ],
			[ 'fa fa-solid fa-book-bookmark' => 'Book Bookmark' ],
			[ 'fa fa-solid fa-bookmark' => 'Bookmark' ],
			[ 'fa fa-regular fa-bookmark' => 'Bookmark' ],
			[ 'fa fa-solid fa-box-archive' => 'Box Archive' ],
			[ 'fa fa-solid fa-envelope' => 'Envelope' ],
			[ 'fa fa-regular fa-envelope' => 'Envelope' ],
			[ 'fa fa-solid fa-envelope-open' => 'Envelope Open' ],
			[ 'fa fa-regular fa-envelope-open' => 'Envelope Open' ],
			[ 'fa fa-solid fa-eraser' => 'Eraser' ],
			[ 'fa fa-solid fa-file' => 'File' ],
			[ 'fa fa-regular fa-file' => 'File' ],
			[ 'fa fa-solid fa-file-lines' => 'File Lines' ],
			[ 'fa fa-regular fa-file-lines' => 'File Lines' ],
			[ 'fa fa-solid fa-folder' => 'Folder' ],
			[ 'fa fa-regular fa-folder' => 'Folder' ],
			[ 'fa fa-solid fa-folder-open' => 'Folder Open' ],
			[ 'fa fa-regular fa-folder-open' => 'Folder Open' ],
			[ 'fa fa-solid fa-keyboard' => 'Keyboard' ],
			[ 'fa fa-regular fa-keyboard' => 'Keyboard' ],
			[ 'fa fa-solid fa-newspaper' => 'Newspaper' ],
			[ 'fa fa-regular fa-newspaper' => 'Newspaper' ],
			[ 'fa fa-solid fa-notdef' => 'Notdef' ],
			[ 'fa fa-solid fa-note-sticky' => 'Note Sticky' ],
			[ 'fa fa-regular fa-note-sticky' => 'Note Sticky' ],
			[ 'fa fa-solid fa-paper-plane' => 'Paper Plane' ],
			[ 'fa fa-regular fa-paper-plane' => 'Paper Plane' ],
			[ 'fa fa-solid fa-paperclip' => 'Paperclip' ],
			[ 'fa fa-solid fa-paragraph' => 'Paragraph' ],
			[ 'fa fa-solid fa-pen' => 'Pen' ],
			[ 'fa fa-solid fa-pen-clip' => 'Pen Clip' ],
			[ 'fa fa-solid fa-pen-to-square' => 'Pen To Square' ],
			[ 'fa fa-regular fa-pen-to-square' => 'Pen To Square' ],
			[ 'fa fa-solid fa-pencil' => 'Pencil' ],
			[ 'fa fa-solid fa-quote-left' => 'Quote Left' ],
			[ 'fa fa-solid fa-quote-right' => 'Quote Right' ],
			[ 'fa fa-solid fa-signature' => 'Signature' ],
			[ 'fa fa-solid fa-square-pen' => 'Square Pen' ],
			[ 'fa fa-solid fa-thumbtack' => 'Thumbtack' ],
			[ 'fa fa-solid fa-thumbtack-slash' => 'Thumbtack Slash' ],
		],
		'Uncategorized' => [
			[ 'fas fa-adjust' => 'Adjust' ],
			[ 'fa fa-solid fa-1' => '1' ],
			[ 'fa fa-solid fa-2' => '2' ],
			[ 'fa fa-solid fa-3' => '3' ],
			[ 'fa fa-solid fa-4' => '4' ],
			[ 'fa fa-solid fa-5' => '5' ],
			[ 'fa fa-solid fa-6' => '6' ],
			[ 'fa fa-solid fa-7' => '7' ],
			[ 'fa fa-solid fa-8' => '8' ],
			[ 'fa fa-solid fa-9' => '9' ],
			[ 'fa fa-brands fa-42-group' => '42 Group' ],
			[ 'fa fa-brands fa-500px' => '500px' ],
			[ 'fa fa-brands fa-accusoft' => 'Accusoft' ],
			[ 'fa fa-brands fa-adn' => 'Adn' ],
			[ 'fa fa-brands fa-adversal' => 'Adversal' ],
			[ 'fa fa-brands fa-affiliatetheme' => 'Affiliatetheme' ],
			[ 'fa fa-brands fa-airbnb' => 'Airbnb' ],
			[ 'fa fa-brands fa-algolia' => 'Algolia' ],
			[ 'fa fa-brands fa-amazon' => 'Amazon' ],
			[ 'fa fa-brands fa-amilia' => 'Amilia' ],
			[ 'fa fa-brands fa-android' => 'Android' ],
			[ 'fa fa-brands fa-angellist' => 'Angellist' ],
			[ 'fa fa-brands fa-angrycreative' => 'Angrycreative' ],
			[ 'fa fa-brands fa-angular' => 'Angular' ],
			[ 'fa fa-brands fa-app-store' => 'App Store' ],
			[ 'fa fa-brands fa-app-store-ios' => 'App Store Ios' ],
			[ 'fa fa-brands fa-apper' => 'Apper' ],
			[ 'fa fa-brands fa-apple' => 'Apple' ],
			[ 'fa fa-brands fa-artstation' => 'Artstation' ],
			[ 'fa fa-brands fa-asymmetrik' => 'Asymmetrik' ],
			[ 'fa fa-brands fa-atlassian' => 'Atlassian' ],
			[ 'fa fa-brands fa-audible' => 'Audible' ],
			[ 'fa fa-brands fa-autoprefixer' => 'Autoprefixer' ],
			[ 'fa fa-brands fa-avianex' => 'Avianex' ],
			[ 'fa fa-brands fa-aviato' => 'Aviato' ],
			[ 'fa fa-brands fa-aws' => 'Aws' ],
			[ 'fa fa-brands fa-bandcamp' => 'Bandcamp' ],
			[ 'fa fa-brands fa-battle-net' => 'Battle Net' ],
			[ 'fa fa-brands fa-behance' => 'Behance' ],
			[ 'fa fa-brands fa-bilibili' => 'Bilibili' ],
			[ 'fa fa-brands fa-bimobject' => 'Bimobject' ],
			[ 'fa fa-brands fa-bitbucket' => 'Bitbucket' ],
			[ 'fa fa-brands fa-bity' => 'Bity' ],
			[ 'fa fa-brands fa-black-tie' => 'Black Tie' ],
			[ 'fa fa-brands fa-blackberry' => 'Blackberry' ],
			[ 'fa fa-brands fa-blogger' => 'Blogger' ],
			[ 'fa fa-brands fa-blogger-b' => 'Blogger B' ],
			[ 'fa fa-brands fa-bootstrap' => 'Bootstrap' ],
			[ 'fa fa-brands fa-bots' => 'Bots' ],
			[ 'fa fa-brands fa-brave' => 'Brave' ],
			[ 'fa fa-brands fa-brave-reverse' => 'Brave Reverse' ],
			[ 'fa fa-brands fa-buffer' => 'Buffer' ],
			[ 'fa fa-brands fa-buromobelexperte' => 'Buromobelexperte' ],
			[ 'fa fa-brands fa-buy-n-large' => 'Buy N Large' ],
			[ 'fa fa-brands fa-buysellads' => 'Buysellads' ],
			[ 'fa fa-brands fa-canadian-vanguarda-leaf' => 'Canadian Vanguarda Leaf' ],
			[ 'fa fa-brands fa-centercode' => 'Centercode' ],
			[ 'fa fa-brands fa-centos' => 'Centos' ],
			[ 'fa fa-brands fa-chrome' => 'Chrome' ],
			[ 'fa fa-brands fa-chromecast' => 'Chromecast' ],
			[ 'fa fa-brands fa-cloudflare' => 'Cloudflare' ],
			[ 'fa fa-brands fa-cloudscale' => 'Cloudscale' ],
			[ 'fa fa-brands fa-cloudsmith' => 'Cloudsmith' ],
			[ 'fa fa-brands fa-cloudversify' => 'Cloudversify' ],
			[ 'fa fa-brands fa-cmplid' => 'Cmplid' ],
			[ 'fa fa-brands fa-codepen' => 'Codepen' ],
			[ 'fa fa-brands fa-codiepie' => 'Codiepie' ],
			[ 'fa fa-brands fa-confluence' => 'Confluence' ],
			[ 'fa fa-brands fa-connectdevelop' => 'Connectdevelop' ],
			[ 'fa fa-brands fa-contao' => 'Contao' ],
			[ 'fa fa-brands fa-cotton-bureau' => 'Cotton Bureau' ],
			[ 'fa fa-brands fa-cpanel' => 'Cpanel' ],
			[ 'fa fa-brands fa-creative-commons' => 'Creative Commons' ],
			[ 'fa fa-brands fa-creative-commons-by' => 'Creative Commons By' ],
			[ 'fa fa-brands fa-creative-commons-nc' => 'Creative Commons Nc' ],
			[ 'fa fa-brands fa-creative-commons-nc-eu' => 'Creative Commons Nc Eu' ],
			[ 'fa fa-brands fa-creative-commons-nc-jp' => 'Creative Commons Nc Jp' ],
			[ 'fa fa-brands fa-creative-commons-nd' => 'Creative Commons Nd' ],
			[ 'fa fa-brands fa-creative-commons-pd' => 'Creative Commons Pd' ],
			[ 'fa fa-brands fa-creative-commons-pd-alt' => 'Creative Commons Pd Alt' ],
			[ 'fa fa-brands fa-creative-commons-remix' => 'Creative Commons Remix' ],
			[ 'fa fa-brands fa-creative-commons-sa' => 'Creative Commons Sa' ],
			[ 'fa fa-brands fa-creative-commons-sampling' => 'Creative Commons Sampling' ],
			[ 'fa fa-brands fa-creative-commons-sampling-plus' => 'Creative Commons Sampling Plus' ],
			[ 'fa fa-brands fa-creative-commons-share' => 'Creative Commons Share' ],
			[ 'fa fa-brands fa-creative-commons-zero' => 'Creative Commons Zero' ],
			[ 'fa fa-brands fa-css3' => 'Css3' ],
			[ 'fa fa-brands fa-css3-alt' => 'Css3 Alt' ],
			[ 'fa fa-brands fa-cuttlefish' => 'Cuttlefish' ],
			[ 'fa fa-brands fa-dailymotion' => 'Dailymotion' ],
			[ 'fa fa-brands fa-dart-lang' => 'Dart Lang' ],
			[ 'fa fa-brands fa-dashcube' => 'Dashcube' ],
			[ 'fa fa-brands fa-debian' => 'Debian' ],
			[ 'fa fa-brands fa-deezer' => 'Deezer' ],
			[ 'fa fa-brands fa-delicious' => 'Delicious' ],
			[ 'fa fa-brands fa-deploydog' => 'Deploydog' ],
			[ 'fa fa-brands fa-deskpro' => 'Deskpro' ],
			[ 'fa fa-brands fa-dev' => 'Dev' ],
			[ 'fa fa-brands fa-deviantart' => 'Deviantart' ],
			[ 'fa fa-brands fa-dhl' => 'Dhl' ],
			[ 'fa fa-brands fa-diaspora' => 'Diaspora' ],
			[ 'fa fa-brands fa-digg' => 'Digg' ],
			[ 'fa fa-brands fa-digital-ocean' => 'Digital Ocean' ],
			[ 'fa fa-brands fa-discord' => 'Discord' ],
			[ 'fa fa-brands fa-discourse' => 'Discourse' ],
			[ 'fa fa-brands fa-dochub' => 'Dochub' ],
			[ 'fa fa-brands fa-docker' => 'Docker' ],
			[ 'fa fa-brands fa-draft2digital' => 'Draft2digital' ],
			[ 'fa fa-brands fa-dribbble' => 'Dribbble' ],
			[ 'fa fa-brands fa-dropbox' => 'Dropbox' ],
			[ 'fa fa-brands fa-drupal' => 'Drupal' ],
			[ 'fa fa-brands fa-dyalog' => 'Dyalog' ],
			[ 'fa fa-brands fa-earlybirds' => 'Earlybirds' ],
			[ 'fa fa-brands fa-ebay' => 'Ebay' ],
			[ 'fa fa-brands fa-edge' => 'Edge' ],
			[ 'fa fa-brands fa-edge-legacy' => 'Edge Legacy' ],
			[ 'fa fa-brands fa-elementor' => 'Elementor' ],
			[ 'fa fa-brands fa-ello' => 'Ello' ],
			[ 'fa fa-brands fa-ember' => 'Ember' ],
			[ 'fa fa-brands fa-empire' => 'Empire' ],
			[ 'fa fa-brands fa-envira' => 'Envira' ],
			[ 'fa fa-brands fa-erlang' => 'Erlang' ],
			[ 'fa fa-brands fa-etsy' => 'Etsy' ],
			[ 'fa fa-brands fa-evernote' => 'Evernote' ],
			[ 'fa fa-brands fa-expeditedssl' => 'Expeditedssl' ],
			[ 'fa fa-brands fa-facebook-f' => 'Facebook F' ],
			[ 'fa fa-brands fa-facebook-messenger' => 'Facebook Messenger' ],
			[ 'fa fa-brands fa-fedex' => 'Fedex' ],
			[ 'fa fa-brands fa-fedora' => 'Fedora' ],
			[ 'fa fa-brands fa-figma' => 'Figma' ],
			[ 'fa fa-brands fa-files-pinwheel' => 'Files Pinwheel' ],
			[ 'fa fa-brands fa-firefox' => 'Firefox' ],
			[ 'fa fa-brands fa-firefox-browser' => 'Firefox Browser' ],
			[ 'fa fa-brands fa-first-order' => 'First Order' ],
			[ 'fa fa-brands fa-first-order-alt' => 'First Order Alt' ],
			[ 'fa fa-brands fa-firstdraft' => 'Firstdraft' ],
			[ 'fa fa-brands fa-flickr' => 'Flickr' ],
			[ 'fa fa-brands fa-flipboard' => 'Flipboard' ],
			[ 'fa fa-brands fa-flutter' => 'Flutter' ],
			[ 'fa fa-brands fa-fly' => 'Fly' ],
			[ 'fa fa-brands fa-fonticons' => 'Fonticons' ],
			[ 'fa fa-brands fa-fonticons-fi' => 'Fonticons Fi' ],
			[ 'fa fa-brands fa-fort-awesome' => 'Fort Awesome' ],
			[ 'fa fa-brands fa-fort-awesome-alt' => 'Fort Awesome Alt' ],
			[ 'fa fa-brands fa-forumbee' => 'Forumbee' ],
			[ 'fa fa-brands fa-foursquare' => 'Foursquare' ],
			[ 'fa fa-brands fa-free-code-camp' => 'Free Code Camp' ],
			[ 'fa fa-brands fa-freebsd' => 'Freebsd' ],
			[ 'fa fa-brands fa-fulcrum' => 'Fulcrum' ],
			[ 'fa fa-brands fa-get-pocket' => 'Get Pocket' ],
			[ 'fa fa-brands fa-git' => 'Git' ],
			[ 'fa fa-brands fa-git-alt' => 'Git Alt' ],
			[ 'fa fa-brands fa-github' => 'Github' ],
			[ 'fa fa-brands fa-github-alt' => 'Github Alt' ],
			[ 'fa fa-brands fa-gitkraken' => 'Gitkraken' ],
			[ 'fa fa-brands fa-gitlab' => 'Gitlab' ],
			[ 'fa fa-brands fa-gitter' => 'Gitter' ],
			[ 'fa fa-brands fa-glide' => 'Glide' ],
			[ 'fa fa-brands fa-glide-g' => 'Glide G' ],
			[ 'fa fa-brands fa-gofore' => 'Gofore' ],
			[ 'fa fa-brands fa-golang' => 'Golang' ],
			[ 'fa fa-brands fa-goodreads' => 'Goodreads' ],
			[ 'fa fa-brands fa-goodreads-g' => 'Goodreads G' ],
			[ 'fa fa-brands fa-google' => 'Google' ],
			[ 'fa fa-brands fa-google-drive' => 'Google Drive' ],
			[ 'fa fa-brands fa-google-play' => 'Google Play' ],
			[ 'fa fa-brands fa-google-plus' => 'Google Plus' ],
			[ 'fa fa-brands fa-google-plus-g' => 'Google Plus G' ],
			[ 'fa fa-brands fa-google-scholar' => 'Google Scholar' ],
			[ 'fa fa-brands fa-gratipay' => 'Gratipay' ],
			[ 'fa fa-brands fa-grav' => 'Grav' ],
			[ 'fa fa-brands fa-gripfire' => 'Gripfire' ],
			[ 'fa fa-brands fa-grunt' => 'Grunt' ],
			[ 'fa fa-brands fa-guilded' => 'Guilded' ],
			[ 'fa fa-brands fa-gulp' => 'Gulp' ],
			[ 'fa fa-brands fa-hacker-news' => 'Hacker News' ],
			[ 'fa fa-brands fa-hackerrank' => 'Hackerrank' ],
			[ 'fa fa-brands fa-hashnode' => 'Hashnode' ],
			[ 'fa fa-brands fa-hips' => 'Hips' ],
			[ 'fa fa-brands fa-hire-a-helper' => 'Hire A Helper' ],
			[ 'fa fa-brands fa-hive' => 'Hive' ],
			[ 'fa fa-brands fa-hooli' => 'Hooli' ],
			[ 'fa fa-brands fa-hornbill' => 'Hornbill' ],
			[ 'fa fa-brands fa-hotjar' => 'Hotjar' ],
			[ 'fa fa-brands fa-houzz' => 'Houzz' ],
			[ 'fa fa-brands fa-html5' => 'Html5' ],
			[ 'fa fa-brands fa-hubspot' => 'Hubspot' ],
			[ 'fa fa-brands fa-ideal' => 'Ideal' ],
			[ 'fa fa-brands fa-imdb' => 'Imdb' ],
			[ 'fa fa-brands fa-instagram' => 'Instagram' ],
			[ 'fa fa-brands fa-instalod' => 'Instalod' ],
			[ 'fa fa-brands fa-intercom' => 'Intercom' ],
			[ 'fa fa-brands fa-internet-explorer' => 'Internet Explorer' ],
			[ 'fa fa-brands fa-invision' => 'Invision' ],
			[ 'fa fa-brands fa-ioxhost' => 'Ioxhost' ],
			[ 'fa fa-brands fa-itch-io' => 'Itch Io' ],
			[ 'fa fa-brands fa-itunes' => 'Itunes' ],
			[ 'fa fa-brands fa-itunes-note' => 'Itunes Note' ],
			[ 'fa fa-brands fa-java' => 'Java' ],
			[ 'fa fa-brands fa-jenkins' => 'Jenkins' ],
			[ 'fa fa-brands fa-jira' => 'Jira' ],
			[ 'fa fa-brands fa-joget' => 'Joget' ],
			[ 'fa fa-brands fa-joomla' => 'Joomla' ],
			[ 'fa fa-brands fa-js' => 'Js' ],
			[ 'fa fa-brands fa-jsfiddle' => 'Jsfiddle' ],
			[ 'fa fa-brands fa-jxl' => 'Jxl' ],
			[ 'fa fa-brands fa-kaggle' => 'Kaggle' ],
			[ 'fa fa-brands fa-keybase' => 'Keybase' ],
			[ 'fa fa-brands fa-keycdn' => 'Keycdn' ],
			[ 'fa fa-brands fa-kickstarter' => 'Kickstarter' ],
			[ 'fa fa-brands fa-kickstarter-k' => 'Kickstarter K' ],
			[ 'fa fa-brands fa-korvue' => 'Korvue' ],
			[ 'fa fa-brands fa-laravel' => 'Laravel' ],
			[ 'fa fa-brands fa-lastfm' => 'Lastfm' ],
			[ 'fa fa-brands fa-leanpub' => 'Leanpub' ],
			[ 'fa fa-brands fa-less' => 'Less' ],
			[ 'fa fa-brands fa-letterboxd' => 'Letterboxd' ],
			[ 'fa fa-brands fa-line' => 'Line' ],
			[ 'fa fa-brands fa-linkedin' => 'Linkedin' ],
			[ 'fa fa-brands fa-linkedin-in' => 'Linkedin In' ],
			[ 'fa fa-brands fa-linode' => 'Linode' ],
			[ 'fa fa-brands fa-linux' => 'Linux' ],
			[ 'fa fa-brands fa-lyft' => 'Lyft' ],
			[ 'fa fa-brands fa-magento' => 'Magento' ],
			[ 'fa fa-brands fa-mailchimp' => 'Mailchimp' ],
			[ 'fa fa-brands fa-mandalorian' => 'Mandalorian' ],
			[ 'fa fa-brands fa-markdown' => 'Markdown' ],
			[ 'fa fa-brands fa-mastodon' => 'Mastodon' ],
			[ 'fa fa-brands fa-maxcdn' => 'Maxcdn' ],
			[ 'fa fa-brands fa-mdb' => 'Mdb' ],
			[ 'fa fa-brands fa-medapps' => 'Medapps' ],
			[ 'fa fa-brands fa-medium' => 'Medium' ],
			[ 'fa fa-brands fa-medrt' => 'Medrt' ],
			[ 'fa fa-brands fa-meetup' => 'Meetup' ],
			[ 'fa fa-brands fa-megaport' => 'Megaport' ],
			[ 'fa fa-brands fa-mendeley' => 'Mendeley' ],
			[ 'fa fa-brands fa-meta' => 'Meta' ],
			[ 'fa fa-brands fa-microblog' => 'Microblog' ],
			[ 'fa fa-brands fa-microsoft' => 'Microsoft' ],
			[ 'fa fa-brands fa-mintbit' => 'Mintbit' ],
			[ 'fa fa-brands fa-mix' => 'Mix' ],
			[ 'fa fa-brands fa-mixcloud' => 'Mixcloud' ],
			[ 'fa fa-brands fa-mixer' => 'Mixer' ],
			[ 'fa fa-brands fa-mizuni' => 'Mizuni' ],
			[ 'fa fa-brands fa-modx' => 'Modx' ],
			[ 'fa fa-brands fa-monero' => 'Monero' ],
			[ 'fa fa-solid fa-n' => 'N' ],
			[ 'fa fa-brands fa-neos' => 'Neos' ],
			[ 'fa fa-brands fa-nimblr' => 'Nimblr' ],
			[ 'fa fa-brands fa-node' => 'Node' ],
			[ 'fa fa-brands fa-node-js' => 'Node Js' ],
			[ 'fa fa-brands fa-npm' => 'Npm' ],
			[ 'fa fa-brands fa-ns8' => 'Ns8' ],
			[ 'fa fa-brands fa-nutritionix' => 'Nutritionix' ],
			[ 'fa fa-brands fa-octopus-deploy' => 'Octopus Deploy' ],
			[ 'fa fa-brands fa-odnoklassniki' => 'Odnoklassniki' ],
			[ 'fa fa-brands fa-odysee' => 'Odysee' ],
			[ 'fa fa-brands fa-opencart' => 'Opencart' ],
			[ 'fa fa-brands fa-openid' => 'Openid' ],
			[ 'fa fa-brands fa-opensuse' => 'Opensuse' ],
			[ 'fa fa-brands fa-opera' => 'Opera' ],
			[ 'fa fa-brands fa-optin-monster' => 'Optin Monster' ],
			[ 'fa fa-brands fa-orcid' => 'Orcid' ],
			[ 'fa fa-brands fa-osi' => 'Osi' ],
			[ 'fa fa-brands fa-padlet' => 'Padlet' ],
			[ 'fa fa-brands fa-page4' => 'Page4' ],
			[ 'fa fa-brands fa-pagelines' => 'Pagelines' ],
			[ 'fa fa-brands fa-palfed' => 'Palfed' ],
			[ 'fa fa-brands fa-patreon' => 'Patreon' ],
			[ 'fa fa-brands fa-perbyte' => 'Perbyte' ],
			[ 'fa fa-brands fa-periscope' => 'Periscope' ],
			[ 'fa fa-brands fa-phabricator' => 'Phabricator' ],
			[ 'fa fa-brands fa-phoenix-framework' => 'Phoenix Framework' ],
			[ 'fa fa-brands fa-phoenix-squadron' => 'Phoenix Squadron' ],
			[ 'fa fa-brands fa-php' => 'Php' ],
			[ 'fa fa-brands fa-pied-piper' => 'Pied Piper' ],
			[ 'fa fa-brands fa-pied-piper-alt' => 'Pied Piper Alt' ],
			[ 'fa fa-brands fa-pied-piper-hat' => 'Pied Piper Hat' ],
			[ 'fa fa-brands fa-pied-piper-pp' => 'Pied Piper Pp' ],
			[ 'fa fa-brands fa-pinterest' => 'Pinterest' ],
			[ 'fa fa-brands fa-pinterest-p' => 'Pinterest P' ],
			[ 'fa fa-brands fa-pix' => 'Pix' ],
			[ 'fa fa-brands fa-pixiv' => 'Pixiv' ],
			[ 'fa fa-brands fa-product-hunt' => 'Product Hunt' ],
			[ 'fa fa-brands fa-pushed' => 'Pushed' ],
			[ 'fa fa-brands fa-python' => 'Python' ],
			[ 'fa fa-brands fa-qq' => 'Qq' ],
			[ 'fa fa-brands fa-quinscape' => 'Quinscape' ],
			[ 'fa fa-brands fa-quora' => 'Quora' ],
			[ 'fa fa-brands fa-r-project' => 'R Project' ],
			[ 'fa fa-brands fa-raspberry-pi' => 'Raspberry Pi' ],
			[ 'fa fa-brands fa-ravelry' => 'Ravelry' ],
			[ 'fa fa-brands fa-react' => 'React' ],
			[ 'fa fa-brands fa-reacteurope' => 'Reacteurope' ],
			[ 'fa fa-brands fa-readme' => 'Readme' ],
			[ 'fa fa-brands fa-rebel' => 'Rebel' ],
			[ 'fa fa-brands fa-red-river' => 'Red River' ],
			[ 'fa fa-brands fa-reddit' => 'Reddit' ],
			[ 'fa fa-brands fa-reddit-alien' => 'Reddit Alien' ],
			[ 'fa fa-brands fa-redhat' => 'Redhat' ],
			[ 'fa fa-brands fa-renren' => 'Renren' ],
			[ 'fa fa-brands fa-replyd' => 'Replyd' ],
			[ 'fa fa-brands fa-researchgate' => 'Researchgate' ],
			[ 'fa fa-brands fa-resolving' => 'Resolving' ],
			[ 'fa fa-brands fa-rev' => 'Rev' ],
			[ 'fa fa-brands fa-rocketchat' => 'Rocketchat' ],
			[ 'fa fa-brands fa-rockrms' => 'Rockrms' ],
			[ 'fa fa-brands fa-rust' => 'Rust' ],
			[ 'fa fa-brands fa-safari' => 'Safari' ],
			[ 'fa fa-brands fa-salesforce' => 'Salesforce' ],
			[ 'fa fa-brands fa-sass' => 'Sass' ],
			[ 'fa fa-brands fa-schlix' => 'Schlix' ],
			[ 'fa fa-brands fa-screenpal' => 'Screenpal' ],
			[ 'fa fa-brands fa-scribd' => 'Scribd' ],
			[ 'fa fa-brands fa-searchengin' => 'Searchengin' ],
			[ 'fa fa-brands fa-sellcast' => 'Sellcast' ],
			[ 'fa fa-brands fa-sellsy' => 'Sellsy' ],
			[ 'fa fa-brands fa-servicestack' => 'Servicestack' ],
			[ 'fa fa-brands fa-shirtsinbulk' => 'Shirtsinbulk' ],
			[ 'fa fa-brands fa-shoelace' => 'Shoelace' ],
			[ 'fa fa-brands fa-shopify' => 'Shopify' ],
			[ 'fa fa-brands fa-shopware' => 'Shopware' ],
			[ 'fa fa-brands fa-signal-messenger' => 'Signal Messenger' ],
			[ 'fa fa-brands fa-simplybuilt' => 'Simplybuilt' ],
			[ 'fa fa-brands fa-sistrix' => 'Sistrix' ],
			[ 'fa fa-brands fa-sith' => 'Sith' ],
			[ 'fa fa-brands fa-sitrox' => 'Sitrox' ],
			[ 'fa fa-brands fa-sketch' => 'Sketch' ],
			[ 'fa fa-brands fa-skyatlas' => 'Skyatlas' ],
			[ 'fa fa-brands fa-skype' => 'Skype' ],
			[ 'fa fa-brands fa-slack' => 'Slack' ],
			[ 'fa fa-brands fa-slideshare' => 'Slideshare' ],
			[ 'fa fa-brands fa-snapchat' => 'Snapchat' ],
			[ 'fa fa-brands fa-sourcetree' => 'Sourcetree' ],
			[ 'fa fa-brands fa-speakap' => 'Speakap' ],
			[ 'fa fa-brands fa-speaker-deck' => 'Speaker Deck' ],
			[ 'fa fa-brands fa-square-behance' => 'Square Behance' ],
			[ 'fa fa-brands fa-square-dribbble' => 'Square Dribbble' ],
			[ 'fa fa-brands fa-square-facebook' => 'Square Facebook' ],
			[ 'fa fa-brands fa-square-font-awesome' => 'Square Font Awesome' ],
			[ 'fa fa-brands fa-square-font-awesome-stroke' => 'Square Font Awesome Stroke' ],
			[ 'fa fa-brands fa-square-git' => 'Square Git' ],
			[ 'fa fa-brands fa-square-github' => 'Square Github' ],
			[ 'fa fa-brands fa-square-gitlab' => 'Square Gitlab' ],
			[ 'fa fa-brands fa-square-google-plus' => 'Square Google Plus' ],
			[ 'fa fa-brands fa-square-hacker-news' => 'Square Hacker News' ],
			[ 'fa fa-brands fa-square-instagram' => 'Square Instagram' ],
			[ 'fa fa-brands fa-square-js' => 'Square Js' ],
			[ 'fa fa-brands fa-square-lastfm' => 'Square Lastfm' ],
			[ 'fa fa-brands fa-square-letterboxd' => 'Square Letterboxd' ],
			[ 'fa fa-brands fa-square-odnoklassniki' => 'Square Odnoklassniki' ],
			[ 'fa fa-brands fa-square-pied-piper' => 'Square Pied Piper' ],
			[ 'fa fa-brands fa-square-pinterest' => 'Square Pinterest' ],
			[ 'fa fa-brands fa-square-reddit' => 'Square Reddit' ],
			[ 'fa fa-brands fa-square-snapchat' => 'Square Snapchat' ],
			[ 'fa fa-brands fa-square-threads' => 'Square Threads' ],
			[ 'fa fa-brands fa-square-tumblr' => 'Square Tumblr' ],
			[ 'fa fa-brands fa-square-twitter' => 'Square Twitter' ],
			[ 'fa fa-brands fa-square-upwork' => 'Square Upwork' ],
			[ 'fa fa-brands fa-square-viadeo' => 'Square Viadeo' ],
			[ 'fa fa-brands fa-square-vimeo' => 'Square Vimeo' ],
			[ 'fa fa-brands fa-square-web-awesome' => 'Square Web Awesome' ],
			[ 'fa fa-brands fa-square-web-awesome-stroke' => 'Square Web Awesome Stroke' ],
			[ 'fa fa-brands fa-square-whatsapp' => 'Square Whatsapp' ],
			[ 'fa fa-brands fa-square-x-twitter' => 'Square X Twitter' ],
			[ 'fa fa-brands fa-square-xing' => 'Square Xing' ],
			[ 'fa fa-brands fa-square-youtube' => 'Square Youtube' ],
			[ 'fa fa-brands fa-squarespace' => 'Squarespace' ],
			[ 'fa fa-brands fa-stack-exchange' => 'Stack Exchange' ],
			[ 'fa fa-brands fa-stack-overflow' => 'Stack Overflow' ],
			[ 'fa fa-brands fa-stackpath' => 'Stackpath' ],
			[ 'fa fa-brands fa-staylinked' => 'Staylinked' ],
			[ 'fa fa-brands fa-sticker-mule' => 'Sticker Mule' ],
			[ 'fa fa-brands fa-strava' => 'Strava' ],
			[ 'fa fa-brands fa-stubber' => 'Stubber' ],
			[ 'fa fa-brands fa-studiovinari' => 'Studiovinari' ],
			[ 'fa fa-brands fa-stumbleupon' => 'Stumbleupon' ],
			[ 'fa fa-brands fa-stumbleupon-circle' => 'Stumbleupon Circle' ],
			[ 'fa fa-brands fa-superpowers' => 'Superpowers' ],
			[ 'fa fa-brands fa-supple' => 'Supple' ],
			[ 'fa fa-brands fa-suse' => 'Suse' ],
			[ 'fa fa-brands fa-swift' => 'Swift' ],
			[ 'fa fa-brands fa-symfony' => 'Symfony' ],
			[ 'fa fa-brands fa-teamspeak' => 'Teamspeak' ],
			[ 'fa fa-brands fa-telegram' => 'Telegram' ],
			[ 'fa fa-brands fa-tencent-weibo' => 'Tencent Weibo' ],
			[ 'fa fa-brands fa-the-red-yeti' => 'The Red Yeti' ],
			[ 'fa fa-brands fa-themeco' => 'Themeco' ],
			[ 'fa fa-brands fa-themeisle' => 'Themeisle' ],
			[ 'fa fa-brands fa-think-peaks' => 'Think Peaks' ],
			[ 'fa fa-brands fa-threads' => 'Threads' ],
			[ 'fa fa-brands fa-tiktok' => 'Tiktok' ],
			[ 'fa fa-brands fa-trade-federation' => 'Trade Federation' ],
			[ 'fa fa-brands fa-trello' => 'Trello' ],
			[ 'fa fa-brands fa-tumblr' => 'Tumblr' ],
			[ 'fa fa-brands fa-twitter' => 'Twitter' ],
			[ 'fa fa-brands fa-typo3' => 'Typo3' ],
			[ 'fa fa-brands fa-uber' => 'Uber' ],
			[ 'fa fa-brands fa-ubuntu' => 'Ubuntu' ],
			[ 'fa fa-brands fa-uikit' => 'Uikit' ],
			[ 'fa fa-brands fa-umbraco' => 'Umbraco' ],
			[ 'fa fa-brands fa-uncharted' => 'Uncharted' ],
			[ 'fa fa-brands fa-uniregistry' => 'Uniregistry' ],
			[ 'fa fa-brands fa-unity' => 'Unity' ],
			[ 'fa fa-brands fa-untappd' => 'Untappd' ],
			[ 'fa fa-brands fa-ups' => 'Ups' ],
			[ 'fa fa-brands fa-upwork' => 'Upwork' ],
			[ 'fa fa-brands fa-usb' => 'Usb' ],
			[ 'fa fa-brands fa-usps' => 'Usps' ],
			[ 'fa fa-brands fa-ussunnah' => 'Ussunnah' ],
			[ 'fa fa-brands fa-vaadin' => 'Vaadin' ],
			[ 'fa fa-brands fa-viacoin' => 'Viacoin' ],
			[ 'fa fa-brands fa-viadeo' => 'Viadeo' ],
			[ 'fa fa-brands fa-viber' => 'Viber' ],
			[ 'fa fa-brands fa-vimeo' => 'Vimeo' ],
			[ 'fa fa-brands fa-vimeo-v' => 'Vimeo V' ],
			[ 'fa fa-brands fa-vine' => 'Vine' ],
			[ 'fa fa-brands fa-vk' => 'Vk' ],
			[ 'fa fa-brands fa-vnv' => 'Vnv' ],
			[ 'fa fa-brands fa-vuejs' => 'Vuejs' ],
			[ 'fa fa-brands fa-watchman-monitoring' => 'Watchman Monitoring' ],
			[ 'fa fa-brands fa-waze' => 'Waze' ],
			[ 'fa fa-brands fa-webflow' => 'Webflow' ],
			[ 'fa fa-brands fa-weebly' => 'Weebly' ],
			[ 'fa fa-brands fa-weibo' => 'Weibo' ],
			[ 'fa fa-brands fa-weixin' => 'Weixin' ],
			[ 'fa fa-brands fa-whatsapp' => 'Whatsapp' ],
			[ 'fa fa-brands fa-whmcs' => 'Whmcs' ],
			[ 'fa fa-brands fa-wikipedia-w' => 'Wikipedia W' ],
			[ 'fa fa-brands fa-windows' => 'Windows' ],
			[ 'fa fa-brands fa-wirsindhandwerk' => 'Wirsindhandwerk' ],
			[ 'fa fa-brands fa-wix' => 'Wix' ],
			[ 'fa fa-brands fa-wodu' => 'Wodu' ],
			[ 'fa fa-brands fa-wolf-pack-battalion' => 'Wolf Pack Battalion' ],
			[ 'fa fa-brands fa-wordpress' => 'Wordpress' ],
			[ 'fa fa-brands fa-wordpress-simple' => 'Wordpress Simple' ],
			[ 'fa fa-brands fa-wpbeginner' => 'Wpbeginner' ],
			[ 'fa fa-brands fa-wpexplorer' => 'Wpexplorer' ],
			[ 'fa fa-brands fa-wpforms' => 'Wpforms' ],
			[ 'fa fa-brands fa-wpressr' => 'Wpressr' ],
			[ 'fa fa-brands fa-x-twitter' => 'X Twitter' ],
			[ 'fa fa-brands fa-xing' => 'Xing' ],
			[ 'fa fa-solid fa-y' => 'Y' ],
			[ 'fa fa-brands fa-y-combinator' => 'Y Combinator' ],
			[ 'fa fa-brands fa-yahoo' => 'Yahoo' ],
			[ 'fa fa-brands fa-yammer' => 'Yammer' ],
			[ 'fa fa-brands fa-yandex' => 'Yandex' ],
			[ 'fa fa-brands fa-yandex-international' => 'Yandex International' ],
			[ 'fa fa-brands fa-yarn' => 'Yarn' ],
			[ 'fa fa-brands fa-yelp' => 'Yelp' ],
			[ 'fa fa-brands fa-yoast' => 'Yoast' ],
			[ 'fa fa-brands fa-zhihu' => 'Zhihu' ],
		],
	];

	return array_merge( $icons, $fontawesome_icons );
}

add_filter( 'vc_iconpicker-type-openiconic', 'vc_iconpicker_type_openiconic' );

/**
 * Openicons icons from fontello.com
 *
 * @param array $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown.
 *
 * @return array - of icons for iconpicker, can be categorized, or not.
 * @since 4.4
 */
function vc_iconpicker_type_openiconic( $icons ) {
	$openiconic_icons = [
		[ 'vc-oi vc-oi-dial' => 'Dial' ],
		[ 'vc-oi vc-oi-pilcrow' => 'Pilcrow' ],
		[ 'vc-oi vc-oi-at' => 'At' ],
		[ 'vc-oi vc-oi-hash' => 'Hash' ],
		[ 'vc-oi vc-oi-key-inv' => 'Key-inv' ],
		[ 'vc-oi vc-oi-key' => 'Key' ],
		[ 'vc-oi vc-oi-chart-pie-alt' => 'Chart-pie-alt' ],
		[ 'vc-oi vc-oi-chart-pie' => 'Chart-pie' ],
		[ 'vc-oi vc-oi-chart-bar' => 'Chart-bar' ],
		[ 'vc-oi vc-oi-umbrella' => 'Umbrella' ],
		[ 'vc-oi vc-oi-moon-inv' => 'Moon-inv' ],
		[ 'vc-oi vc-oi-mobile' => 'Mobile' ],
		[ 'vc-oi vc-oi-cd' => 'Cd' ],
		[ 'vc-oi vc-oi-split' => 'Split' ],
		[ 'vc-oi vc-oi-exchange' => 'Exchange' ],
		[ 'vc-oi vc-oi-block' => 'Block' ],
		[ 'vc-oi vc-oi-resize-full' => 'Resize-full' ],
		[ 'vc-oi vc-oi-article-alt' => 'Article-alt' ],
		[ 'vc-oi vc-oi-article' => 'Article' ],
		[ 'vc-oi vc-oi-pencil-alt' => 'Pencil-alt' ],
		[ 'vc-oi vc-oi-undo' => 'Undo' ],
		[ 'vc-oi vc-oi-attach' => 'Attach' ],
		[ 'vc-oi vc-oi-link' => 'Link' ],
		[ 'vc-oi vc-oi-search' => 'Search' ],
		[ 'vc-oi vc-oi-mail' => 'Mail' ],
		[ 'vc-oi vc-oi-heart' => 'Heart' ],
		[ 'vc-oi vc-oi-comment' => 'Comment' ],
		[ 'vc-oi vc-oi-resize-full-alt' => 'Resize-full-alt' ],
		[ 'vc-oi vc-oi-lock' => 'Lock' ],
		[ 'vc-oi vc-oi-book-open' => 'Book-open' ],
		[ 'vc-oi vc-oi-arrow-curved' => 'Arrow-curved' ],
		[ 'vc-oi vc-oi-equalizer' => 'Equalizer' ],
		[ 'vc-oi vc-oi-heart-empty' => 'Heart-empty' ],
		[ 'vc-oi vc-oi-lock-empty' => 'Lock-empty' ],
		[ 'vc-oi vc-oi-comment-inv' => 'Comment-inv' ],
		[ 'vc-oi vc-oi-folder' => 'Folder' ],
		[ 'vc-oi vc-oi-resize-small' => 'Resize-small' ],
		[ 'vc-oi vc-oi-play' => 'Play' ],
		[ 'vc-oi vc-oi-cursor' => 'Cursor' ],
		[ 'vc-oi vc-oi-aperture' => 'Aperture' ],
		[ 'vc-oi vc-oi-play-circle2' => 'Play-circle2' ],
		[ 'vc-oi vc-oi-resize-small-alt' => 'Resize-small-alt' ],
		[ 'vc-oi vc-oi-folder-empty' => 'Folder-empty' ],
		[ 'vc-oi vc-oi-comment-alt' => 'Comment-alt' ],
		[ 'vc-oi vc-oi-lock-open' => 'Lock-open' ],
		[ 'vc-oi vc-oi-star' => 'Star' ],
		[ 'vc-oi vc-oi-user' => 'User' ],
		[ 'vc-oi vc-oi-lock-open-empty' => 'Lock-open-empty' ],
		[ 'vc-oi vc-oi-box' => 'Box' ],
		[ 'vc-oi vc-oi-resize-vertical' => 'Resize-vertical' ],
		[ 'vc-oi vc-oi-stop' => 'Stop' ],
		[ 'vc-oi vc-oi-aperture-alt' => 'Aperture-alt' ],
		[ 'vc-oi vc-oi-book' => 'Book' ],
		[ 'vc-oi vc-oi-steering-wheel' => 'Steering-wheel' ],
		[ 'vc-oi vc-oi-pause' => 'Pause' ],
		[ 'vc-oi vc-oi-to-start' => 'To-start' ],
		[ 'vc-oi vc-oi-move' => 'Move' ],
		[ 'vc-oi vc-oi-resize-horizontal' => 'Resize-horizontal' ],
		[ 'vc-oi vc-oi-rss-alt' => 'Rss-alt' ],
		[ 'vc-oi vc-oi-comment-alt2' => 'Comment-alt2' ],
		[ 'vc-oi vc-oi-rss' => 'Rss' ],
		[ 'vc-oi vc-oi-comment-inv-alt' => 'Comment-inv-alt' ],
		[ 'vc-oi vc-oi-comment-inv-alt2' => 'Comment-inv-alt2' ],
		[ 'vc-oi vc-oi-eye' => 'Eye' ],
		[ 'vc-oi vc-oi-pin' => 'Pin' ],
		[ 'vc-oi vc-oi-video' => 'Video' ],
		[ 'vc-oi vc-oi-picture' => 'Picture' ],
		[ 'vc-oi vc-oi-camera' => 'Camera' ],
		[ 'vc-oi vc-oi-tag' => 'Tag' ],
		[ 'vc-oi vc-oi-chat' => 'Chat' ],
		[ 'vc-oi vc-oi-cog' => 'Cog' ],
		[ 'vc-oi vc-oi-popup' => 'Popup' ],
		[ 'vc-oi vc-oi-to-end' => 'To-end' ],
		[ 'vc-oi vc-oi-book-alt' => 'Book-alt' ],
		[ 'vc-oi vc-oi-brush' => 'Brush' ],
		[ 'vc-oi vc-oi-eject' => 'Eject' ],
		[ 'vc-oi vc-oi-down' => 'Down' ],
		[ 'vc-oi vc-oi-wrench' => 'Wrench' ],
		[ 'vc-oi vc-oi-chat-inv' => 'Chat-inv' ],
		[ 'vc-oi vc-oi-tag-empty' => 'Tag-empty' ],
		[ 'vc-oi vc-oi-ok' => 'Ok' ],
		[ 'vc-oi vc-oi-ok-circle' => 'Ok-circle' ],
		[ 'vc-oi vc-oi-download' => 'Download' ],
		[ 'vc-oi vc-oi-location' => 'Location' ],
		[ 'vc-oi vc-oi-share' => 'Share' ],
		[ 'vc-oi vc-oi-left' => 'Left' ],
		[ 'vc-oi vc-oi-target' => 'Target' ],
		[ 'vc-oi vc-oi-brush-alt' => 'Brush-alt' ],
		[ 'vc-oi vc-oi-cancel' => 'Cancel' ],
		[ 'vc-oi vc-oi-upload' => 'Upload' ],
		[ 'vc-oi vc-oi-location-inv' => 'Location-inv' ],
		[ 'vc-oi vc-oi-calendar' => 'Calendar' ],
		[ 'vc-oi vc-oi-right' => 'Right' ],
		[ 'vc-oi vc-oi-signal' => 'Signal' ],
		[ 'vc-oi vc-oi-eyedropper' => 'Eyedropper' ],
		[ 'vc-oi vc-oi-layers' => 'Layers' ],
		[ 'vc-oi vc-oi-award' => 'Award' ],
		[ 'vc-oi vc-oi-up' => 'Up' ],
		[ 'vc-oi vc-oi-calendar-inv' => 'Calendar-inv' ],
		[ 'vc-oi vc-oi-location-alt' => 'Location-alt' ],
		[ 'vc-oi vc-oi-download-cloud' => 'Download-cloud' ],
		[ 'vc-oi vc-oi-cancel-circle' => 'Cancel-circle' ],
		[ 'vc-oi vc-oi-plus' => 'Plus' ],
		[ 'vc-oi vc-oi-upload-cloud' => 'Upload-cloud' ],
		[ 'vc-oi vc-oi-compass' => 'Compass' ],
		[ 'vc-oi vc-oi-calendar-alt' => 'Calendar-alt' ],
		[ 'vc-oi vc-oi-down-circle' => 'Down-circle' ],
		[ 'vc-oi vc-oi-award-empty' => 'Award-empty' ],
		[ 'vc-oi vc-oi-layers-alt' => 'Layers-alt' ],
		[ 'vc-oi vc-oi-sun' => 'Sun' ],
		[ 'vc-oi vc-oi-list' => 'List' ],
		[ 'vc-oi vc-oi-left-circle' => 'Left-circle' ],
		[ 'vc-oi vc-oi-mic' => 'Mic' ],
		[ 'vc-oi vc-oi-trash' => 'Trash' ],
		[ 'vc-oi vc-oi-quote-left' => 'Quote-left' ],
		[ 'vc-oi vc-oi-plus-circle' => 'Plus-circle' ],
		[ 'vc-oi vc-oi-minus' => 'Minus' ],
		[ 'vc-oi vc-oi-quote-right' => 'Quote-right' ],
		[ 'vc-oi vc-oi-trash-empty' => 'Trash-empty' ],
		[ 'vc-oi vc-oi-volume-off' => 'Volume-off' ],
		[ 'vc-oi vc-oi-right-circle' => 'Right-circle' ],
		[ 'vc-oi vc-oi-list-nested' => 'List-nested' ],
		[ 'vc-oi vc-oi-sun-inv' => 'Sun-inv' ],
		[ 'vc-oi vc-oi-bat-empty' => 'Bat-empty' ],
		[ 'vc-oi vc-oi-up-circle' => 'Up-circle' ],
		[ 'vc-oi vc-oi-volume-up' => 'Volume-up' ],
		[ 'vc-oi vc-oi-doc' => 'Doc' ],
		[ 'vc-oi vc-oi-quote-left-alt' => 'Quote-left-alt' ],
		[ 'vc-oi vc-oi-minus-circle' => 'Minus-circle' ],
		[ 'vc-oi vc-oi-cloud' => 'Cloud' ],
		[ 'vc-oi vc-oi-rain' => 'Rain' ],
		[ 'vc-oi vc-oi-bat-half' => 'Bat-half' ],
		[ 'vc-oi vc-oi-cw' => 'Cw' ],
		[ 'vc-oi vc-oi-headphones' => 'Headphones' ],
		[ 'vc-oi vc-oi-doc-inv' => 'Doc-inv' ],
		[ 'vc-oi vc-oi-quote-right-alt' => 'Quote-right-alt' ],
		[ 'vc-oi vc-oi-help' => 'Help' ],
		[ 'vc-oi vc-oi-info' => 'Info' ],
		[ 'vc-oi vc-oi-pencil' => 'Pencil' ],
		[ 'vc-oi vc-oi-doc-alt' => 'Doc-alt' ],
		[ 'vc-oi vc-oi-clock' => 'Clock' ],
		[ 'vc-oi vc-oi-loop' => 'Loop' ],
		[ 'vc-oi vc-oi-bat-full' => 'Bat-full' ],
		[ 'vc-oi vc-oi-flash' => 'Flash' ],
		[ 'vc-oi vc-oi-moon' => 'Moon' ],
		[ 'vc-oi vc-oi-bat-charge' => 'Bat-charge' ],
		[ 'vc-oi vc-oi-loop-alt' => 'Loop-alt' ],
		[ 'vc-oi vc-oi-lamp' => 'Lamp' ],
		[ 'vc-oi vc-oi-doc-inv-alt' => 'Doc-inv-alt' ],
		[ 'vc-oi vc-oi-pencil-neg' => 'Pencil-neg' ],
		[ 'vc-oi vc-oi-home' => 'Home' ],
	];

	return array_merge( $icons, $openiconic_icons );
}

add_filter( 'vc_iconpicker-type-typicons', 'vc_iconpicker_type_typicons' );

/**
 * Typicons icons from github.com/stephenhutchings/typicons.font
 *
 * @param array $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown.
 *
 * @return array - of icons for iconpicker, can be categorized, or not.
 * @since 4.4
 */
function vc_iconpicker_type_typicons( $icons ) {
	$typicons_icons = [
		[ 'typcn typcn-adjust-brightness' => 'Adjust Brightness' ],
		[ 'typcn typcn-adjust-contrast' => 'Adjust Contrast' ],
		[ 'typcn typcn-anchor-outline' => 'Anchor Outline' ],
		[ 'typcn typcn-anchor' => 'Anchor' ],
		[ 'typcn typcn-archive' => 'Archive' ],
		[ 'typcn typcn-arrow-back-outline' => 'Arrow Back Outline' ],
		[ 'typcn typcn-arrow-back' => 'Arrow Back' ],
		[ 'typcn typcn-arrow-down-outline' => 'Arrow Down Outline' ],
		[ 'typcn typcn-arrow-down-thick' => 'Arrow Down Thick' ],
		[ 'typcn typcn-arrow-down' => 'Arrow Down' ],
		[ 'typcn typcn-arrow-forward-outline' => 'Arrow Forward Outline' ],
		[ 'typcn typcn-arrow-forward' => 'Arrow Forward' ],
		[ 'typcn typcn-arrow-left-outline' => 'Arrow Left Outline' ],
		[ 'typcn typcn-arrow-left-thick' => 'Arrow Left Thick' ],
		[ 'typcn typcn-arrow-left' => 'Arrow Left' ],
		[ 'typcn typcn-arrow-loop-outline' => 'Arrow Loop Outline' ],
		[ 'typcn typcn-arrow-loop' => 'Arrow Loop' ],
		[ 'typcn typcn-arrow-maximise-outline' => 'Arrow Maximise Outline' ],
		[ 'typcn typcn-arrow-maximise' => 'Arrow Maximise' ],
		[ 'typcn typcn-arrow-minimise-outline' => 'Arrow Minimise Outline' ],
		[ 'typcn typcn-arrow-minimise' => 'Arrow Minimise' ],
		[ 'typcn typcn-arrow-move-outline' => 'Arrow Move Outline' ],
		[ 'typcn typcn-arrow-move' => 'Arrow Move' ],
		[ 'typcn typcn-arrow-repeat-outline' => 'Arrow Repeat Outline' ],
		[ 'typcn typcn-arrow-repeat' => 'Arrow Repeat' ],
		[ 'typcn typcn-arrow-right-outline' => 'Arrow Right Outline' ],
		[ 'typcn typcn-arrow-right-thick' => 'Arrow Right Thick' ],
		[ 'typcn typcn-arrow-right' => 'Arrow Right' ],
		[ 'typcn typcn-arrow-shuffle' => 'Arrow Shuffle' ],
		[ 'typcn typcn-arrow-sorted-down' => 'Arrow Sorted Down' ],
		[ 'typcn typcn-arrow-sorted-up' => 'Arrow Sorted Up' ],
		[ 'typcn typcn-arrow-sync-outline' => 'Arrow Sync Outline' ],
		[ 'typcn typcn-arrow-sync' => 'Arrow Sync' ],
		[ 'typcn typcn-arrow-unsorted' => 'Arrow Unsorted' ],
		[ 'typcn typcn-arrow-up-outline' => 'Arrow Up Outline' ],
		[ 'typcn typcn-arrow-up-thick' => 'Arrow Up Thick' ],
		[ 'typcn typcn-arrow-up' => 'Arrow Up' ],
		[ 'typcn typcn-at' => 'At' ],
		[ 'typcn typcn-attachment-outline' => 'Attachment Outline' ],
		[ 'typcn typcn-attachment' => 'Attachment' ],
		[ 'typcn typcn-backspace-outline' => 'Backspace Outline' ],
		[ 'typcn typcn-backspace' => 'Backspace' ],
		[ 'typcn typcn-battery-charge' => 'Battery Charge' ],
		[ 'typcn typcn-battery-full' => 'Battery Full' ],
		[ 'typcn typcn-battery-high' => 'Battery High' ],
		[ 'typcn typcn-battery-low' => 'Battery Low' ],
		[ 'typcn typcn-battery-mid' => 'Battery Mid' ],
		[ 'typcn typcn-beaker' => 'Beaker' ],
		[ 'typcn typcn-beer' => 'Beer' ],
		[ 'typcn typcn-bell' => 'Bell' ],
		[ 'typcn typcn-book' => 'Book' ],
		[ 'typcn typcn-bookmark' => 'Bookmark' ],
		[ 'typcn typcn-briefcase' => 'Briefcase' ],
		[ 'typcn typcn-brush' => 'Brush' ],
		[ 'typcn typcn-business-card' => 'Business Card' ],
		[ 'typcn typcn-calculator' => 'Calculator' ],
		[ 'typcn typcn-calendar-outline' => 'Calendar Outline' ],
		[ 'typcn typcn-calendar' => 'Calendar' ],
		[ 'typcn typcn-camera-outline' => 'Camera Outline' ],
		[ 'typcn typcn-camera' => 'Camera' ],
		[ 'typcn typcn-cancel-outline' => 'Cancel Outline' ],
		[ 'typcn typcn-cancel' => 'Cancel' ],
		[ 'typcn typcn-chart-area-outline' => 'Chart Area Outline' ],
		[ 'typcn typcn-chart-area' => 'Chart Area' ],
		[ 'typcn typcn-chart-bar-outline' => 'Chart Bar Outline' ],
		[ 'typcn typcn-chart-bar' => 'Chart Bar' ],
		[ 'typcn typcn-chart-line-outline' => 'Chart Line Outline' ],
		[ 'typcn typcn-chart-line' => 'Chart Line' ],
		[ 'typcn typcn-chart-pie-outline' => 'Chart Pie Outline' ],
		[ 'typcn typcn-chart-pie' => 'Chart Pie' ],
		[ 'typcn typcn-chevron-left-outline' => 'Chevron Left Outline' ],
		[ 'typcn typcn-chevron-left' => 'Chevron Left' ],
		[ 'typcn typcn-chevron-right-outline' => 'Chevron Right Outline' ],
		[ 'typcn typcn-chevron-right' => 'Chevron Right' ],
		[ 'typcn typcn-clipboard' => 'Clipboard' ],
		[ 'typcn typcn-cloud-storage' => 'Cloud Storage' ],
		[ 'typcn typcn-cloud-storage-outline' => 'Cloud Storage Outline' ],
		[ 'typcn typcn-code-outline' => 'Code Outline' ],
		[ 'typcn typcn-code' => 'Code' ],
		[ 'typcn typcn-coffee' => 'Coffee' ],
		[ 'typcn typcn-cog-outline' => 'Cog Outline' ],
		[ 'typcn typcn-cog' => 'Cog' ],
		[ 'typcn typcn-compass' => 'Compass' ],
		[ 'typcn typcn-contacts' => 'Contacts' ],
		[ 'typcn typcn-credit-card' => 'Credit Card' ],
		[ 'typcn typcn-css3' => 'Css3' ],
		[ 'typcn typcn-database' => 'Database' ],
		[ 'typcn typcn-delete-outline' => 'Delete Outline' ],
		[ 'typcn typcn-delete' => 'Delete' ],
		[ 'typcn typcn-device-desktop' => 'Device Desktop' ],
		[ 'typcn typcn-device-laptop' => 'Device Laptop' ],
		[ 'typcn typcn-device-phone' => 'Device Phone' ],
		[ 'typcn typcn-device-tablet' => 'Device Tablet' ],
		[ 'typcn typcn-directions' => 'Directions' ],
		[ 'typcn typcn-divide-outline' => 'Divide Outline' ],
		[ 'typcn typcn-divide' => 'Divide' ],
		[ 'typcn typcn-document-add' => 'Document Add' ],
		[ 'typcn typcn-document-delete' => 'Document Delete' ],
		[ 'typcn typcn-document-text' => 'Document Text' ],
		[ 'typcn typcn-document' => 'Document' ],
		[ 'typcn typcn-download-outline' => 'Download Outline' ],
		[ 'typcn typcn-download' => 'Download' ],
		[ 'typcn typcn-dropbox' => 'Dropbox' ],
		[ 'typcn typcn-edit' => 'Edit' ],
		[ 'typcn typcn-eject-outline' => 'Eject Outline' ],
		[ 'typcn typcn-eject' => 'Eject' ],
		[ 'typcn typcn-equals-outline' => 'Equals Outline' ],
		[ 'typcn typcn-equals' => 'Equals' ],
		[ 'typcn typcn-export-outline' => 'Export Outline' ],
		[ 'typcn typcn-export' => 'Export' ],
		[ 'typcn typcn-eye-outline' => 'Eye Outline' ],
		[ 'typcn typcn-eye' => 'Eye' ],
		[ 'typcn typcn-feather' => 'Feather' ],
		[ 'typcn typcn-film' => 'Film' ],
		[ 'typcn typcn-filter' => 'Filter' ],
		[ 'typcn typcn-flag-outline' => 'Flag Outline' ],
		[ 'typcn typcn-flag' => 'Flag' ],
		[ 'typcn typcn-flash-outline' => 'Flash Outline' ],
		[ 'typcn typcn-flash' => 'Flash' ],
		[ 'typcn typcn-flow-children' => 'Flow Children' ],
		[ 'typcn typcn-flow-merge' => 'Flow Merge' ],
		[ 'typcn typcn-flow-parallel' => 'Flow Parallel' ],
		[ 'typcn typcn-flow-switch' => 'Flow Switch' ],
		[ 'typcn typcn-folder-add' => 'Folder Add' ],
		[ 'typcn typcn-folder-delete' => 'Folder Delete' ],
		[ 'typcn typcn-folder-open' => 'Folder Open' ],
		[ 'typcn typcn-folder' => 'Folder' ],
		[ 'typcn typcn-gift' => 'Gift' ],
		[ 'typcn typcn-globe-outline' => 'Globe Outline' ],
		[ 'typcn typcn-globe' => 'Globe' ],
		[ 'typcn typcn-group-outline' => 'Group Outline' ],
		[ 'typcn typcn-group' => 'Group' ],
		[ 'typcn typcn-headphones' => 'Headphones' ],
		[ 'typcn typcn-heart-full-outline' => 'Heart Full Outline' ],
		[ 'typcn typcn-heart-half-outline' => 'Heart Half Outline' ],
		[ 'typcn typcn-heart-outline' => 'Heart Outline' ],
		[ 'typcn typcn-heart' => 'Heart' ],
		[ 'typcn typcn-home-outline' => 'Home Outline' ],
		[ 'typcn typcn-home' => 'Home' ],
		[ 'typcn typcn-html5' => 'Html5' ],
		[ 'typcn typcn-image-outline' => 'Image Outline' ],
		[ 'typcn typcn-image' => 'Image' ],
		[ 'typcn typcn-infinity-outline' => 'Infinity Outline' ],
		[ 'typcn typcn-infinity' => 'Infinity' ],
		[ 'typcn typcn-info-large-outline' => 'Info Large Outline' ],
		[ 'typcn typcn-info-large' => 'Info Large' ],
		[ 'typcn typcn-info-outline' => 'Info Outline' ],
		[ 'typcn typcn-info' => 'Info' ],
		[ 'typcn typcn-input-checked-outline' => 'Input Checked Outline' ],
		[ 'typcn typcn-input-checked' => 'Input Checked' ],
		[ 'typcn typcn-key-outline' => 'Key Outline' ],
		[ 'typcn typcn-key' => 'Key' ],
		[ 'typcn typcn-keyboard' => 'Keyboard' ],
		[ 'typcn typcn-leaf' => 'Leaf' ],
		[ 'typcn typcn-lightbulb' => 'Lightbulb' ],
		[ 'typcn typcn-link-outline' => 'Link Outline' ],
		[ 'typcn typcn-link' => 'Link' ],
		[ 'typcn typcn-location-arrow-outline' => 'Location Arrow Outline' ],
		[ 'typcn typcn-location-arrow' => 'Location Arrow' ],
		[ 'typcn typcn-location-outline' => 'Location Outline' ],
		[ 'typcn typcn-location' => 'Location' ],
		[ 'typcn typcn-lock-closed-outline' => 'Lock Closed Outline' ],
		[ 'typcn typcn-lock-closed' => 'Lock Closed' ],
		[ 'typcn typcn-lock-open-outline' => 'Lock Open Outline' ],
		[ 'typcn typcn-lock-open' => 'Lock Open' ],
		[ 'typcn typcn-mail' => 'Mail' ],
		[ 'typcn typcn-map' => 'Map' ],
		[ 'typcn typcn-media-eject-outline' => 'Media Eject Outline' ],
		[ 'typcn typcn-media-eject' => 'Media Eject' ],
		[ 'typcn typcn-media-fast-forward-outline' => 'Media Fast Forward Outline' ],
		[ 'typcn typcn-media-fast-forward' => 'Media Fast Forward' ],
		[ 'typcn typcn-media-pause-outline' => 'Media Pause Outline' ],
		[ 'typcn typcn-media-pause' => 'Media Pause' ],
		[ 'typcn typcn-media-play-outline' => 'Media Play Outline' ],
		[ 'typcn typcn-media-play-reverse-outline' => 'Media Play Reverse Outline' ],
		[ 'typcn typcn-media-play-reverse' => 'Media Play Reverse' ],
		[ 'typcn typcn-media-play' => 'Media Play' ],
		[ 'typcn typcn-media-record-outline' => 'Media Record Outline' ],
		[ 'typcn typcn-media-record' => 'Media Record' ],
		[ 'typcn typcn-media-rewind-outline' => 'Media Rewind Outline' ],
		[ 'typcn typcn-media-rewind' => 'Media Rewind' ],
		[ 'typcn typcn-media-stop-outline' => 'Media Stop Outline' ],
		[ 'typcn typcn-media-stop' => 'Media Stop' ],
		[ 'typcn typcn-message-typing' => 'Message Typing' ],
		[ 'typcn typcn-message' => 'Message' ],
		[ 'typcn typcn-messages' => 'Messages' ],
		[ 'typcn typcn-microphone-outline' => 'Microphone Outline' ],
		[ 'typcn typcn-microphone' => 'Microphone' ],
		[ 'typcn typcn-minus-outline' => 'Minus Outline' ],
		[ 'typcn typcn-minus' => 'Minus' ],
		[ 'typcn typcn-mortar-board' => 'Mortar Board' ],
		[ 'typcn typcn-news' => 'News' ],
		[ 'typcn typcn-notes-outline' => 'Notes Outline' ],
		[ 'typcn typcn-notes' => 'Notes' ],
		[ 'typcn typcn-pen' => 'Pen' ],
		[ 'typcn typcn-pencil' => 'Pencil' ],
		[ 'typcn typcn-phone-outline' => 'Phone Outline' ],
		[ 'typcn typcn-phone' => 'Phone' ],
		[ 'typcn typcn-pi-outline' => 'Pi Outline' ],
		[ 'typcn typcn-pi' => 'Pi' ],
		[ 'typcn typcn-pin-outline' => 'Pin Outline' ],
		[ 'typcn typcn-pin' => 'Pin' ],
		[ 'typcn typcn-pipette' => 'Pipette' ],
		[ 'typcn typcn-plane-outline' => 'Plane Outline' ],
		[ 'typcn typcn-plane' => 'Plane' ],
		[ 'typcn typcn-plug' => 'Plug' ],
		[ 'typcn typcn-plus-outline' => 'Plus Outline' ],
		[ 'typcn typcn-plus' => 'Plus' ],
		[ 'typcn typcn-point-of-interest-outline' => 'Point Of Interest Outline' ],
		[ 'typcn typcn-point-of-interest' => 'Point Of Interest' ],
		[ 'typcn typcn-power-outline' => 'Power Outline' ],
		[ 'typcn typcn-power' => 'Power' ],
		[ 'typcn typcn-printer' => 'Printer' ],
		[ 'typcn typcn-puzzle-outline' => 'Puzzle Outline' ],
		[ 'typcn typcn-puzzle' => 'Puzzle' ],
		[ 'typcn typcn-radar-outline' => 'Radar Outline' ],
		[ 'typcn typcn-radar' => 'Radar' ],
		[ 'typcn typcn-refresh-outline' => 'Refresh Outline' ],
		[ 'typcn typcn-refresh' => 'Refresh' ],
		[ 'typcn typcn-rss-outline' => 'Rss Outline' ],
		[ 'typcn typcn-rss' => 'Rss' ],
		[ 'typcn typcn-scissors-outline' => 'Scissors Outline' ],
		[ 'typcn typcn-scissors' => 'Scissors' ],
		[ 'typcn typcn-shopping-bag' => 'Shopping Bag' ],
		[ 'typcn typcn-shopping-cart' => 'Shopping Cart' ],
		[ 'typcn typcn-social-at-circular' => 'Social At Circular' ],
		[ 'typcn typcn-social-dribbble-circular' => 'Social Dribbble Circular' ],
		[ 'typcn typcn-social-dribbble' => 'Social Dribbble' ],
		[ 'typcn typcn-social-facebook-circular' => 'Social Facebook Circular' ],
		[ 'typcn typcn-social-facebook' => 'Social Facebook' ],
		[ 'typcn typcn-social-flickr-circular' => 'Social Flickr Circular' ],
		[ 'typcn typcn-social-flickr' => 'Social Flickr' ],
		[ 'typcn typcn-social-github-circular' => 'Social Github Circular' ],
		[ 'typcn typcn-social-github' => 'Social Github' ],
		[ 'typcn typcn-social-google-plus-circular' => 'Social Google Plus Circular' ],
		[ 'typcn typcn-social-google-plus' => 'Social Google Plus' ],
		[ 'typcn typcn-social-instagram-circular' => 'Social Instagram Circular' ],
		[ 'typcn typcn-social-instagram' => 'Social Instagram' ],
		[ 'typcn typcn-social-last-fm-circular' => 'Social Last Fm Circular' ],
		[ 'typcn typcn-social-last-fm' => 'Social Last Fm' ],
		[ 'typcn typcn-social-linkedin-circular' => 'Social Linkedin Circular' ],
		[ 'typcn typcn-social-linkedin' => 'Social Linkedin' ],
		[ 'typcn typcn-social-pinterest-circular' => 'Social Pinterest Circular' ],
		[ 'typcn typcn-social-pinterest' => 'Social Pinterest' ],
		[ 'typcn typcn-social-skype-outline' => 'Social Skype Outline' ],
		[ 'typcn typcn-social-skype' => 'Social Skype' ],
		[ 'typcn typcn-social-tumbler-circular' => 'Social Tumbler Circular' ],
		[ 'typcn typcn-social-tumbler' => 'Social Tumbler' ],
		[ 'typcn typcn-social-twitter-circular' => 'Social Twitter Circular' ],
		[ 'typcn typcn-social-twitter' => 'Social Twitter' ],
		[ 'typcn typcn-social-vimeo-circular' => 'Social Vimeo Circular' ],
		[ 'typcn typcn-social-vimeo' => 'Social Vimeo' ],
		[ 'typcn typcn-social-youtube-circular' => 'Social Youtube Circular' ],
		[ 'typcn typcn-social-youtube' => 'Social Youtube' ],
		[ 'typcn typcn-sort-alphabetically-outline' => 'Sort Alphabetically Outline' ],
		[ 'typcn typcn-sort-alphabetically' => 'Sort Alphabetically' ],
		[ 'typcn typcn-sort-numerically-outline' => 'Sort Numerically Outline' ],
		[ 'typcn typcn-sort-numerically' => 'Sort Numerically' ],
		[ 'typcn typcn-spanner-outline' => 'Spanner Outline' ],
		[ 'typcn typcn-spanner' => 'Spanner' ],
		[ 'typcn typcn-spiral' => 'Spiral' ],
		[ 'typcn typcn-star-full-outline' => 'Star Full Outline' ],
		[ 'typcn typcn-star-half-outline' => 'Star Half Outline' ],
		[ 'typcn typcn-star-half' => 'Star Half' ],
		[ 'typcn typcn-star-outline' => 'Star Outline' ],
		[ 'typcn typcn-star' => 'Star' ],
		[ 'typcn typcn-starburst-outline' => 'Starburst Outline' ],
		[ 'typcn typcn-starburst' => 'Starburst' ],
		[ 'typcn typcn-stopwatch' => 'Stopwatch' ],
		[ 'typcn typcn-support' => 'Support' ],
		[ 'typcn typcn-tabs-outline' => 'Tabs Outline' ],
		[ 'typcn typcn-tag' => 'Tag' ],
		[ 'typcn typcn-tags' => 'Tags' ],
		[ 'typcn typcn-th-large-outline' => 'Th Large Outline' ],
		[ 'typcn typcn-th-large' => 'Th Large' ],
		[ 'typcn typcn-th-list-outline' => 'Th List Outline' ],
		[ 'typcn typcn-th-list' => 'Th List' ],
		[ 'typcn typcn-th-menu-outline' => 'Th Menu Outline' ],
		[ 'typcn typcn-th-menu' => 'Th Menu' ],
		[ 'typcn typcn-th-small-outline' => 'Th Small Outline' ],
		[ 'typcn typcn-th-small' => 'Th Small' ],
		[ 'typcn typcn-thermometer' => 'Thermometer' ],
		[ 'typcn typcn-thumbs-down' => 'Thumbs Down' ],
		[ 'typcn typcn-thumbs-ok' => 'Thumbs Ok' ],
		[ 'typcn typcn-thumbs-up' => 'Thumbs Up' ],
		[ 'typcn typcn-tick-outline' => 'Tick Outline' ],
		[ 'typcn typcn-tick' => 'Tick' ],
		[ 'typcn typcn-ticket' => 'Ticket' ],
		[ 'typcn typcn-time' => 'Time' ],
		[ 'typcn typcn-times-outline' => 'Times Outline' ],
		[ 'typcn typcn-times' => 'Times' ],
		[ 'typcn typcn-trash' => 'Trash' ],
		[ 'typcn typcn-tree' => 'Tree' ],
		[ 'typcn typcn-upload-outline' => 'Upload Outline' ],
		[ 'typcn typcn-upload' => 'Upload' ],
		[ 'typcn typcn-user-add-outline' => 'User Add Outline' ],
		[ 'typcn typcn-user-add' => 'User Add' ],
		[ 'typcn typcn-user-delete-outline' => 'User Delete Outline' ],
		[ 'typcn typcn-user-delete' => 'User Delete' ],
		[ 'typcn typcn-user-outline' => 'User Outline' ],
		[ 'typcn typcn-user' => 'User' ],
		[ 'typcn typcn-vendor-android' => 'Vendor Android' ],
		[ 'typcn typcn-vendor-apple' => 'Vendor Apple' ],
		[ 'typcn typcn-vendor-microsoft' => 'Vendor Microsoft' ],
		[ 'typcn typcn-video-outline' => 'Video Outline' ],
		[ 'typcn typcn-video' => 'Video' ],
		[ 'typcn typcn-volume-down' => 'Volume Down' ],
		[ 'typcn typcn-volume-mute' => 'Volume Mute' ],
		[ 'typcn typcn-volume-up' => 'Volume Up' ],
		[ 'typcn typcn-volume' => 'Volume' ],
		[ 'typcn typcn-warning-outline' => 'Warning Outline' ],
		[ 'typcn typcn-warning' => 'Warning' ],
		[ 'typcn typcn-watch' => 'Watch' ],
		[ 'typcn typcn-waves-outline' => 'Waves Outline' ],
		[ 'typcn typcn-waves' => 'Waves' ],
		[ 'typcn typcn-weather-cloudy' => 'Weather Cloudy' ],
		[ 'typcn typcn-weather-downpour' => 'Weather Downpour' ],
		[ 'typcn typcn-weather-night' => 'Weather Night' ],
		[ 'typcn typcn-weather-partly-sunny' => 'Weather Partly Sunny' ],
		[ 'typcn typcn-weather-shower' => 'Weather Shower' ],
		[ 'typcn typcn-weather-snow' => 'Weather Snow' ],
		[ 'typcn typcn-weather-stormy' => 'Weather Stormy' ],
		[ 'typcn typcn-weather-sunny' => 'Weather Sunny' ],
		[ 'typcn typcn-weather-windy-cloudy' => 'Weather Windy Cloudy' ],
		[ 'typcn typcn-weather-windy' => 'Weather Windy' ],
		[ 'typcn typcn-wi-fi-outline' => 'Wi Fi Outline' ],
		[ 'typcn typcn-wi-fi' => 'Wi Fi' ],
		[ 'typcn typcn-wine' => 'Wine' ],
		[ 'typcn typcn-world-outline' => 'World Outline' ],
		[ 'typcn typcn-world' => 'World' ],
		[ 'typcn typcn-zoom-in-outline' => 'Zoom In Outline' ],
		[ 'typcn typcn-zoom-in' => 'Zoom In' ],
		[ 'typcn typcn-zoom-out-outline' => 'Zoom Out Outline' ],
		[ 'typcn typcn-zoom-out' => 'Zoom Out' ],
		[ 'typcn typcn-zoom-outline' => 'Zoom Outline' ],
		[ 'typcn typcn-zoom' => 'Zoom' ],
	];

	return array_merge( $icons, $typicons_icons );
}

add_filter( 'vc_iconpicker-type-entypo', 'vc_iconpicker_type_entypo' );
/**
 * Entypo icons from github.com/danielbruce/entypo
 *
 * @param array $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown.
 *
 * @return array - of icons for iconpicker, can be categorized, or not.
 * @since 4.4
 */
function vc_iconpicker_type_entypo( $icons ) {
	$entypo_icons = [
		[ 'entypo-icon entypo-icon-note' => 'Note' ],
		[ 'entypo-icon entypo-icon-note-beamed' => 'Note Beamed' ],
		[ 'entypo-icon entypo-icon-music' => 'Music' ],
		[ 'entypo-icon entypo-icon-search' => 'Search' ],
		[ 'entypo-icon entypo-icon-flashlight' => 'Flashlight' ],
		[ 'entypo-icon entypo-icon-mail' => 'Mail' ],
		[ 'entypo-icon entypo-icon-heart' => 'Heart' ],
		[ 'entypo-icon entypo-icon-heart-empty' => 'Heart Empty' ],
		[ 'entypo-icon entypo-icon-star' => 'Star' ],
		[ 'entypo-icon entypo-icon-star-empty' => 'Star Empty' ],
		[ 'entypo-icon entypo-icon-user' => 'User' ],
		[ 'entypo-icon entypo-icon-users' => 'Users' ],
		[ 'entypo-icon entypo-icon-user-add' => 'User Add' ],
		[ 'entypo-icon entypo-icon-video' => 'Video' ],
		[ 'entypo-icon entypo-icon-picture' => 'Picture' ],
		[ 'entypo-icon entypo-icon-camera' => 'Camera' ],
		[ 'entypo-icon entypo-icon-layout' => 'Layout' ],
		[ 'entypo-icon entypo-icon-menu' => 'Menu' ],
		[ 'entypo-icon entypo-icon-check' => 'Check' ],
		[ 'entypo-icon entypo-icon-cancel' => 'Cancel' ],
		[ 'entypo-icon entypo-icon-cancel-circled' => 'Cancel Circled' ],
		[ 'entypo-icon entypo-icon-cancel-squared' => 'Cancel Squared' ],
		[ 'entypo-icon entypo-icon-plus' => 'Plus' ],
		[ 'entypo-icon entypo-icon-plus-circled' => 'Plus Circled' ],
		[ 'entypo-icon entypo-icon-plus-squared' => 'Plus Squared' ],
		[ 'entypo-icon entypo-icon-minus' => 'Minus' ],
		[ 'entypo-icon entypo-icon-minus-circled' => 'Minus Circled' ],
		[ 'entypo-icon entypo-icon-minus-squared' => 'Minus Squared' ],
		[ 'entypo-icon entypo-icon-help' => 'Help' ],
		[ 'entypo-icon entypo-icon-help-circled' => 'Help Circled' ],
		[ 'entypo-icon entypo-icon-info' => 'Info' ],
		[ 'entypo-icon entypo-icon-info-circled' => 'Info Circled' ],
		[ 'entypo-icon entypo-icon-back' => 'Back' ],
		[ 'entypo-icon entypo-icon-home' => 'Home' ],
		[ 'entypo-icon entypo-icon-link' => 'Link' ],
		[ 'entypo-icon entypo-icon-attach' => 'Attach' ],
		[ 'entypo-icon entypo-icon-lock' => 'Lock' ],
		[ 'entypo-icon entypo-icon-lock-open' => 'Lock Open' ],
		[ 'entypo-icon entypo-icon-eye' => 'Eye' ],
		[ 'entypo-icon entypo-icon-tag' => 'Tag' ],
		[ 'entypo-icon entypo-icon-bookmark' => 'Bookmark' ],
		[ 'entypo-icon entypo-icon-bookmarks' => 'Bookmarks' ],
		[ 'entypo-icon entypo-icon-flag' => 'Flag' ],
		[ 'entypo-icon entypo-icon-thumbs-up' => 'Thumbs Up' ],
		[ 'entypo-icon entypo-icon-thumbs-down' => 'Thumbs Down' ],
		[ 'entypo-icon entypo-icon-download' => 'Download' ],
		[ 'entypo-icon entypo-icon-upload' => 'Upload' ],
		[ 'entypo-icon entypo-icon-upload-cloud' => 'Upload Cloud' ],
		[ 'entypo-icon entypo-icon-reply' => 'Reply' ],
		[ 'entypo-icon entypo-icon-reply-all' => 'Reply All' ],
		[ 'entypo-icon entypo-icon-forward' => 'Forward' ],
		[ 'entypo-icon entypo-icon-quote' => 'Quote' ],
		[ 'entypo-icon entypo-icon-code' => 'Code' ],
		[ 'entypo-icon entypo-icon-export' => 'Export' ],
		[ 'entypo-icon entypo-icon-pencil' => 'Pencil' ],
		[ 'entypo-icon entypo-icon-feather' => 'Feather' ],
		[ 'entypo-icon entypo-icon-print' => 'Print' ],
		[ 'entypo-icon entypo-icon-retweet' => 'Retweet' ],
		[ 'entypo-icon entypo-icon-keyboard' => 'Keyboard' ],
		[ 'entypo-icon entypo-icon-comment' => 'Comment' ],
		[ 'entypo-icon entypo-icon-chat' => 'Chat' ],
		[ 'entypo-icon entypo-icon-bell' => 'Bell' ],
		[ 'entypo-icon entypo-icon-attention' => 'Attention' ],
		[ 'entypo-icon entypo-icon-alert' => 'Alert' ],
		[ 'entypo-icon entypo-icon-vcard' => 'Vcard' ],
		[ 'entypo-icon entypo-icon-address' => 'Address' ],
		[ 'entypo-icon entypo-icon-location' => 'Location' ],
		[ 'entypo-icon entypo-icon-map' => 'Map' ],
		[ 'entypo-icon entypo-icon-direction' => 'Direction' ],
		[ 'entypo-icon entypo-icon-compass' => 'Compass' ],
		[ 'entypo-icon entypo-icon-cup' => 'Cup' ],
		[ 'entypo-icon entypo-icon-trash' => 'Trash' ],
		[ 'entypo-icon entypo-icon-doc' => 'Doc' ],
		[ 'entypo-icon entypo-icon-docs' => 'Docs' ],
		[ 'entypo-icon entypo-icon-doc-landscape' => 'Doc Landscape' ],
		[ 'entypo-icon entypo-icon-doc-text' => 'Doc Text' ],
		[ 'entypo-icon entypo-icon-doc-text-inv' => 'Doc Text Inv' ],
		[ 'entypo-icon entypo-icon-newspaper' => 'Newspaper' ],
		[ 'entypo-icon entypo-icon-book-open' => 'Book Open' ],
		[ 'entypo-icon entypo-icon-book' => 'Book' ],
		[ 'entypo-icon entypo-icon-folder' => 'Folder' ],
		[ 'entypo-icon entypo-icon-archive' => 'Archive' ],
		[ 'entypo-icon entypo-icon-box' => 'Box' ],
		[ 'entypo-icon entypo-icon-rss' => 'Rss' ],
		[ 'entypo-icon entypo-icon-phone' => 'Phone' ],
		[ 'entypo-icon entypo-icon-cog' => 'Cog' ],
		[ 'entypo-icon entypo-icon-tools' => 'Tools' ],
		[ 'entypo-icon entypo-icon-share' => 'Share' ],
		[ 'entypo-icon entypo-icon-shareable' => 'Shareable' ],
		[ 'entypo-icon entypo-icon-basket' => 'Basket' ],
		[ 'entypo-icon entypo-icon-bag' => 'Bag' ],
		[ 'entypo-icon entypo-icon-calendar' => 'Calendar' ],
		[ 'entypo-icon entypo-icon-login' => 'Login' ],
		[ 'entypo-icon entypo-icon-logout' => 'Logout' ],
		[ 'entypo-icon entypo-icon-mic' => 'Mic' ],
		[ 'entypo-icon entypo-icon-mute' => 'Mute' ],
		[ 'entypo-icon entypo-icon-sound' => 'Sound' ],
		[ 'entypo-icon entypo-icon-volume' => 'Volume' ],
		[ 'entypo-icon entypo-icon-clock' => 'Clock' ],
		[ 'entypo-icon entypo-icon-hourglass' => 'Hourglass' ],
		[ 'entypo-icon entypo-icon-lamp' => 'Lamp' ],
		[ 'entypo-icon entypo-icon-light-down' => 'Light Down' ],
		[ 'entypo-icon entypo-icon-light-up' => 'Light Up' ],
		[ 'entypo-icon entypo-icon-adjust' => 'Adjust' ],
		[ 'entypo-icon entypo-icon-block' => 'Block' ],
		[ 'entypo-icon entypo-icon-resize-full' => 'Resize Full' ],
		[ 'entypo-icon entypo-icon-resize-small' => 'Resize Small' ],
		[ 'entypo-icon entypo-icon-popup' => 'Popup' ],
		[ 'entypo-icon entypo-icon-publish' => 'Publish' ],
		[ 'entypo-icon entypo-icon-window' => 'Window' ],
		[ 'entypo-icon entypo-icon-arrow-combo' => 'Arrow Combo' ],
		[ 'entypo-icon entypo-icon-down-circled' => 'Down Circled' ],
		[ 'entypo-icon entypo-icon-left-circled' => 'Left Circled' ],
		[ 'entypo-icon entypo-icon-right-circled' => 'Right Circled' ],
		[ 'entypo-icon entypo-icon-up-circled' => 'Up Circled' ],
		[ 'entypo-icon entypo-icon-down-open' => 'Down Open' ],
		[ 'entypo-icon entypo-icon-left-open' => 'Left Open' ],
		[ 'entypo-icon entypo-icon-right-open' => 'Right Open' ],
		[ 'entypo-icon entypo-icon-up-open' => 'Up Open' ],
		[ 'entypo-icon entypo-icon-down-open-mini' => 'Down Open Mini' ],
		[ 'entypo-icon entypo-icon-left-open-mini' => 'Left Open Mini' ],
		[ 'entypo-icon entypo-icon-right-open-mini' => 'Right Open Mini' ],
		[ 'entypo-icon entypo-icon-up-open-mini' => 'Up Open Mini' ],
		[ 'entypo-icon entypo-icon-down-open-big' => 'Down Open Big' ],
		[ 'entypo-icon entypo-icon-left-open-big' => 'Left Open Big' ],
		[ 'entypo-icon entypo-icon-right-open-big' => 'Right Open Big' ],
		[ 'entypo-icon entypo-icon-up-open-big' => 'Up Open Big' ],
		[ 'entypo-icon entypo-icon-down' => 'Down' ],
		[ 'entypo-icon entypo-icon-left' => 'Left' ],
		[ 'entypo-icon entypo-icon-right' => 'Right' ],
		[ 'entypo-icon entypo-icon-up' => 'Up' ],
		[ 'entypo-icon entypo-icon-down-dir' => 'Down Dir' ],
		[ 'entypo-icon entypo-icon-left-dir' => 'Left Dir' ],
		[ 'entypo-icon entypo-icon-right-dir' => 'Right Dir' ],
		[ 'entypo-icon entypo-icon-up-dir' => 'Up Dir' ],
		[ 'entypo-icon entypo-icon-down-bold' => 'Down Bold' ],
		[ 'entypo-icon entypo-icon-left-bold' => 'Left Bold' ],
		[ 'entypo-icon entypo-icon-right-bold' => 'Right Bold' ],
		[ 'entypo-icon entypo-icon-up-bold' => 'Up Bold' ],
		[ 'entypo-icon entypo-icon-down-thin' => 'Down Thin' ],
		[ 'entypo-icon entypo-icon-left-thin' => 'Left Thin' ],
		[ 'entypo-icon entypo-icon-right-thin' => 'Right Thin' ],
		[ 'entypo-icon entypo-icon-up-thin' => 'Up Thin' ],
		[ 'entypo-icon entypo-icon-ccw' => 'Ccw' ],
		[ 'entypo-icon entypo-icon-cw' => 'Cw' ],
		[ 'entypo-icon entypo-icon-arrows-ccw' => 'Arrows Ccw' ],
		[ 'entypo-icon entypo-icon-level-down' => 'Level Down' ],
		[ 'entypo-icon entypo-icon-level-up' => 'Level Up' ],
		[ 'entypo-icon entypo-icon-shuffle' => 'Shuffle' ],
		[ 'entypo-icon entypo-icon-loop' => 'Loop' ],
		[ 'entypo-icon entypo-icon-switch' => 'Switch' ],
		[ 'entypo-icon entypo-icon-play' => 'Play' ],
		[ 'entypo-icon entypo-icon-stop' => 'Stop' ],
		[ 'entypo-icon entypo-icon-pause' => 'Pause' ],
		[ 'entypo-icon entypo-icon-record' => 'Record' ],
		[ 'entypo-icon entypo-icon-to-end' => 'To End' ],
		[ 'entypo-icon entypo-icon-to-start' => 'To Start' ],
		[ 'entypo-icon entypo-icon-fast-forward' => 'Fast Forward' ],
		[ 'entypo-icon entypo-icon-fast-backward' => 'Fast Backward' ],
		[ 'entypo-icon entypo-icon-progress-0' => 'Progress 0' ],
		[ 'entypo-icon entypo-icon-progress-1' => 'Progress 1' ],
		[ 'entypo-icon entypo-icon-progress-2' => 'Progress 2' ],
		[ 'entypo-icon entypo-icon-progress-3' => 'Progress 3' ],
		[ 'entypo-icon entypo-icon-target' => 'Target' ],
		[ 'entypo-icon entypo-icon-palette' => 'Palette' ],
		[ 'entypo-icon entypo-icon-list' => 'List' ],
		[ 'entypo-icon entypo-icon-list-add' => 'List Add' ],
		[ 'entypo-icon entypo-icon-signal' => 'Signal' ],
		[ 'entypo-icon entypo-icon-trophy' => 'Trophy' ],
		[ 'entypo-icon entypo-icon-battery' => 'Battery' ],
		[ 'entypo-icon entypo-icon-back-in-time' => 'Back In Time' ],
		[ 'entypo-icon entypo-icon-monitor' => 'Monitor' ],
		[ 'entypo-icon entypo-icon-mobile' => 'Mobile' ],
		[ 'entypo-icon entypo-icon-network' => 'Network' ],
		[ 'entypo-icon entypo-icon-cd' => 'Cd' ],
		[ 'entypo-icon entypo-icon-inbox' => 'Inbox' ],
		[ 'entypo-icon entypo-icon-install' => 'Install' ],
		[ 'entypo-icon entypo-icon-globe' => 'Globe' ],
		[ 'entypo-icon entypo-icon-cloud' => 'Cloud' ],
		[ 'entypo-icon entypo-icon-cloud-thunder' => 'Cloud Thunder' ],
		[ 'entypo-icon entypo-icon-flash' => 'Flash' ],
		[ 'entypo-icon entypo-icon-moon' => 'Moon' ],
		[ 'entypo-icon entypo-icon-flight' => 'Flight' ],
		[ 'entypo-icon entypo-icon-paper-plane' => 'Paper Plane' ],
		[ 'entypo-icon entypo-icon-leaf' => 'Leaf' ],
		[ 'entypo-icon entypo-icon-lifebuoy' => 'Lifebuoy' ],
		[ 'entypo-icon entypo-icon-mouse' => 'Mouse' ],
		[ 'entypo-icon entypo-icon-briefcase' => 'Briefcase' ],
		[ 'entypo-icon entypo-icon-suitcase' => 'Suitcase' ],
		[ 'entypo-icon entypo-icon-dot' => 'Dot' ],
		[ 'entypo-icon entypo-icon-dot-2' => 'Dot 2' ],
		[ 'entypo-icon entypo-icon-dot-3' => 'Dot 3' ],
		[ 'entypo-icon entypo-icon-brush' => 'Brush' ],
		[ 'entypo-icon entypo-icon-magnet' => 'Magnet' ],
		[ 'entypo-icon entypo-icon-infinity' => 'Infinity' ],
		[ 'entypo-icon entypo-icon-erase' => 'Erase' ],
		[ 'entypo-icon entypo-icon-chart-pie' => 'Chart Pie' ],
		[ 'entypo-icon entypo-icon-chart-line' => 'Chart Line' ],
		[ 'entypo-icon entypo-icon-chart-bar' => 'Chart Bar' ],
		[ 'entypo-icon entypo-icon-chart-area' => 'Chart Area' ],
		[ 'entypo-icon entypo-icon-tape' => 'Tape' ],
		[ 'entypo-icon entypo-icon-graduation-cap' => 'Graduation Cap' ],
		[ 'entypo-icon entypo-icon-language' => 'Language' ],
		[ 'entypo-icon entypo-icon-ticket' => 'Ticket' ],
		[ 'entypo-icon entypo-icon-water' => 'Water' ],
		[ 'entypo-icon entypo-icon-droplet' => 'Droplet' ],
		[ 'entypo-icon entypo-icon-air' => 'Air' ],
		[ 'entypo-icon entypo-icon-credit-card' => 'Credit Card' ],
		[ 'entypo-icon entypo-icon-floppy' => 'Floppy' ],
		[ 'entypo-icon entypo-icon-clipboard' => 'Clipboard' ],
		[ 'entypo-icon entypo-icon-megaphone' => 'Megaphone' ],
		[ 'entypo-icon entypo-icon-database' => 'Database' ],
		[ 'entypo-icon entypo-icon-drive' => 'Drive' ],
		[ 'entypo-icon entypo-icon-bucket' => 'Bucket' ],
		[ 'entypo-icon entypo-icon-thermometer' => 'Thermometer' ],
		[ 'entypo-icon entypo-icon-key' => 'Key' ],
		[ 'entypo-icon entypo-icon-flow-cascade' => 'Flow Cascade' ],
		[ 'entypo-icon entypo-icon-flow-branch' => 'Flow Branch' ],
		[ 'entypo-icon entypo-icon-flow-tree' => 'Flow Tree' ],
		[ 'entypo-icon entypo-icon-flow-line' => 'Flow Line' ],
		[ 'entypo-icon entypo-icon-flow-parallel' => 'Flow Parallel' ],
		[ 'entypo-icon entypo-icon-rocket' => 'Rocket' ],
		[ 'entypo-icon entypo-icon-gauge' => 'Gauge' ],
		[ 'entypo-icon entypo-icon-traffic-cone' => 'Traffic Cone' ],
		[ 'entypo-icon entypo-icon-cc' => 'Cc' ],
		[ 'entypo-icon entypo-icon-cc-by' => 'Cc By' ],
		[ 'entypo-icon entypo-icon-cc-nc' => 'Cc Nc' ],
		[ 'entypo-icon entypo-icon-cc-nc-eu' => 'Cc Nc Eu' ],
		[ 'entypo-icon entypo-icon-cc-nc-jp' => 'Cc Nc Jp' ],
		[ 'entypo-icon entypo-icon-cc-sa' => 'Cc Sa' ],
		[ 'entypo-icon entypo-icon-cc-nd' => 'Cc Nd' ],
		[ 'entypo-icon entypo-icon-cc-pd' => 'Cc Pd' ],
		[ 'entypo-icon entypo-icon-cc-zero' => 'Cc Zero' ],
		[ 'entypo-icon entypo-icon-cc-share' => 'Cc Share' ],
		[ 'entypo-icon entypo-icon-cc-remix' => 'Cc Remix' ],
		[ 'entypo-icon entypo-icon-github' => 'Github' ],
		[ 'entypo-icon entypo-icon-github-circled' => 'Github Circled' ],
		[ 'entypo-icon entypo-icon-flickr' => 'Flickr' ],
		[ 'entypo-icon entypo-icon-flickr-circled' => 'Flickr Circled' ],
		[ 'entypo-icon entypo-icon-vimeo' => 'Vimeo' ],
		[ 'entypo-icon entypo-icon-vimeo-circled' => 'Vimeo Circled' ],
		[ 'entypo-icon entypo-icon-twitter' => 'Twitter' ],
		[ 'entypo-icon entypo-icon-twitter-circled' => 'Twitter Circled' ],
		[ 'entypo-icon entypo-icon-facebook' => 'Facebook' ],
		[ 'entypo-icon entypo-icon-facebook-circled' => 'Facebook Circled' ],
		[ 'entypo-icon entypo-icon-facebook-squared' => 'Facebook Squared' ],
		[ 'entypo-icon entypo-icon-gplus' => 'Gplus' ],
		[ 'entypo-icon entypo-icon-gplus-circled' => 'Gplus Circled' ],
		[ 'entypo-icon entypo-icon-pinterest' => 'Pinterest' ],
		[ 'entypo-icon entypo-icon-pinterest-circled' => 'Pinterest Circled' ],
		[ 'entypo-icon entypo-icon-tumblr' => 'Tumblr' ],
		[ 'entypo-icon entypo-icon-tumblr-circled' => 'Tumblr Circled' ],
		[ 'entypo-icon entypo-icon-linkedin' => 'Linkedin' ],
		[ 'entypo-icon entypo-icon-linkedin-circled' => 'Linkedin Circled' ],
		[ 'entypo-icon entypo-icon-dribbble' => 'Dribbble' ],
		[ 'entypo-icon entypo-icon-dribbble-circled' => 'Dribbble Circled' ],
		[ 'entypo-icon entypo-icon-stumbleupon' => 'Stumbleupon' ],
		[ 'entypo-icon entypo-icon-stumbleupon-circled' => 'Stumbleupon Circled' ],
		[ 'entypo-icon entypo-icon-lastfm' => 'Lastfm' ],
		[ 'entypo-icon entypo-icon-lastfm-circled' => 'Lastfm Circled' ],
		[ 'entypo-icon entypo-icon-rdio' => 'Rdio' ],
		[ 'entypo-icon entypo-icon-rdio-circled' => 'Rdio Circled' ],
		[ 'entypo-icon entypo-icon-spotify' => 'Spotify' ],
		[ 'entypo-icon entypo-icon-spotify-circled' => 'Spotify Circled' ],
		[ 'entypo-icon entypo-icon-qq' => 'Qq' ],
		[ 'entypo-icon entypo-icon-instagrem' => 'Instagrem' ],
		[ 'entypo-icon entypo-icon-dropbox' => 'Dropbox' ],
		[ 'entypo-icon entypo-icon-evernote' => 'Evernote' ],
		[ 'entypo-icon entypo-icon-flattr' => 'Flattr' ],
		[ 'entypo-icon entypo-icon-skype' => 'Skype' ],
		[ 'entypo-icon entypo-icon-skype-circled' => 'Skype Circled' ],
		[ 'entypo-icon entypo-icon-renren' => 'Renren' ],
		[ 'entypo-icon entypo-icon-sina-weibo' => 'Sina Weibo' ],
		[ 'entypo-icon entypo-icon-paypal' => 'Paypal' ],
		[ 'entypo-icon entypo-icon-picasa' => 'Picasa' ],
		[ 'entypo-icon entypo-icon-soundcloud' => 'Soundcloud' ],
		[ 'entypo-icon entypo-icon-mixi' => 'Mixi' ],
		[ 'entypo-icon entypo-icon-behance' => 'Behance' ],
		[ 'entypo-icon entypo-icon-google-circles' => 'Google Circles' ],
		[ 'entypo-icon entypo-icon-vkontakte' => 'Vkontakte' ],
		[ 'entypo-icon entypo-icon-smashing' => 'Smashing' ],
		[ 'entypo-icon entypo-icon-sweden' => 'Sweden' ],
		[ 'entypo-icon entypo-icon-db-shape' => 'Db Shape' ],
		[ 'entypo-icon entypo-icon-logo-db' => 'Logo Db' ],
	];

	return array_merge( $icons, $entypo_icons );
}

add_filter( 'vc_iconpicker-type-linecons', 'vc_iconpicker_type_linecons' );

/**
 * Linecons icons from fontello.com
 *
 * @param array $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown.
 *
 * @return array - of icons for iconpicker, can be categorized, or not.
 * @since 4.4
 */
function vc_iconpicker_type_linecons( $icons ) {
	$linecons_icons = [
		[ 'vc_li vc_li-heart' => 'Heart' ],
		[ 'vc_li vc_li-cloud' => 'Cloud' ],
		[ 'vc_li vc_li-star' => 'Star' ],
		[ 'vc_li vc_li-tv' => 'Tv' ],
		[ 'vc_li vc_li-sound' => 'Sound' ],
		[ 'vc_li vc_li-video' => 'Video' ],
		[ 'vc_li vc_li-trash' => 'Trash' ],
		[ 'vc_li vc_li-user' => 'User' ],
		[ 'vc_li vc_li-key' => 'Key' ],
		[ 'vc_li vc_li-search' => 'Search' ],
		[ 'vc_li vc_li-settings' => 'Settings' ],
		[ 'vc_li vc_li-camera' => 'Camera' ],
		[ 'vc_li vc_li-tag' => 'Tag' ],
		[ 'vc_li vc_li-lock' => 'Lock' ],
		[ 'vc_li vc_li-bulb' => 'Bulb' ],
		[ 'vc_li vc_li-pen' => 'Pen' ],
		[ 'vc_li vc_li-diamond' => 'Diamond' ],
		[ 'vc_li vc_li-display' => 'Display' ],
		[ 'vc_li vc_li-location' => 'Location' ],
		[ 'vc_li vc_li-eye' => 'Eye' ],
		[ 'vc_li vc_li-bubble' => 'Bubble' ],
		[ 'vc_li vc_li-stack' => 'Stack' ],
		[ 'vc_li vc_li-cup' => 'Cup' ],
		[ 'vc_li vc_li-phone' => 'Phone' ],
		[ 'vc_li vc_li-news' => 'News' ],
		[ 'vc_li vc_li-mail' => 'Mail' ],
		[ 'vc_li vc_li-like' => 'Like' ],
		[ 'vc_li vc_li-photo' => 'Photo' ],
		[ 'vc_li vc_li-note' => 'Note' ],
		[ 'vc_li vc_li-clock' => 'Clock' ],
		[ 'vc_li vc_li-paperplane' => 'Paperplane' ],
		[ 'vc_li vc_li-params' => 'Params' ],
		[ 'vc_li vc_li-banknote' => 'Banknote' ],
		[ 'vc_li vc_li-data' => 'Data' ],
		[ 'vc_li vc_li-music' => 'Music' ],
		[ 'vc_li vc_li-megaphone' => 'Megaphone' ],
		[ 'vc_li vc_li-study' => 'Study' ],
		[ 'vc_li vc_li-lab' => 'Lab' ],
		[ 'vc_li vc_li-food' => 'Food' ],
		[ 'vc_li vc_li-t-shirt' => 'T Shirt' ],
		[ 'vc_li vc_li-fire' => 'Fire' ],
		[ 'vc_li vc_li-clip' => 'Clip' ],
		[ 'vc_li vc_li-shop' => 'Shop' ],
		[ 'vc_li vc_li-calendar' => 'Calendar' ],
		[ 'vc_li vc_li-vallet' => 'Vallet' ],
		[ 'vc_li vc_li-vynil' => 'Vynil' ],
		[ 'vc_li vc_li-truck' => 'Truck' ],
		[ 'vc_li vc_li-world' => 'World' ],
	];

	return array_merge( $icons, $linecons_icons );
}

add_filter( 'vc_iconpicker-type-monosocial', 'vc_iconpicker_type_monosocial' );

/**
 * Monosocial icons from drinchev.github.io/monosocialiconsfont
 *
 * @param array $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown.
 *
 * @return array - of icons for iconpicker, can be categorized, or not.
 * @since 4.4
 */
function vc_iconpicker_type_monosocial( $icons ) {
	$monosocial = [
		[ 'vc-mono vc-mono-fivehundredpx' => 'Five Hundred px' ],
		[ 'vc-mono vc-mono-aboutme' => 'About me' ],
		[ 'vc-mono vc-mono-addme' => 'Add me' ],
		[ 'vc-mono vc-mono-amazon' => 'Amazon' ],
		[ 'vc-mono vc-mono-aol' => 'Aol' ],
		[ 'vc-mono vc-mono-appstorealt' => 'App-store-alt' ],
		[ 'vc-mono vc-mono-appstore' => 'Appstore' ],
		[ 'vc-mono vc-mono-apple' => 'Apple' ],
		[ 'vc-mono vc-mono-bebo' => 'Bebo' ],
		[ 'vc-mono vc-mono-behance' => 'Behance' ],
		[ 'vc-mono vc-mono-bing' => 'Bing' ],
		[ 'vc-mono vc-mono-blip' => 'Blip' ],
		[ 'vc-mono vc-mono-blogger' => 'Blogger' ],
		[ 'vc-mono vc-mono-coroflot' => 'Coroflot' ],
		[ 'vc-mono vc-mono-daytum' => 'Daytum' ],
		[ 'vc-mono vc-mono-delicious' => 'Delicious' ],
		[ 'vc-mono vc-mono-designbump' => 'Design bump' ],
		[ 'vc-mono vc-mono-designfloat' => 'Design float' ],
		[ 'vc-mono vc-mono-deviantart' => 'Deviant-art' ],
		[ 'vc-mono vc-mono-diggalt' => 'Digg-alt' ],
		[ 'vc-mono vc-mono-digg' => 'Digg' ],
		[ 'vc-mono vc-mono-dribble' => 'Dribble' ],
		[ 'vc-mono vc-mono-drupal' => 'Drupal' ],
		[ 'vc-mono vc-mono-ebay' => 'Ebay' ],
		[ 'vc-mono vc-mono-email' => 'Email' ],
		[ 'vc-mono vc-mono-emberapp' => 'Ember app' ],
		[ 'vc-mono vc-mono-etsy' => 'Etsy' ],
		[ 'vc-mono vc-mono-facebook' => 'Facebook' ],
		[ 'vc-mono vc-mono-feedburner' => 'Feed burner' ],
		[ 'vc-mono vc-mono-flickr' => 'Flickr' ],
		[ 'vc-mono vc-mono-foodspotting' => 'Food spotting' ],
		[ 'vc-mono vc-mono-forrst' => 'Forrst' ],
		[ 'vc-mono vc-mono-foursquare' => 'Fours quare' ],
		[ 'vc-mono vc-mono-friendsfeed' => 'Friends feed' ],
		[ 'vc-mono vc-mono-friendstar' => 'Friend star' ],
		[ 'vc-mono vc-mono-gdgt' => 'Gdgt' ],
		[ 'vc-mono vc-mono-github' => 'Github' ],
		[ 'vc-mono vc-mono-githubalt' => 'Github-alt' ],
		[ 'vc-mono vc-mono-googlebuzz' => 'Google buzz' ],
		[ 'vc-mono vc-mono-googleplus' => 'Google plus' ],
		[ 'vc-mono vc-mono-googletalk' => 'Google talk' ],
		[ 'vc-mono vc-mono-gowallapin' => 'Gowallapin' ],
		[ 'vc-mono vc-mono-gowalla' => 'Gowalla' ],
		[ 'vc-mono vc-mono-grooveshark' => 'Groove shark' ],
		[ 'vc-mono vc-mono-heart' => 'Heart' ],
		[ 'vc-mono vc-mono-hyves' => 'Hyves' ],
		[ 'vc-mono vc-mono-icondock' => 'Icondock' ],
		[ 'vc-mono vc-mono-icq' => 'Icq' ],
		[ 'vc-mono vc-mono-identica' => 'Identica' ],
		[ 'vc-mono vc-mono-imessage' => 'I message' ],
		[ 'vc-mono vc-mono-itunes' => 'I-tunes' ],
		[ 'vc-mono vc-mono-lastfm' => 'Lastfm' ],
		[ 'vc-mono vc-mono-linkedin' => 'Linkedin' ],
		[ 'vc-mono vc-mono-meetup' => 'Meetup' ],
		[ 'vc-mono vc-mono-metacafe' => 'Metacafe' ],
		[ 'vc-mono vc-mono-mixx' => 'Mixx' ],
		[ 'vc-mono vc-mono-mobileme' => 'Mobile me' ],
		[ 'vc-mono vc-mono-mrwong' => 'Mrwong' ],
		[ 'vc-mono vc-mono-msn' => 'Msn' ],
		[ 'vc-mono vc-mono-myspace' => 'Myspace' ],
		[ 'vc-mono vc-mono-newsvine' => 'Newsvine' ],
		[ 'vc-mono vc-mono-paypal' => 'Paypal' ],
		[ 'vc-mono vc-mono-photobucket' => 'Photo bucket' ],
		[ 'vc-mono vc-mono-picasa' => 'Picasa' ],
		[ 'vc-mono vc-mono-pinterest' => 'Pinterest' ],
		[ 'vc-mono vc-mono-podcast' => 'Podcast' ],
		[ 'vc-mono vc-mono-posterous' => 'Posterous' ],
		[ 'vc-mono vc-mono-qik' => 'Qik' ],
		[ 'vc-mono vc-mono-quora' => 'Quora' ],
		[ 'vc-mono vc-mono-reddit' => 'Reddit' ],
		[ 'vc-mono vc-mono-retweet' => 'Retweet' ],
		[ 'vc-mono vc-mono-rss' => 'Rss' ],
		[ 'vc-mono vc-mono-scribd' => 'Scribd' ],
		[ 'vc-mono vc-mono-sharethis' => 'Sharethis' ],
		[ 'vc-mono vc-mono-skype' => 'Skype' ],
		[ 'vc-mono vc-mono-slashdot' => 'Slashdot' ],
		[ 'vc-mono vc-mono-slideshare' => 'Slideshare' ],
		[ 'vc-mono vc-mono-smugmug' => 'Smugmug' ],
		[ 'vc-mono vc-mono-soundcloud' => 'Soundcloud' ],
		[ 'vc-mono vc-mono-spotify' => 'Spotify' ],
		[ 'vc-mono vc-mono-squidoo' => 'Squidoo' ],
		[ 'vc-mono vc-mono-stackoverflow' => 'Stackoverflow' ],
		[ 'vc-mono vc-mono-star' => 'Star' ],
		[ 'vc-mono vc-mono-stumbleupon' => 'Stumble upon' ],
		[ 'vc-mono vc-mono-technorati' => 'Technorati' ],
		[ 'vc-mono vc-mono-tumblr' => 'Tumblr' ],
		[ 'vc-mono vc-mono-twitterbird' => 'Twitterbird' ],
		[ 'vc-mono vc-mono-twitter' => 'Twitter' ],
		[ 'vc-mono vc-mono-viddler' => 'Viddler' ],
		[ 'vc-mono vc-mono-vimeo' => 'Vimeo' ],
		[ 'vc-mono vc-mono-virb' => 'Virb' ],
		[ 'vc-mono vc-mono-www' => 'Www' ],
		[ 'vc-mono vc-mono-wikipedia' => 'Wikipedia' ],
		[ 'vc-mono vc-mono-windows' => 'Windows' ],
		[ 'vc-mono vc-mono-wordpress' => 'WordPress' ],
		[ 'vc-mono vc-mono-xing' => 'Xing' ],
		[ 'vc-mono vc-mono-yahoobuzz' => 'Yahoo buzz' ],
		[ 'vc-mono vc-mono-yahoo' => 'Yahoo' ],
		[ 'vc-mono vc-mono-yelp' => 'Yelp' ],
		[ 'vc-mono vc-mono-youtube' => 'Youtube' ],
		[ 'vc-mono vc-mono-instagram' => 'Instagram' ],
	];

	return array_merge( $icons, $monosocial );
}

add_filter( 'vc_iconpicker-type-material', 'vc_iconpicker_type_material' );
/**
 * Material icon set from Google.
 *
 * @param array $icons
 *
 * @return array
 * @since 5.0
 */
function vc_iconpicker_type_material( $icons ) {
	$material = [
		[ 'vc-material vc-material-3d_rotation' => '3d rotation' ],
		[ 'vc-material vc-material-ac_unit' => 'ac unit' ],
		[ 'vc-material vc-material-alarm' => 'alarm' ],
		[ 'vc-material vc-material-access_alarms' => 'access alarms' ],
		[ 'vc-material vc-material-schedule' => 'schedule' ],
		[ 'vc-material vc-material-accessibility' => 'accessibility' ],
		[ 'vc-material vc-material-accessible' => 'accessible' ],
		[ 'vc-material vc-material-account_balance' => 'account balance' ],
		[ 'vc-material vc-material-account_balance_wallet' => 'account balance wallet' ],
		[ 'vc-material vc-material-account_box' => 'account box' ],
		[ 'vc-material vc-material-account_circle' => 'account circle' ],
		[ 'vc-material vc-material-adb' => 'adb' ],
		[ 'vc-material vc-material-add' => 'add' ],
		[ 'vc-material vc-material-add_a_photo' => 'add a photo' ],
		[ 'vc-material vc-material-alarm_add' => 'alarm add' ],
		[ 'vc-material vc-material-add_alert' => 'add alert' ],
		[ 'vc-material vc-material-add_box' => 'add box' ],
		[ 'vc-material vc-material-add_circle' => 'add circle' ],
		[ 'vc-material vc-material-control_point' => 'control point' ],
		[ 'vc-material vc-material-add_location' => 'add location' ],
		[ 'vc-material vc-material-add_shopping_cart' => 'add shopping cart' ],
		[ 'vc-material vc-material-queue' => 'queue' ],
		[ 'vc-material vc-material-add_to_queue' => 'add to queue' ],
		[ 'vc-material vc-material-adjust' => 'adjust' ],
		[ 'vc-material vc-material-airline_seat_flat' => 'airline seat flat' ],
		[ 'vc-material vc-material-airline_seat_flat_angled' => 'airline seat flat angled' ],
		[ 'vc-material vc-material-airline_seat_individual_suite' => 'airline seat individual suite' ],
		[ 'vc-material vc-material-airline_seat_legroom_extra' => 'airline seat legroom extra' ],
		[ 'vc-material vc-material-airline_seat_legroom_normal' => 'airline seat legroom normal' ],
		[ 'vc-material vc-material-airline_seat_legroom_reduced' => 'airline seat legroom reduced' ],
		[ 'vc-material vc-material-airline_seat_recline_extra' => 'airline seat recline extra' ],
		[ 'vc-material vc-material-airline_seat_recline_normal' => 'airline seat recline normal' ],
		[ 'vc-material vc-material-flight' => 'flight' ],
		[ 'vc-material vc-material-airplanemode_inactive' => 'airplanemode inactive' ],
		[ 'vc-material vc-material-airplay' => 'airplay' ],
		[ 'vc-material vc-material-airport_shuttle' => 'airport shuttle' ],
		[ 'vc-material vc-material-alarm_off' => 'alarm off' ],
		[ 'vc-material vc-material-alarm_on' => 'alarm on' ],
		[ 'vc-material vc-material-album' => 'album' ],
		[ 'vc-material vc-material-all_inclusive' => 'all inclusive' ],
		[ 'vc-material vc-material-all_out' => 'all out' ],
		[ 'vc-material vc-material-android' => 'android' ],
		[ 'vc-material vc-material-announcement' => 'announcement' ],
		[ 'vc-material vc-material-apps' => 'apps' ],
		[ 'vc-material vc-material-archive' => 'archive' ],
		[ 'vc-material vc-material-arrow_back' => 'arrow back' ],
		[ 'vc-material vc-material-arrow_downward' => 'arrow downward' ],
		[ 'vc-material vc-material-arrow_drop_down' => 'arrow drop down' ],
		[ 'vc-material vc-material-arrow_drop_down_circle' => 'arrow drop down circle' ],
		[ 'vc-material vc-material-arrow_drop_up' => 'arrow drop up' ],
		[ 'vc-material vc-material-arrow_forward' => 'arrow forward' ],
		[ 'vc-material vc-material-arrow_upward' => 'arrow upward' ],
		[ 'vc-material vc-material-art_track' => 'art track' ],
		[ 'vc-material vc-material-aspect_ratio' => 'aspect ratio' ],
		[ 'vc-material vc-material-poll' => 'poll' ],
		[ 'vc-material vc-material-assignment' => 'assignment' ],
		[ 'vc-material vc-material-assignment_ind' => 'assignment ind' ],
		[ 'vc-material vc-material-assignment_late' => 'assignment late' ],
		[ 'vc-material vc-material-assignment_return' => 'assignment return' ],
		[ 'vc-material vc-material-assignment_returned' => 'assignment returned' ],
		[ 'vc-material vc-material-assignment_turned_in' => 'assignment turned in' ],
		[ 'vc-material vc-material-assistant' => 'assistant' ],
		[ 'vc-material vc-material-flag' => 'flag' ],
		[ 'vc-material vc-material-attach_file' => 'attach file' ],
		[ 'vc-material vc-material-attach_money' => 'attach money' ],
		[ 'vc-material vc-material-attachment' => 'attachment' ],
		[ 'vc-material vc-material-audiotrack' => 'audiotrack' ],
		[ 'vc-material vc-material-autorenew' => 'autorenew' ],
		[ 'vc-material vc-material-av_timer' => 'av timer' ],
		[ 'vc-material vc-material-backspace' => 'backspace' ],
		[ 'vc-material vc-material-cloud_upload' => 'cloud upload' ],
		[ 'vc-material vc-material-battery_alert' => 'battery alert' ],
		[ 'vc-material vc-material-battery_charging_full' => 'battery charging full' ],
		[ 'vc-material vc-material-battery_std' => 'battery std' ],
		[ 'vc-material vc-material-battery_unknown' => 'battery unknown' ],
		[ 'vc-material vc-material-beach_access' => 'beach access' ],
		[ 'vc-material vc-material-beenhere' => 'beenhere' ],
		[ 'vc-material vc-material-block' => 'block' ],
		[ 'vc-material vc-material-bluetooth' => 'bluetooth' ],
		[ 'vc-material vc-material-bluetooth_searching' => 'bluetooth searching' ],
		[ 'vc-material vc-material-bluetooth_connected' => 'bluetooth connected' ],
		[ 'vc-material vc-material-bluetooth_disabled' => 'bluetooth disabled' ],
		[ 'vc-material vc-material-blur_circular' => 'blur circular' ],
		[ 'vc-material vc-material-blur_linear' => 'blur linear' ],
		[ 'vc-material vc-material-blur_off' => 'blur off' ],
		[ 'vc-material vc-material-blur_on' => 'blur on' ],
		[ 'vc-material vc-material-class' => 'class' ],
		[ 'vc-material vc-material-turned_in' => 'turned in' ],
		[ 'vc-material vc-material-turned_in_not' => 'turned in not' ],
		[ 'vc-material vc-material-border_all' => 'border all' ],
		[ 'vc-material vc-material-border_bottom' => 'border bottom' ],
		[ 'vc-material vc-material-border_clear' => 'border clear' ],
		[ 'vc-material vc-material-border_color' => 'border color' ],
		[ 'vc-material vc-material-border_horizontal' => 'border horizontal' ],
		[ 'vc-material vc-material-border_inner' => 'border inner' ],
		[ 'vc-material vc-material-border_left' => 'border left' ],
		[ 'vc-material vc-material-border_outer' => 'border outer' ],
		[ 'vc-material vc-material-border_right' => 'border right' ],
		[ 'vc-material vc-material-border_style' => 'border style' ],
		[ 'vc-material vc-material-border_top' => 'border top' ],
		[ 'vc-material vc-material-border_vertical' => 'border vertical' ],
		[ 'vc-material vc-material-branding_watermark' => 'branding watermark' ],
		[ 'vc-material vc-material-brightness_1' => 'brightness 1' ],
		[ 'vc-material vc-material-brightness_2' => 'brightness 2' ],
		[ 'vc-material vc-material-brightness_3' => 'brightness 3' ],
		[ 'vc-material vc-material-brightness_4' => 'brightness 4' ],
		[ 'vc-material vc-material-brightness_low' => 'brightness low' ],
		[ 'vc-material vc-material-brightness_medium' => 'brightness medium' ],
		[ 'vc-material vc-material-brightness_high' => 'brightness high' ],
		[ 'vc-material vc-material-brightness_auto' => 'brightness auto' ],
		[ 'vc-material vc-material-broken_image' => 'broken image' ],
		[ 'vc-material vc-material-brush' => 'brush' ],
		[ 'vc-material vc-material-bubble_chart' => 'bubble chart' ],
		[ 'vc-material vc-material-bug_report' => 'bug report' ],
		[ 'vc-material vc-material-build' => 'build' ],
		[ 'vc-material vc-material-burst_mode' => 'burst mode' ],
		[ 'vc-material vc-material-domain' => 'domain' ],
		[ 'vc-material vc-material-business_center' => 'business center' ],
		[ 'vc-material vc-material-cached' => 'cached' ],
		[ 'vc-material vc-material-cake' => 'cake' ],
		[ 'vc-material vc-material-phone' => 'phone' ],
		[ 'vc-material vc-material-call_end' => 'call end' ],
		[ 'vc-material vc-material-call_made' => 'call made' ],
		[ 'vc-material vc-material-merge_type' => 'merge type' ],
		[ 'vc-material vc-material-call_missed' => 'call missed' ],
		[ 'vc-material vc-material-call_missed_outgoing' => 'call missed outgoing' ],
		[ 'vc-material vc-material-call_received' => 'call received' ],
		[ 'vc-material vc-material-call_split' => 'call split' ],
		[ 'vc-material vc-material-call_to_action' => 'call to action' ],
		[ 'vc-material vc-material-camera' => 'camera' ],
		[ 'vc-material vc-material-photo_camera' => 'photo camera' ],
		[ 'vc-material vc-material-camera_enhance' => 'camera enhance' ],
		[ 'vc-material vc-material-camera_front' => 'camera front' ],
		[ 'vc-material vc-material-camera_rear' => 'camera rear' ],
		[ 'vc-material vc-material-camera_roll' => 'camera roll' ],
		[ 'vc-material vc-material-cancel' => 'cancel' ],
		[ 'vc-material vc-material-redeem' => 'redeem' ],
		[ 'vc-material vc-material-card_membership' => 'card membership' ],
		[ 'vc-material vc-material-card_travel' => 'card travel' ],
		[ 'vc-material vc-material-casino' => 'casino' ],
		[ 'vc-material vc-material-cast' => 'cast' ],
		[ 'vc-material vc-material-cast_connected' => 'cast connected' ],
		[ 'vc-material vc-material-center_focus_strong' => 'center focus strong' ],
		[ 'vc-material vc-material-center_focus_weak' => 'center focus weak' ],
		[ 'vc-material vc-material-change_history' => 'change history' ],
		[ 'vc-material vc-material-chat' => 'chat' ],
		[ 'vc-material vc-material-chat_bubble' => 'chat bubble' ],
		[ 'vc-material vc-material-chat_bubble_outline' => 'chat bubble outline' ],
		[ 'vc-material vc-material-check' => 'check' ],
		[ 'vc-material vc-material-check_box' => 'check box' ],
		[ 'vc-material vc-material-check_box_outline_blank' => 'check box outline blank' ],
		[ 'vc-material vc-material-check_circle' => 'check circle' ],
		[ 'vc-material vc-material-navigate_before' => 'navigate before' ],
		[ 'vc-material vc-material-navigate_next' => 'navigate next' ],
		[ 'vc-material vc-material-child_care' => 'child care' ],
		[ 'vc-material vc-material-child_friendly' => 'child friendly' ],
		[ 'vc-material vc-material-chrome_reader_mode' => 'chrome reader mode' ],
		[ 'vc-material vc-material-close' => 'close' ],
		[ 'vc-material vc-material-clear_all' => 'clear all' ],
		[ 'vc-material vc-material-closed_caption' => 'closed caption' ],
		[ 'vc-material vc-material-wb_cloudy' => 'wb cloudy' ],
		[ 'vc-material vc-material-cloud_circle' => 'cloud circle' ],
		[ 'vc-material vc-material-cloud_done' => 'cloud done' ],
		[ 'vc-material vc-material-cloud_download' => 'cloud download' ],
		[ 'vc-material vc-material-cloud_off' => 'cloud off' ],
		[ 'vc-material vc-material-cloud_queue' => 'cloud queue' ],
		[ 'vc-material vc-material-code' => 'code' ],
		[ 'vc-material vc-material-photo_library' => 'photo library' ],
		[ 'vc-material vc-material-collections_bookmark' => 'collections bookmark' ],
		[ 'vc-material vc-material-palette' => 'palette' ],
		[ 'vc-material vc-material-colorize' => 'colorize' ],
		[ 'vc-material vc-material-comment' => 'comment' ],
		[ 'vc-material vc-material-compare' => 'compare' ],
		[ 'vc-material vc-material-compare_arrows' => 'compare arrows' ],
		[ 'vc-material vc-material-laptop' => 'laptop' ],
		[ 'vc-material vc-material-confirmation_number' => 'confirmation number' ],
		[ 'vc-material vc-material-contact_mail' => 'contact mail' ],
		[ 'vc-material vc-material-contact_phone' => 'contact phone' ],
		[ 'vc-material vc-material-contacts' => 'contacts' ],
		[ 'vc-material vc-material-content_copy' => 'content copy' ],
		[ 'vc-material vc-material-content_cut' => 'content cut' ],
		[ 'vc-material vc-material-content_paste' => 'content paste' ],
		[ 'vc-material vc-material-control_point_duplicate' => 'control point duplicate' ],
		[ 'vc-material vc-material-copyright' => 'copyright' ],
		[ 'vc-material vc-material-mode_edit' => 'mode edit' ],
		[ 'vc-material vc-material-create_new_folder' => 'create new folder' ],
		[ 'vc-material vc-material-payment' => 'payment' ],
		[ 'vc-material vc-material-crop' => 'crop' ],
		[ 'vc-material vc-material-crop_16_9' => 'crop 16 9' ],
		[ 'vc-material vc-material-crop_3_2' => 'crop 3 2' ],
		[ 'vc-material vc-material-crop_landscape' => 'crop landscape' ],
		[ 'vc-material vc-material-crop_7_5' => 'crop 7 5' ],
		[ 'vc-material vc-material-crop_din' => 'crop din' ],
		[ 'vc-material vc-material-crop_free' => 'crop free' ],
		[ 'vc-material vc-material-crop_original' => 'crop original' ],
		[ 'vc-material vc-material-crop_portrait' => 'crop portrait' ],
		[ 'vc-material vc-material-crop_rotate' => 'crop rotate' ],
		[ 'vc-material vc-material-crop_square' => 'crop square' ],
		[ 'vc-material vc-material-dashboard' => 'dashboard' ],
		[ 'vc-material vc-material-data_usage' => 'data usage' ],
		[ 'vc-material vc-material-date_range' => 'date range' ],
		[ 'vc-material vc-material-dehaze' => 'dehaze' ],
		[ 'vc-material vc-material-delete' => 'delete' ],
		[ 'vc-material vc-material-delete_forever' => 'delete forever' ],
		[ 'vc-material vc-material-delete_sweep' => 'delete sweep' ],
		[ 'vc-material vc-material-description' => 'description' ],
		[ 'vc-material vc-material-desktop_mac' => 'desktop mac' ],
		[ 'vc-material vc-material-desktop_windows' => 'desktop windows' ],
		[ 'vc-material vc-material-details' => 'details' ],
		[ 'vc-material vc-material-developer_board' => 'developer board' ],
		[ 'vc-material vc-material-developer_mode' => 'developer mode' ],
		[ 'vc-material vc-material-device_hub' => 'device hub' ],
		[ 'vc-material vc-material-phonelink' => 'phonelink' ],
		[ 'vc-material vc-material-devices_other' => 'devices other' ],
		[ 'vc-material vc-material-dialer_sip' => 'dialer sip' ],
		[ 'vc-material vc-material-dialpad' => 'dialpad' ],
		[ 'vc-material vc-material-directions' => 'directions' ],
		[ 'vc-material vc-material-directions_bike' => 'directions bike' ],
		[ 'vc-material vc-material-directions_boat' => 'directions boat' ],
		[ 'vc-material vc-material-directions_bus' => 'directions bus' ],
		[ 'vc-material vc-material-directions_car' => 'directions car' ],
		[ 'vc-material vc-material-directions_railway' => 'directions railway' ],
		[ 'vc-material vc-material-directions_run' => 'directions run' ],
		[ 'vc-material vc-material-directions_transit' => 'directions transit' ],
		[ 'vc-material vc-material-directions_walk' => 'directions walk' ],
		[ 'vc-material vc-material-disc_full' => 'disc full' ],
		[ 'vc-material vc-material-dns' => 'dns' ],
		[ 'vc-material vc-material-not_interested' => 'not interested' ],
		[ 'vc-material vc-material-do_not_disturb_alt' => 'do not disturb alt' ],
		[ 'vc-material vc-material-do_not_disturb_off' => 'do not disturb off' ],
		[ 'vc-material vc-material-remove_circle' => 'remove circle' ],
		[ 'vc-material vc-material-dock' => 'dock' ],
		[ 'vc-material vc-material-done' => 'done' ],
		[ 'vc-material vc-material-done_all' => 'done all' ],
		[ 'vc-material vc-material-donut_large' => 'donut large' ],
		[ 'vc-material vc-material-donut_small' => 'donut small' ],
		[ 'vc-material vc-material-drafts' => 'drafts' ],
		[ 'vc-material vc-material-drag_handle' => 'drag handle' ],
		[ 'vc-material vc-material-time_to_leave' => 'time to leave' ],
		[ 'vc-material vc-material-dvr' => 'dvr' ],
		[ 'vc-material vc-material-edit_location' => 'edit location' ],
		[ 'vc-material vc-material-eject' => 'eject' ],
		[ 'vc-material vc-material-markunread' => 'markunread' ],
		[ 'vc-material vc-material-enhanced_encryption' => 'enhanced encryption' ],
		[ 'vc-material vc-material-equalizer' => 'equalizer' ],
		[ 'vc-material vc-material-error' => 'error' ],
		[ 'vc-material vc-material-error_outline' => 'error outline' ],
		[ 'vc-material vc-material-euro_symbol' => 'euro symbol' ],
		[ 'vc-material vc-material-ev_station' => 'ev station' ],
		[ 'vc-material vc-material-insert_invitation' => 'insert invitation' ],
		[ 'vc-material vc-material-event_available' => 'event available' ],
		[ 'vc-material vc-material-event_busy' => 'event busy' ],
		[ 'vc-material vc-material-event_note' => 'event note' ],
		[ 'vc-material vc-material-event_seat' => 'event seat' ],
		[ 'vc-material vc-material-exit_to_app' => 'exit to app' ],
		[ 'vc-material vc-material-expand_less' => 'expand less' ],
		[ 'vc-material vc-material-expand_more' => 'expand more' ],
		[ 'vc-material vc-material-explicit' => 'explicit' ],
		[ 'vc-material vc-material-explore' => 'explore' ],
		[ 'vc-material vc-material-exposure' => 'exposure' ],
		[ 'vc-material vc-material-exposure_neg_1' => 'exposure neg 1' ],
		[ 'vc-material vc-material-exposure_neg_2' => 'exposure neg 2' ],
		[ 'vc-material vc-material-exposure_plus_1' => 'exposure plus 1' ],
		[ 'vc-material vc-material-exposure_plus_2' => 'exposure plus 2' ],
		[ 'vc-material vc-material-exposure_zero' => 'exposure zero' ],
		[ 'vc-material vc-material-extension' => 'extension' ],
		[ 'vc-material vc-material-face' => 'face' ],
		[ 'vc-material vc-material-fast_forward' => 'fast forward' ],
		[ 'vc-material vc-material-fast_rewind' => 'fast rewind' ],
		[ 'vc-material vc-material-favorite' => 'favorite' ],
		[ 'vc-material vc-material-favorite_border' => 'favorite border' ],
		[ 'vc-material vc-material-featured_play_list' => 'featured play list' ],
		[ 'vc-material vc-material-featured_video' => 'featured video' ],
		[ 'vc-material vc-material-sms_failed' => 'sms failed' ],
		[ 'vc-material vc-material-fiber_dvr' => 'fiber dvr' ],
		[ 'vc-material vc-material-fiber_manual_record' => 'fiber manual record' ],
		[ 'vc-material vc-material-fiber_new' => 'fiber new' ],
		[ 'vc-material vc-material-fiber_pin' => 'fiber pin' ],
		[ 'vc-material vc-material-fiber_smart_record' => 'fiber smart record' ],
		[ 'vc-material vc-material-get_app' => 'get app' ],
		[ 'vc-material vc-material-file_upload' => 'file upload' ],
		[ 'vc-material vc-material-filter' => 'filter' ],
		[ 'vc-material vc-material-filter_1' => 'filter 1' ],
		[ 'vc-material vc-material-filter_2' => 'filter 2' ],
		[ 'vc-material vc-material-filter_3' => 'filter 3' ],
		[ 'vc-material vc-material-filter_4' => 'filter 4' ],
		[ 'vc-material vc-material-filter_5' => 'filter 5' ],
		[ 'vc-material vc-material-filter_6' => 'filter 6' ],
		[ 'vc-material vc-material-filter_7' => 'filter 7' ],
		[ 'vc-material vc-material-filter_8' => 'filter 8' ],
		[ 'vc-material vc-material-filter_9' => 'filter 9' ],
		[ 'vc-material vc-material-filter_9_plus' => 'filter 9 plus' ],
		[ 'vc-material vc-material-filter_b_and_w' => 'filter b and w' ],
		[ 'vc-material vc-material-filter_center_focus' => 'filter center focus' ],
		[ 'vc-material vc-material-filter_drama' => 'filter drama' ],
		[ 'vc-material vc-material-filter_frames' => 'filter frames' ],
		[ 'vc-material vc-material-terrain' => 'terrain' ],
		[ 'vc-material vc-material-filter_list' => 'filter list' ],
		[ 'vc-material vc-material-filter_none' => 'filter none' ],
		[ 'vc-material vc-material-filter_tilt_shift' => 'filter tilt shift' ],
		[ 'vc-material vc-material-filter_vintage' => 'filter vintage' ],
		[ 'vc-material vc-material-find_in_page' => 'find in page' ],
		[ 'vc-material vc-material-find_replace' => 'find replace' ],
		[ 'vc-material vc-material-fingerprint' => 'fingerprint' ],
		[ 'vc-material vc-material-first_page' => 'first page' ],
		[ 'vc-material vc-material-fitness_center' => 'fitness center' ],
		[ 'vc-material vc-material-flare' => 'flare' ],
		[ 'vc-material vc-material-flash_auto' => 'flash auto' ],
		[ 'vc-material vc-material-flash_off' => 'flash off' ],
		[ 'vc-material vc-material-flash_on' => 'flash on' ],
		[ 'vc-material vc-material-flight_land' => 'flight land' ],
		[ 'vc-material vc-material-flight_takeoff' => 'flight takeoff' ],
		[ 'vc-material vc-material-flip' => 'flip' ],
		[ 'vc-material vc-material-flip_to_back' => 'flip to back' ],
		[ 'vc-material vc-material-flip_to_front' => 'flip to front' ],
		[ 'vc-material vc-material-folder' => 'folder' ],
		[ 'vc-material vc-material-folder_open' => 'folder open' ],
		[ 'vc-material vc-material-folder_shared' => 'folder shared' ],
		[ 'vc-material vc-material-folder_special' => 'folder special' ],
		[ 'vc-material vc-material-font_download' => 'font download' ],
		[ 'vc-material vc-material-format_align_center' => 'format align center' ],
		[ 'vc-material vc-material-format_align_justify' => 'format align justify' ],
		[ 'vc-material vc-material-format_align_left' => 'format align left' ],
		[ 'vc-material vc-material-format_align_right' => 'format align right' ],
		[ 'vc-material vc-material-format_bold' => 'format bold' ],
		[ 'vc-material vc-material-format_clear' => 'format clear' ],
		[ 'vc-material vc-material-format_color_fill' => 'format color fill' ],
		[ 'vc-material vc-material-format_color_reset' => 'format color reset' ],
		[ 'vc-material vc-material-format_color_text' => 'format color text' ],
		[ 'vc-material vc-material-format_indent_decrease' => 'format indent decrease' ],
		[ 'vc-material vc-material-format_indent_increase' => 'format indent increase' ],
		[ 'vc-material vc-material-format_italic' => 'format italic' ],
		[ 'vc-material vc-material-format_line_spacing' => 'format line spacing' ],
		[ 'vc-material vc-material-format_list_bulleted' => 'format list bulleted' ],
		[ 'vc-material vc-material-format_list_numbered' => 'format list numbered' ],
		[ 'vc-material vc-material-format_paint' => 'format paint' ],
		[ 'vc-material vc-material-format_quote' => 'format quote' ],
		[ 'vc-material vc-material-format_shapes' => 'format shapes' ],
		[ 'vc-material vc-material-format_size' => 'format size' ],
		[ 'vc-material vc-material-format_strikethrough' => 'format strikethrough' ],
		[ 'vc-material vc-material-format_textdirection_l_to_r' => 'format textdirection l to r' ],
		[ 'vc-material vc-material-format_textdirection_r_to_l' => 'format textdirection r to l' ],
		[ 'vc-material vc-material-format_underlined' => 'format underlined' ],
		[ 'vc-material vc-material-question_answer' => 'question answer' ],
		[ 'vc-material vc-material-forward' => 'forward' ],
		[ 'vc-material vc-material-forward_10' => 'forward 10' ],
		[ 'vc-material vc-material-forward_30' => 'forward 30' ],
		[ 'vc-material vc-material-forward_5' => 'forward 5' ],
		[ 'vc-material vc-material-free_breakfast' => 'free breakfast' ],
		[ 'vc-material vc-material-fullscreen' => 'fullscreen' ],
		[ 'vc-material vc-material-fullscreen_exit' => 'fullscreen exit' ],
		[ 'vc-material vc-material-functions' => 'functions' ],
		[ 'vc-material vc-material-g_translate' => 'g translate' ],
		[ 'vc-material vc-material-games' => 'games' ],
		[ 'vc-material vc-material-gavel' => 'gavel' ],
		[ 'vc-material vc-material-gesture' => 'gesture' ],
		[ 'vc-material vc-material-gif' => 'gif' ],
		[ 'vc-material vc-material-goat' => 'goat' ],
		[ 'vc-material vc-material-golf_course' => 'golf course' ],
		[ 'vc-material vc-material-my_location' => 'my location' ],
		[ 'vc-material vc-material-location_searching' => 'location searching' ],
		[ 'vc-material vc-material-location_disabled' => 'location disabled' ],
		[ 'vc-material vc-material-star' => 'star' ],
		[ 'vc-material vc-material-gradient' => 'gradient' ],
		[ 'vc-material vc-material-grain' => 'grain' ],
		[ 'vc-material vc-material-graphic_eq' => 'graphic eq' ],
		[ 'vc-material vc-material-grid_off' => 'grid off' ],
		[ 'vc-material vc-material-grid_on' => 'grid on' ],
		[ 'vc-material vc-material-people' => 'people' ],
		[ 'vc-material vc-material-group_add' => 'group add' ],
		[ 'vc-material vc-material-group_work' => 'group work' ],
		[ 'vc-material vc-material-hd' => 'hd' ],
		[ 'vc-material vc-material-hdr_off' => 'hdr off' ],
		[ 'vc-material vc-material-hdr_on' => 'hdr on' ],
		[ 'vc-material vc-material-hdr_strong' => 'hdr strong' ],
		[ 'vc-material vc-material-hdr_weak' => 'hdr weak' ],
		[ 'vc-material vc-material-headset' => 'headset' ],
		[ 'vc-material vc-material-headset_mic' => 'headset mic' ],
		[ 'vc-material vc-material-healing' => 'healing' ],
		[ 'vc-material vc-material-hearing' => 'hearing' ],
		[ 'vc-material vc-material-help' => 'help' ],
		[ 'vc-material vc-material-help_outline' => 'help outline' ],
		[ 'vc-material vc-material-high_quality' => 'high quality' ],
		[ 'vc-material vc-material-highlight' => 'highlight' ],
		[ 'vc-material vc-material-highlight_off' => 'highlight off' ],
		[ 'vc-material vc-material-restore' => 'restore' ],
		[ 'vc-material vc-material-home' => 'home' ],
		[ 'vc-material vc-material-hot_tub' => 'hot tub' ],
		[ 'vc-material vc-material-local_hotel' => 'local hotel' ],
		[ 'vc-material vc-material-hourglass_empty' => 'hourglass empty' ],
		[ 'vc-material vc-material-hourglass_full' => 'hourglass full' ],
		[ 'vc-material vc-material-http' => 'http' ],
		[ 'vc-material vc-material-lock' => 'lock' ],
		[ 'vc-material vc-material-photo' => 'photo' ],
		[ 'vc-material vc-material-image_aspect_ratio' => 'image aspect ratio' ],
		[ 'vc-material vc-material-import_contacts' => 'import contacts' ],
		[ 'vc-material vc-material-import_export' => 'import export' ],
		[ 'vc-material vc-material-important_devices' => 'important devices' ],
		[ 'vc-material vc-material-inbox' => 'inbox' ],
		[ 'vc-material vc-material-indeterminate_check_box' => 'indeterminate check box' ],
		[ 'vc-material vc-material-info' => 'info' ],
		[ 'vc-material vc-material-info_outline' => 'info outline' ],
		[ 'vc-material vc-material-input' => 'input' ],
		[ 'vc-material vc-material-insert_comment' => 'insert comment' ],
		[ 'vc-material vc-material-insert_drive_file' => 'insert drive file' ],
		[ 'vc-material vc-material-tag_faces' => 'tag faces' ],
		[ 'vc-material vc-material-link' => 'link' ],
		[ 'vc-material vc-material-invert_colors' => 'invert colors' ],
		[ 'vc-material vc-material-invert_colors_off' => 'invert colors off' ],
		[ 'vc-material vc-material-iso' => 'iso' ],
		[ 'vc-material vc-material-keyboard' => 'keyboard' ],
		[ 'vc-material vc-material-keyboard_arrow_down' => 'keyboard arrow down' ],
		[ 'vc-material vc-material-keyboard_arrow_left' => 'keyboard arrow left' ],
		[ 'vc-material vc-material-keyboard_arrow_right' => 'keyboard arrow right' ],
		[ 'vc-material vc-material-keyboard_arrow_up' => 'keyboard arrow up' ],
		[ 'vc-material vc-material-keyboard_backspace' => 'keyboard backspace' ],
		[ 'vc-material vc-material-keyboard_capslock' => 'keyboard capslock' ],
		[ 'vc-material vc-material-keyboard_hide' => 'keyboard hide' ],
		[ 'vc-material vc-material-keyboard_return' => 'keyboard return' ],
		[ 'vc-material vc-material-keyboard_tab' => 'keyboard tab' ],
		[ 'vc-material vc-material-keyboard_voice' => 'keyboard voice' ],
		[ 'vc-material vc-material-kitchen' => 'kitchen' ],
		[ 'vc-material vc-material-label' => 'label' ],
		[ 'vc-material vc-material-label_outline' => 'label outline' ],
		[ 'vc-material vc-material-language' => 'language' ],
		[ 'vc-material vc-material-laptop_chromebook' => 'laptop chromebook' ],
		[ 'vc-material vc-material-laptop_mac' => 'laptop mac' ],
		[ 'vc-material vc-material-laptop_windows' => 'laptop windows' ],
		[ 'vc-material vc-material-last_page' => 'last page' ],
		[ 'vc-material vc-material-open_in_new' => 'open in new' ],
		[ 'vc-material vc-material-layers' => 'layers' ],
		[ 'vc-material vc-material-layers_clear' => 'layers clear' ],
		[ 'vc-material vc-material-leak_add' => 'leak add' ],
		[ 'vc-material vc-material-leak_remove' => 'leak remove' ],
		[ 'vc-material vc-material-lens' => 'lens' ],
		[ 'vc-material vc-material-library_books' => 'library books' ],
		[ 'vc-material vc-material-library_music' => 'library music' ],
		[ 'vc-material vc-material-lightbulb_outline' => 'lightbulb outline' ],
		[ 'vc-material vc-material-line_style' => 'line style' ],
		[ 'vc-material vc-material-line_weight' => 'line weight' ],
		[ 'vc-material vc-material-linear_scale' => 'linear scale' ],
		[ 'vc-material vc-material-linked_camera' => 'linked camera' ],
		[ 'vc-material vc-material-list' => 'list' ],
		[ 'vc-material vc-material-live_help' => 'live help' ],
		[ 'vc-material vc-material-live_tv' => 'live tv' ],
		[ 'vc-material vc-material-local_play' => 'local play' ],
		[ 'vc-material vc-material-local_airport' => 'local airport' ],
		[ 'vc-material vc-material-local_atm' => 'local atm' ],
		[ 'vc-material vc-material-local_bar' => 'local bar' ],
		[ 'vc-material vc-material-local_cafe' => 'local cafe' ],
		[ 'vc-material vc-material-local_car_wash' => 'local car wash' ],
		[ 'vc-material vc-material-local_convenience_store' => 'local convenience store' ],
		[ 'vc-material vc-material-restaurant_menu' => 'restaurant menu' ],
		[ 'vc-material vc-material-local_drink' => 'local drink' ],
		[ 'vc-material vc-material-local_florist' => 'local florist' ],
		[ 'vc-material vc-material-local_gas_station' => 'local gas station' ],
		[ 'vc-material vc-material-shopping_cart' => 'shopping cart' ],
		[ 'vc-material vc-material-local_hospital' => 'local hospital' ],
		[ 'vc-material vc-material-local_laundry_service' => 'local laundry service' ],
		[ 'vc-material vc-material-local_library' => 'local library' ],
		[ 'vc-material vc-material-local_mall' => 'local mall' ],
		[ 'vc-material vc-material-theaters' => 'theaters' ],
		[ 'vc-material vc-material-local_offer' => 'local offer' ],
		[ 'vc-material vc-material-local_parking' => 'local parking' ],
		[ 'vc-material vc-material-local_pharmacy' => 'local pharmacy' ],
		[ 'vc-material vc-material-local_pizza' => 'local pizza' ],
		[ 'vc-material vc-material-print' => 'print' ],
		[ 'vc-material vc-material-local_shipping' => 'local shipping' ],
		[ 'vc-material vc-material-local_taxi' => 'local taxi' ],
		[ 'vc-material vc-material-location_city' => 'location city' ],
		[ 'vc-material vc-material-location_off' => 'location off' ],
		[ 'vc-material vc-material-room' => 'room' ],
		[ 'vc-material vc-material-lock_open' => 'lock open' ],
		[ 'vc-material vc-material-lock_outline' => 'lock outline' ],
		[ 'vc-material vc-material-looks' => 'looks' ],
		[ 'vc-material vc-material-looks_3' => 'looks 3' ],
		[ 'vc-material vc-material-looks_4' => 'looks 4' ],
		[ 'vc-material vc-material-looks_5' => 'looks 5' ],
		[ 'vc-material vc-material-looks_6' => 'looks 6' ],
		[ 'vc-material vc-material-looks_one' => 'looks one' ],
		[ 'vc-material vc-material-looks_two' => 'looks two' ],
		[ 'vc-material vc-material-sync' => 'sync' ],
		[ 'vc-material vc-material-loupe' => 'loupe' ],
		[ 'vc-material vc-material-low_priority' => 'low priority' ],
		[ 'vc-material vc-material-loyalty' => 'loyalty' ],
		[ 'vc-material vc-material-mail_outline' => 'mail outline' ],
		[ 'vc-material vc-material-map' => 'map' ],
		[ 'vc-material vc-material-markunread_mailbox' => 'markunread mailbox' ],
		[ 'vc-material vc-material-memory' => 'memory' ],
		[ 'vc-material vc-material-menu' => 'menu' ],
		[ 'vc-material vc-material-message' => 'message' ],
		[ 'vc-material vc-material-mic' => 'mic' ],
		[ 'vc-material vc-material-mic_none' => 'mic none' ],
		[ 'vc-material vc-material-mic_off' => 'mic off' ],
		[ 'vc-material vc-material-mms' => 'mms' ],
		[ 'vc-material vc-material-mode_comment' => 'mode comment' ],
		[ 'vc-material vc-material-monetization_on' => 'monetization on' ],
		[ 'vc-material vc-material-money_off' => 'money off' ],
		[ 'vc-material vc-material-monochrome_photos' => 'monochrome photos' ],
		[ 'vc-material vc-material-mood_bad' => 'mood bad' ],
		[ 'vc-material vc-material-more' => 'more' ],
		[ 'vc-material vc-material-more_horiz' => 'more horiz' ],
		[ 'vc-material vc-material-more_vert' => 'more vert' ],
		[ 'vc-material vc-material-motorcycle' => 'motorcycle' ],
		[ 'vc-material vc-material-mouse' => 'mouse' ],
		[ 'vc-material vc-material-move_to_inbox' => 'move to inbox' ],
		[ 'vc-material vc-material-movie_creation' => 'movie creation' ],
		[ 'vc-material vc-material-movie_filter' => 'movie filter' ],
		[ 'vc-material vc-material-multiline_chart' => 'multiline chart' ],
		[ 'vc-material vc-material-music_note' => 'music note' ],
		[ 'vc-material vc-material-music_video' => 'music video' ],
		[ 'vc-material vc-material-nature' => 'nature' ],
		[ 'vc-material vc-material-nature_people' => 'nature people' ],
		[ 'vc-material vc-material-navigation' => 'navigation' ],
		[ 'vc-material vc-material-near_me' => 'near me' ],
		[ 'vc-material vc-material-network_cell' => 'network cell' ],
		[ 'vc-material vc-material-network_check' => 'network check' ],
		[ 'vc-material vc-material-network_locked' => 'network locked' ],
		[ 'vc-material vc-material-network_wifi' => 'network wifi' ],
		[ 'vc-material vc-material-new_releases' => 'new releases' ],
		[ 'vc-material vc-material-next_week' => 'next week' ],
		[ 'vc-material vc-material-nfc' => 'nfc' ],
		[ 'vc-material vc-material-no_encryption' => 'no encryption' ],
		[ 'vc-material vc-material-signal_cellular_no_sim' => 'signal cellular no sim' ],
		[ 'vc-material vc-material-note' => 'note' ],
		[ 'vc-material vc-material-note_add' => 'note add' ],
		[ 'vc-material vc-material-notifications' => 'notifications' ],
		[ 'vc-material vc-material-notifications_active' => 'notifications active' ],
		[ 'vc-material vc-material-notifications_none' => 'notifications none' ],
		[ 'vc-material vc-material-notifications_off' => 'notifications off' ],
		[ 'vc-material vc-material-notifications_paused' => 'notifications paused' ],
		[ 'vc-material vc-material-offline_pin' => 'offline pin' ],
		[ 'vc-material vc-material-ondemand_video' => 'ondemand video' ],
		[ 'vc-material vc-material-opacity' => 'opacity' ],
		[ 'vc-material vc-material-open_in_browser' => 'open in browser' ],
		[ 'vc-material vc-material-open_with' => 'open with' ],
		[ 'vc-material vc-material-pages' => 'pages' ],
		[ 'vc-material vc-material-pageview' => 'pageview' ],
		[ 'vc-material vc-material-pan_tool' => 'pan tool' ],
		[ 'vc-material vc-material-panorama' => 'panorama' ],
		[ 'vc-material vc-material-radio_button_unchecked' => 'radio button unchecked' ],
		[ 'vc-material vc-material-panorama_horizontal' => 'panorama horizontal' ],
		[ 'vc-material vc-material-panorama_vertical' => 'panorama vertical' ],
		[ 'vc-material vc-material-panorama_wide_angle' => 'panorama wide angle' ],
		[ 'vc-material vc-material-party_mode' => 'party mode' ],
		[ 'vc-material vc-material-pause' => 'pause' ],
		[ 'vc-material vc-material-pause_circle_filled' => 'pause circle filled' ],
		[ 'vc-material vc-material-pause_circle_outline' => 'pause circle outline' ],
		[ 'vc-material vc-material-people_outline' => 'people outline' ],
		[ 'vc-material vc-material-perm_camera_mic' => 'perm camera mic' ],
		[ 'vc-material vc-material-perm_contact_calendar' => 'perm contact calendar' ],
		[ 'vc-material vc-material-perm_data_setting' => 'perm data setting' ],
		[ 'vc-material vc-material-perm_device_information' => 'perm device information' ],
		[ 'vc-material vc-material-person_outline' => 'person outline' ],
		[ 'vc-material vc-material-perm_media' => 'perm media' ],
		[ 'vc-material vc-material-perm_phone_msg' => 'perm phone msg' ],
		[ 'vc-material vc-material-perm_scan_wifi' => 'perm scan wifi' ],
		[ 'vc-material vc-material-person' => 'person' ],
		[ 'vc-material vc-material-person_add' => 'person add' ],
		[ 'vc-material vc-material-person_pin' => 'person pin' ],
		[ 'vc-material vc-material-person_pin_circle' => 'person pin circle' ],
		[ 'vc-material vc-material-personal_video' => 'personal video' ],
		[ 'vc-material vc-material-pets' => 'pets' ],
		[ 'vc-material vc-material-phone_android' => 'phone android' ],
		[ 'vc-material vc-material-phone_bluetooth_speaker' => 'phone bluetooth speaker' ],
		[ 'vc-material vc-material-phone_forwarded' => 'phone forwarded' ],
		[ 'vc-material vc-material-phone_in_talk' => 'phone in talk' ],
		[ 'vc-material vc-material-phone_iphone' => 'phone iphone' ],
		[ 'vc-material vc-material-phone_locked' => 'phone locked' ],
		[ 'vc-material vc-material-phone_missed' => 'phone missed' ],
		[ 'vc-material vc-material-phone_paused' => 'phone paused' ],
		[ 'vc-material vc-material-phonelink_erase' => 'phonelink erase' ],
		[ 'vc-material vc-material-phonelink_lock' => 'phonelink lock' ],
		[ 'vc-material vc-material-phonelink_off' => 'phonelink off' ],
		[ 'vc-material vc-material-phonelink_ring' => 'phonelink ring' ],
		[ 'vc-material vc-material-phonelink_setup' => 'phonelink setup' ],
		[ 'vc-material vc-material-photo_album' => 'photo album' ],
		[ 'vc-material vc-material-photo_filter' => 'photo filter' ],
		[ 'vc-material vc-material-photo_size_select_actual' => 'photo size select actual' ],
		[ 'vc-material vc-material-photo_size_select_large' => 'photo size select large' ],
		[ 'vc-material vc-material-photo_size_select_small' => 'photo size select small' ],
		[ 'vc-material vc-material-picture_as_pdf' => 'picture as pdf' ],
		[ 'vc-material vc-material-picture_in_picture' => 'picture in picture' ],
		[ 'vc-material vc-material-picture_in_picture_alt' => 'picture in picture alt' ],
		[ 'vc-material vc-material-pie_chart' => 'pie chart' ],
		[ 'vc-material vc-material-pie_chart_outlined' => 'pie chart outlined' ],
		[ 'vc-material vc-material-pin_drop' => 'pin drop' ],
		[ 'vc-material vc-material-play_arrow' => 'play arrow' ],
		[ 'vc-material vc-material-play_circle_filled' => 'play circle filled' ],
		[ 'vc-material vc-material-play_circle_outline' => 'play circle outline' ],
		[ 'vc-material vc-material-play_for_work' => 'play for work' ],
		[ 'vc-material vc-material-playlist_add' => 'playlist add' ],
		[ 'vc-material vc-material-playlist_add_check' => 'playlist add check' ],
		[ 'vc-material vc-material-playlist_play' => 'playlist play' ],
		[ 'vc-material vc-material-plus_one' => 'plus one' ],
		[ 'vc-material vc-material-polymer' => 'polymer' ],
		[ 'vc-material vc-material-pool' => 'pool' ],
		[ 'vc-material vc-material-portable_wifi_off' => 'portable wifi off' ],
		[ 'vc-material vc-material-portrait' => 'portrait' ],
		[ 'vc-material vc-material-power' => 'power' ],
		[ 'vc-material vc-material-power_input' => 'power input' ],
		[ 'vc-material vc-material-power_settings_new' => 'power settings new' ],
		[ 'vc-material vc-material-pregnant_woman' => 'pregnant woman' ],
		[ 'vc-material vc-material-present_to_all' => 'present to all' ],
		[ 'vc-material vc-material-priority_high' => 'priority high' ],
		[ 'vc-material vc-material-public' => 'public' ],
		[ 'vc-material vc-material-publish' => 'publish' ],
		[ 'vc-material vc-material-queue_music' => 'queue music' ],
		[ 'vc-material vc-material-queue_play_next' => 'queue play next' ],
		[ 'vc-material vc-material-radio' => 'radio' ],
		[ 'vc-material vc-material-radio_button_checked' => 'radio button checked' ],
		[ 'vc-material vc-material-rate_review' => 'rate review' ],
		[ 'vc-material vc-material-receipt' => 'receipt' ],
		[ 'vc-material vc-material-recent_actors' => 'recent actors' ],
		[ 'vc-material vc-material-record_voice_over' => 'record voice over' ],
		[ 'vc-material vc-material-redo' => 'redo' ],
		[ 'vc-material vc-material-refresh' => 'refresh' ],
		[ 'vc-material vc-material-remove' => 'remove' ],
		[ 'vc-material vc-material-remove_circle_outline' => 'remove circle outline' ],
		[ 'vc-material vc-material-remove_from_queue' => 'remove from queue' ],
		[ 'vc-material vc-material-visibility' => 'visibility' ],
		[ 'vc-material vc-material-remove_shopping_cart' => 'remove shopping cart' ],
		[ 'vc-material vc-material-reorder' => 'reorder' ],
		[ 'vc-material vc-material-repeat' => 'repeat' ],
		[ 'vc-material vc-material-repeat_one' => 'repeat one' ],
		[ 'vc-material vc-material-replay' => 'replay' ],
		[ 'vc-material vc-material-replay_10' => 'replay 10' ],
		[ 'vc-material vc-material-replay_30' => 'replay 30' ],
		[ 'vc-material vc-material-replay_5' => 'replay 5' ],
		[ 'vc-material vc-material-reply' => 'reply' ],
		[ 'vc-material vc-material-reply_all' => 'reply all' ],
		[ 'vc-material vc-material-report' => 'report' ],
		[ 'vc-material vc-material-warning' => 'warning' ],
		[ 'vc-material vc-material-restaurant' => 'restaurant' ],
		[ 'vc-material vc-material-restore_page' => 'restore page' ],
		[ 'vc-material vc-material-ring_volume' => 'ring volume' ],
		[ 'vc-material vc-material-room_service' => 'room service' ],
		[ 'vc-material vc-material-rotate_90_degrees_ccw' => 'rotate 90 degrees ccw' ],
		[ 'vc-material vc-material-rotate_left' => 'rotate left' ],
		[ 'vc-material vc-material-rotate_right' => 'rotate right' ],
		[ 'vc-material vc-material-rounded_corner' => 'rounded corner' ],
		[ 'vc-material vc-material-router' => 'router' ],
		[ 'vc-material vc-material-rowing' => 'rowing' ],
		[ 'vc-material vc-material-rss_feed' => 'rss feed' ],
		[ 'vc-material vc-material-rv_hookup' => 'rv hookup' ],
		[ 'vc-material vc-material-satellite' => 'satellite' ],
		[ 'vc-material vc-material-save' => 'save' ],
		[ 'vc-material vc-material-scanner' => 'scanner' ],
		[ 'vc-material vc-material-school' => 'school' ],
		[ 'vc-material vc-material-screen_lock_landscape' => 'screen lock landscape' ],
		[ 'vc-material vc-material-screen_lock_portrait' => 'screen lock portrait' ],
		[ 'vc-material vc-material-screen_lock_rotation' => 'screen lock rotation' ],
		[ 'vc-material vc-material-screen_rotation' => 'screen rotation' ],
		[ 'vc-material vc-material-screen_share' => 'screen share' ],
		[ 'vc-material vc-material-sd_storage' => 'sd storage' ],
		[ 'vc-material vc-material-search' => 'search' ],
		[ 'vc-material vc-material-security' => 'security' ],
		[ 'vc-material vc-material-select_all' => 'select all' ],
		[ 'vc-material vc-material-send' => 'send' ],
		[ 'vc-material vc-material-sentiment_dissatisfied' => 'sentiment dissatisfied' ],
		[ 'vc-material vc-material-sentiment_neutral' => 'sentiment neutral' ],
		[ 'vc-material vc-material-sentiment_satisfied' => 'sentiment satisfied' ],
		[ 'vc-material vc-material-sentiment_very_dissatisfied' => 'sentiment very dissatisfied' ],
		[ 'vc-material vc-material-sentiment_very_satisfied' => 'sentiment very satisfied' ],
		[ 'vc-material vc-material-settings' => 'settings' ],
		[ 'vc-material vc-material-settings_applications' => 'settings applications' ],
		[ 'vc-material vc-material-settings_backup_restore' => 'settings backup restore' ],
		[ 'vc-material vc-material-settings_bluetooth' => 'settings bluetooth' ],
		[ 'vc-material vc-material-settings_brightness' => 'settings brightness' ],
		[ 'vc-material vc-material-settings_cell' => 'settings cell' ],
		[ 'vc-material vc-material-settings_ethernet' => 'settings ethernet' ],
		[ 'vc-material vc-material-settings_input_antenna' => 'settings input antenna' ],
		[ 'vc-material vc-material-settings_input_composite' => 'settings input composite' ],
		[ 'vc-material vc-material-settings_input_hdmi' => 'settings input hdmi' ],
		[ 'vc-material vc-material-settings_input_svideo' => 'settings input svideo' ],
		[ 'vc-material vc-material-settings_overscan' => 'settings overscan' ],
		[ 'vc-material vc-material-settings_phone' => 'settings phone' ],
		[ 'vc-material vc-material-settings_power' => 'settings power' ],
		[ 'vc-material vc-material-settings_remote' => 'settings remote' ],
		[ 'vc-material vc-material-settings_system_daydream' => 'settings system daydream' ],
		[ 'vc-material vc-material-settings_voice' => 'settings voice' ],
		[ 'vc-material vc-material-share' => 'share' ],
		[ 'vc-material vc-material-shop' => 'shop' ],
		[ 'vc-material vc-material-shop_two' => 'shop two' ],
		[ 'vc-material vc-material-shopping_basket' => 'shopping basket' ],
		[ 'vc-material vc-material-short_text' => 'short text' ],
		[ 'vc-material vc-material-show_chart' => 'show chart' ],
		[ 'vc-material vc-material-shuffle' => 'shuffle' ],
		[ 'vc-material vc-material-signal_cellular_4_bar' => 'signal cellular 4 bar' ],
		[ 'vc-material vc-material-signal_cellular_connected_no_internet_4_bar' => 'signal_cellular_connected_no internet 4 bar' ],
		[ 'vc-material vc-material-signal_cellular_null' => 'signal cellular null' ],
		[ 'vc-material vc-material-signal_cellular_off' => 'signal cellular off' ],
		[ 'vc-material vc-material-signal_wifi_4_bar' => 'signal wifi 4 bar' ],
		[ 'vc-material vc-material-signal_wifi_4_bar_lock' => 'signal wifi 4 bar lock' ],
		[ 'vc-material vc-material-signal_wifi_off' => 'signal wifi off' ],
		[ 'vc-material vc-material-sim_card' => 'sim card' ],
		[ 'vc-material vc-material-sim_card_alert' => 'sim card alert' ],
		[ 'vc-material vc-material-skip_next' => 'skip next' ],
		[ 'vc-material vc-material-skip_previous' => 'skip previous' ],
		[ 'vc-material vc-material-slideshow' => 'slideshow' ],
		[ 'vc-material vc-material-slow_motion_video' => 'slow motion video' ],
		[ 'vc-material vc-material-stay_primary_portrait' => 'stay primary portrait' ],
		[ 'vc-material vc-material-smoke_free' => 'smoke free' ],
		[ 'vc-material vc-material-smoking_rooms' => 'smoking rooms' ],
		[ 'vc-material vc-material-textsms' => 'textsms' ],
		[ 'vc-material vc-material-snooze' => 'snooze' ],
		[ 'vc-material vc-material-sort' => 'sort' ],
		[ 'vc-material vc-material-sort_by_alpha' => 'sort by alpha' ],
		[ 'vc-material vc-material-spa' => 'spa' ],
		[ 'vc-material vc-material-space_bar' => 'space bar' ],
		[ 'vc-material vc-material-speaker' => 'speaker' ],
		[ 'vc-material vc-material-speaker_group' => 'speaker group' ],
		[ 'vc-material vc-material-speaker_notes' => 'speaker notes' ],
		[ 'vc-material vc-material-speaker_notes_off' => 'speaker notes off' ],
		[ 'vc-material vc-material-speaker_phone' => 'speaker phone' ],
		[ 'vc-material vc-material-spellcheck' => 'spellcheck' ],
		[ 'vc-material vc-material-star_border' => 'star border' ],
		[ 'vc-material vc-material-star_half' => 'star half' ],
		[ 'vc-material vc-material-stars' => 'stars' ],
		[ 'vc-material vc-material-stay_primary_landscape' => 'stay primary landscape' ],
		[ 'vc-material vc-material-stop' => 'stop' ],
		[ 'vc-material vc-material-stop_screen_share' => 'stop screen share' ],
		[ 'vc-material vc-material-storage' => 'storage' ],
		[ 'vc-material vc-material-store_mall_directory' => 'store mall directory' ],
		[ 'vc-material vc-material-straighten' => 'straighten' ],
		[ 'vc-material vc-material-streetview' => 'streetview' ],
		[ 'vc-material vc-material-strikethrough_s' => 'strikethrough s' ],
		[ 'vc-material vc-material-style' => 'style' ],
		[ 'vc-material vc-material-subdirectory_arrow_left' => 'subdirectory arrow left' ],
		[ 'vc-material vc-material-subdirectory_arrow_right' => 'subdirectory arrow right' ],
		[ 'vc-material vc-material-subject' => 'subject' ],
		[ 'vc-material vc-material-subscriptions' => 'subscriptions' ],
		[ 'vc-material vc-material-subtitles' => 'subtitles' ],
		[ 'vc-material vc-material-subway' => 'subway' ],
		[ 'vc-material vc-material-supervisor_account' => 'supervisor account' ],
		[ 'vc-material vc-material-surround_sound' => 'surround sound' ],
		[ 'vc-material vc-material-swap_calls' => 'swap calls' ],
		[ 'vc-material vc-material-swap_horiz' => 'swap horiz' ],
		[ 'vc-material vc-material-swap_vert' => 'swap vert' ],
		[ 'vc-material vc-material-swap_vertical_circle' => 'swap vertical circle' ],
		[ 'vc-material vc-material-switch_camera' => 'switch camera' ],
		[ 'vc-material vc-material-switch_video' => 'switch video' ],
		[ 'vc-material vc-material-sync_disabled' => 'sync disabled' ],
		[ 'vc-material vc-material-sync_problem' => 'sync problem' ],
		[ 'vc-material vc-material-system_update' => 'system update' ],
		[ 'vc-material vc-material-system_update_alt' => 'system update alt' ],
		[ 'vc-material vc-material-tab' => 'tab' ],
		[ 'vc-material vc-material-tab_unselected' => 'tab unselected' ],
		[ 'vc-material vc-material-tablet' => 'tablet' ],
		[ 'vc-material vc-material-tablet_android' => 'tablet android' ],
		[ 'vc-material vc-material-tablet_mac' => 'tablet mac' ],
		[ 'vc-material vc-material-tap_and_play' => 'tap and play' ],
		[ 'vc-material vc-material-text_fields' => 'text fields' ],
		[ 'vc-material vc-material-text_format' => 'text format' ],
		[ 'vc-material vc-material-texture' => 'texture' ],
		[ 'vc-material vc-material-thumb_down' => 'thumb down' ],
		[ 'vc-material vc-material-thumb_up' => 'thumb up' ],
		[ 'vc-material vc-material-thumbs_up_down' => 'thumbs up down' ],
		[ 'vc-material vc-material-timelapse' => 'timelapse' ],
		[ 'vc-material vc-material-timeline' => 'timeline' ],
		[ 'vc-material vc-material-timer' => 'timer' ],
		[ 'vc-material vc-material-timer_10' => 'timer 10' ],
		[ 'vc-material vc-material-timer_3' => 'timer 3' ],
		[ 'vc-material vc-material-timer_off' => 'timer off' ],
		[ 'vc-material vc-material-title' => 'title' ],
		[ 'vc-material vc-material-toc' => 'toc' ],
		[ 'vc-material vc-material-today' => 'today' ],
		[ 'vc-material vc-material-toll' => 'toll' ],
		[ 'vc-material vc-material-tonality' => 'tonality' ],
		[ 'vc-material vc-material-touch_app' => 'touch app' ],
		[ 'vc-material vc-material-toys' => 'toys' ],
		[ 'vc-material vc-material-track_changes' => 'track changes' ],
		[ 'vc-material vc-material-traffic' => 'traffic' ],
		[ 'vc-material vc-material-train' => 'train' ],
		[ 'vc-material vc-material-tram' => 'tram' ],
		[ 'vc-material vc-material-transfer_within_a_station' => 'transfer within a station' ],
		[ 'vc-material vc-material-transform' => 'transform' ],
		[ 'vc-material vc-material-translate' => 'translate' ],
		[ 'vc-material vc-material-trending_down' => 'trending down' ],
		[ 'vc-material vc-material-trending_flat' => 'trending flat' ],
		[ 'vc-material vc-material-trending_up' => 'trending up' ],
		[ 'vc-material vc-material-tune' => 'tune' ],
		[ 'vc-material vc-material-tv' => 'tv' ],
		[ 'vc-material vc-material-unarchive' => 'unarchive' ],
		[ 'vc-material vc-material-undo' => 'undo' ],
		[ 'vc-material vc-material-unfold_less' => 'unfold less' ],
		[ 'vc-material vc-material-unfold_more' => 'unfold more' ],
		[ 'vc-material vc-material-update' => 'update' ],
		[ 'vc-material vc-material-usb' => 'usb' ],
		[ 'vc-material vc-material-verified_user' => 'verified user' ],
		[ 'vc-material vc-material-vertical_align_bottom' => 'vertical align bottom' ],
		[ 'vc-material vc-material-vertical_align_center' => 'vertical align center' ],
		[ 'vc-material vc-material-vertical_align_top' => 'vertical align top' ],
		[ 'vc-material vc-material-vibration' => 'vibration' ],
		[ 'vc-material vc-material-video_call' => 'video call' ],
		[ 'vc-material vc-material-video_label' => 'video label' ],
		[ 'vc-material vc-material-video_library' => 'video library' ],
		[ 'vc-material vc-material-videocam' => 'videocam' ],
		[ 'vc-material vc-material-videocam_off' => 'videocam off' ],
		[ 'vc-material vc-material-videogame_asset' => 'videogame asset' ],
		[ 'vc-material vc-material-view_agenda' => 'view agenda' ],
		[ 'vc-material vc-material-view_array' => 'view array' ],
		[ 'vc-material vc-material-view_carousel' => 'view carousel' ],
		[ 'vc-material vc-material-view_column' => 'view column' ],
		[ 'vc-material vc-material-view_comfy' => 'view comfy' ],
		[ 'vc-material vc-material-view_compact' => 'view compact' ],
		[ 'vc-material vc-material-view_day' => 'view day' ],
		[ 'vc-material vc-material-view_headline' => 'view headline' ],
		[ 'vc-material vc-material-view_list' => 'view list' ],
		[ 'vc-material vc-material-view_module' => 'view module' ],
		[ 'vc-material vc-material-view_quilt' => 'view quilt' ],
		[ 'vc-material vc-material-view_stream' => 'view stream' ],
		[ 'vc-material vc-material-view_week' => 'view week' ],
		[ 'vc-material vc-material-vignette' => 'vignette' ],
		[ 'vc-material vc-material-visibility_off' => 'visibility off' ],
		[ 'vc-material vc-material-voice_chat' => 'voice chat' ],
		[ 'vc-material vc-material-voicemail' => 'voicemail' ],
		[ 'vc-material vc-material-volume_down' => 'volume down' ],
		[ 'vc-material vc-material-volume_mute' => 'volume mute' ],
		[ 'vc-material vc-material-volume_off' => 'volume off' ],
		[ 'vc-material vc-material-volume_up' => 'volume up' ],
		[ 'vc-material vc-material-vpn_key' => 'vpn key' ],
		[ 'vc-material vc-material-vpn_lock' => 'vpn lock' ],
		[ 'vc-material vc-material-wallpaper' => 'wallpaper' ],
		[ 'vc-material vc-material-watch' => 'watch' ],
		[ 'vc-material vc-material-watch_later' => 'watch later' ],
		[ 'vc-material vc-material-wb_auto' => 'wb auto' ],
		[ 'vc-material vc-material-wb_incandescent' => 'wb incandescent' ],
		[ 'vc-material vc-material-wb_iridescent' => 'wb iridescent' ],
		[ 'vc-material vc-material-wb_sunny' => 'wb sunny' ],
		[ 'vc-material vc-material-wc' => 'wc' ],
		[ 'vc-material vc-material-web' => 'web' ],
		[ 'vc-material vc-material-web_asset' => 'web asset' ],
		[ 'vc-material vc-material-weekend' => 'weekend' ],
		[ 'vc-material vc-material-whatshot' => 'whatshot' ],
		[ 'vc-material vc-material-widgets' => 'widgets' ],
		[ 'vc-material vc-material-wifi' => 'wifi' ],
		[ 'vc-material vc-material-wifi_lock' => 'wifi lock' ],
		[ 'vc-material vc-material-wifi_tethering' => 'wifi tethering' ],
		[ 'vc-material vc-material-work' => 'work' ],
		[ 'vc-material vc-material-wrap_text' => 'wrap text' ],
		[ 'vc-material vc-material-youtube_searched_for' => 'youtube searched for' ],
		[ 'vc-material vc-material-zoom_in' => 'zoom in' ],
		[ 'vc-material vc-material-zoom_out' => 'zoom out' ],
		[ 'vc-material vc-material-zoom_out_map' => 'zoom out map' ],
	];

	return array_merge( $icons, $material );
}
